#!/bin/bash

# f-cpu/vhdl/tools.desc/all_tools.sh
# created Sun Jun 23 07:39:35 CEST 2002
#     by Yann Guidon (whygee@f-cpu.org) for the F-CPU project.
# version Sun Jun 23 23:35:57 CEST 2002

# ------------------------BEGIN-SH-LICENSE---------------------------------
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# ---------------------------END-SH-LICENSE--------------------------------


# some cut&paste from /etc/init.d/functions
SETCOLOR_SUCCESS="echo -en \\033[1;32m"
SETCOLOR_FAILURE="echo -en \\033[1;31m"
SETCOLOR_WARNING="echo -en \\033[1;33m"
SETCOLOR_NORMAL="echo -en \\033[0;39m"

function echo_success() {
  echo -n "   [ "
  $SETCOLOR_SUCCESS
  echo -n $*
  $SETCOLOR_NORMAL
  echo " ]"
}

function echo_warning() {
  echo -n "   [ "
  $SETCOLOR_WARNING
  echo -n $*
  $SETCOLOR_NORMAL
  echo " ]"
}

function echo_failure() {
  echo -n "   [ "
  $SETCOLOR_FAILURE
  echo -n $*
  $SETCOLOR_NORMAL
  echo " ]"
  return 1
}

TOOLNAMES_FILE=.vhdl_tools

KNOWN_VHDL_TOOLNAMES="ncsim riviera simili vanilla modelsim"
 # don't forget to update it if you modify "all_tools" !


# this is the big fat main central core.
# please respect the alphabetical order for $VHDL_TOOL and
# the natural order for $VHDL_COMMAND. Keep it simple, too.
function all_tools () {
  case $VHDL_TOOL in

    # Copy and paste this if you want to add a tool :
    template)
      case $VHDL_COMMAND in
           detect)  ;;
          prepare)  ;;
          compile)  ;;
        elaborate)  ;;
         simulate)  ;;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    # not finished / untested
    modelsim)
      case $VHDL_COMMAND in
           detect)  echo_failure "Sorry : don't know how to detect $VHDL_TOOL" ;;
          prepare)  vlib work ;;
          compile)  for FILENAME in $FILENAMELIST
                    do # i don't know if it can compile several files at a time
                      vcom $FILENAME
                    done ;;
        elaborate)  echo -n ;;
         simulate)  vsim $ENTITY ;;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    ncsim)
      # corrects the syntax :
      if [ -n "$ARCHITECTURE" ]
      then
        LOCAL_ARCHITECTURE=":$ARCHITECTURE"
      fi

      case $VHDL_COMMAND in
           detect)  ncsim | grep 'Cadence Design Systems, Inc' ;;
          prepare)  mkdir -p ./cds_work_dir &&
                    # the paths are probably incorrect.
                    if [ ! -e cds.lib ] ; then
                      echo_warning "missing cds.lib, trying to link it"
                      ln -s ../cds.lib cds.lib
                    fi &&
                    if [ ! -e hdl.var ] ; then
                      echo_warning "missing hdl.var, trying to link it"
                      ln -s ../hdl.var hdl.var
                    fi ;;
          compile)  ncvhdl -NOLOG -V93 -MESSAGES $FILENAMELIST ;;
        elaborate)  ncelab -NOLOG -V93 -UPDATE work.$ENTITY$LOCAL_ARCHITECTURE ;;
         simulate)  ncsim  -NOLOG -UPDATE work.$ENTITY$LOCAL_ARCHITECTURE ;;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    # not tested yet :
    riviera)
      case $VHDL_COMMAND in
           detect)  echo_failure "sorry, no detection yet." ;;
          prepare)  vlib work &&
                    # set worklib work ;;   (???)
                    export worklib=work ;;
          compile)  vcom -work $FILENAMELIST ;;
        elaborate)  echo -n ;;
         simulate)  asim $ENTITY $ARCHITECTURE ;;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    simili)
      # corrects the syntax :
      if [ -n "$ARCHITECTURE" ]
      then
        LOCAL_ARCHITECTURE="($ARCHITECTURE)"
      fi

      case $VHDL_COMMAND in
           detect)  vhdle | grep 'Symphony EDA (R) VHDL Compiler/Simulator Module VhdlE' ;;
          prepare)  echo -n ;;
          compile)  vhdlp -strict -nostderr $FILENAMELIST;;
        elaborate)  echo -n ;;
         simulate)  vhdle -strict -nostderr $ENTITY$LOCAL_ARCHITECTURE ;;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    vanilla)
      # corrects the syntax :
      if [ -n "$ARCHITECTURE" ]
      then
        LOCAL_ARCHITECTURE=".$ARCHITECTURE"
      fi

      case $VHDL_COMMAND in
           detect)  echo q | vv87 | grep 'Vanilla Cad VHDL System';;
          prepare)  echo "mkvlib work" | vv87 | grep -v 'Vanilla\|software\|Default' ;;
          compile)  for FILENAME in $FILENAMELIST
                    do # loop because it only compiles one file at a time
                      echo "analyze -x $FILENAME" | vv87 \
                       | grep -v 'Vanilla\|software\|Default' || return -1
                    done ;;
        elaborate)  echo -n ;;
         simulate)  echo "simulate work.$ENTITY$LOCAL_ARCHITECTURE
run" \
                       | vv87 | grep -v 'Vanilla\|software\|Default';;
                *)  echo_failure "Error : command $VHDL_COMMAND not supported" ;;
      esac
      ;;

    *)
      # force the re-detection for the next time :
      unset INSTALLED_TOOLS
      rm -f $TOOLNAMES_FILE
      echo_failure "Error : unknown tool \"$VHDL_TOOL\""
      ;;

  esac
}


function detect_tools () {
  INSTALLED_TOOLS=''
  VHDL_COMMAND="detect"
  for VHDL_TOOL in $KNOWN_VHDL_TOOLNAMES
  do
    echo "  ------ Trying $VHDL_TOOL ------"
    all_tools &&
    # Add the newfound tool to our list and be happy :
    INSTALLED_TOOLS="$INSTALLED_TOOLS $VHDL_TOOL" &&
    echo_success "tool $VHDL_TOOL detected."
  done

  # Complain or complete :
  echo "
 -------------------------------"
  if [ -z "$INSTALLED_TOOLS" ]
  then
    rm -f $TOOLNAMES_FILE
    echo_failure " Sorry : no tool detected. "
    echo "If you have a VHDL simulator installed on this computer,
then you can add its description inside this file (vhdl_tools.sh).
"
    return 1
  else
    echo $INSTALLED_TOOLS > $TOOLNAMES_FILE
    echo " The following tools were found : "
    echo_warning "$INSTALLED_TOOLS"
    echo "This configuration is saved in $TOOLNAMES_FILE and you
can override this behaviour by editing it."
  fi
  export VHDL_TOOLNAMES=$INSTALLED_TOOLS
  echo " -------------------------------
"

}


function vhdl_compile () {
  # check if all the necessary variables are correctly set :

  if [ -z "$VHDL_TOOLNAMES" ]
  then
    echo_failure "error ! \$VHDL_TOOLNAMES not set !"
    return -1
  fi

  if [[ -z "$FILENAMELIST" && -z "$ENTITY" ]]
  then
    echo_failure "error ! nothing to do ! (compilation or simulation)"
    return -1
  fi

  # perform all the compilations sequentially,
  # otherwise there might be a work directory clash.
  for VHDL_TOOL in $VHDL_TOOLNAMES
  do
    echo "  ------ Starting $VHDL_TOOL ------"

    if [ -n "$FILENAMELIST" ]
    then
      VHDL_COMMAND="prepare" &&
      all_tools &&
      VHDL_COMMAND="compile" &&
      all_tools
    fi &&

    if [ -n "$ENTITY" ]
    then
      VHDL_COMMAND="elaborate" &&
      all_tools &&
      VHDL_COMMAND="simulate" &&
      all_tools
    fi ||

    {
      echo_failure "tool $VHDL_TOOL : command \"$VHDL_COMMAND\" failed."
      return -1
    }

  done &&
  echo_success "ok."
}


# the real start :

if [ -e $TOOLNAMES_FILE ]
then
  export VHDL_TOOLNAMES=$(< $TOOLNAMES_FILE)
fi

if [ -z "$VHDL_TOOLNAMES" ]
then
  echo " starting VHDL autodetection ..."
  detect_tools
fi
