-- ciadd2.vhdl -- Generic Carry-Increment Adder
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: ciadd2.vhdl,v 1.8 2000/11/17 18:27:18 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity CIAdd is
	generic (
		WIDTH : natural := 64
	);
	port (
		-- summands
		A : in std_ulogic_vector(WIDTH-1 downto 0);
		B : in std_ulogic_vector(WIDTH-1 downto 0);
		-- sum output
		Y : out std_ulogic_vector(WIDTH-1 downto 0);
		-- increment output
		C : out std_ulogic_vector(WIDTH-1 downto 0);
		-- generate output
		G : out std_ulogic;
		-- propagate output
		P : out std_ulogic
	);
end CIAdd;

-- This is a variable-width carry-increment adder.  It adds the `A'
-- and `B' inputs and provides the result on the `Y' output.  The `C'
-- output provides an `increment' vector that can be used to add 1 to
-- the result (`Y' xor `C' = `A' + `B' + 1); note that this could
-- also be derived from `Y' itself, but that would take longer :)

-- If you need a carry input, use `Y xor (C and (C'range => Cin))'
-- to calculate the final result.  A carry output can be derived
-- from the `P' (propagate) and `G' (generate) outputs, if necessary:
-- `Cout <= G or (P and Cin)'.

architecture Struct_1 of CIAdd is
	component AND2 is
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component XOR2 is
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component CIA_Row is
		generic (WIDTH : natural := 64);
		port (
			Gi : in std_ulogic_vector(WIDTH-1 downto 0);
			Pi : in std_ulogic_vector(WIDTH-1 downto 0);
			Co : out std_ulogic_vector(WIDTH-1 downto 0);
			Io : out std_ulogic_vector(WIDTH-1 downto 0);
			Go : out std_ulogic_vector((WIDTH-1)/4 downto 0);
			Po : out std_ulogic_vector((WIDTH-1)/4 downto 0)
		);
	end component;

	-- integer logarithm (rounded up)
	function ilog (x : natural; base : natural := 2) return natural is
		variable y : natural := 1;
	begin
		while x > base ** y loop
			y := y + 1;
		end loop;
		return y;
	end ilog;

	constant LEVELS : natural := ilog(WIDTH, 4);

	subtype matrix_row is std_ulogic_vector(WIDTH-1 downto 0);
	type matrix is array (natural range <>) of matrix_row;

	signal Ym, Cm : matrix(LEVELS downto 0);
	-- these are actually narrower in higher levels, but...
	signal Gm, Pm : matrix(LEVELS downto 0);
begin
	assert (WIDTH <= 4 ** LEVELS) and (4 * WIDTH > 4 ** LEVELS)
		report "ilog failure"
		severity failure;

	-- a row of half adders
	-- d=1
	input : for i in 0 to WIDTH-1 generate
		sum   : XOR2 port map (A(i), B(i), Pm(0)(i));
		carry : AND2 port map (A(i), B(i), Gm(0)(i));
	end generate;

	-- first-level inputs
	-- d=1
	Ym(0) <= Pm(0);
	Cm(0) <= (others => '1');

	-- carry-increment tree
	-- d=3+2*LEVELS (4 bit: d=5, 16 bit: d=7, 64 bit: d=9, 256 bit: d=11)
	all_levels : for level in 0 to LEVELS-1 generate
		level_n : block
			constant step : natural := 4 ** level;
			constant left : natural := (WIDTH - 1) / step;

			signal Ct, It : std_ulogic_vector(left downto 0);
			signal t : std_ulogic_vector(WIDTH-1 downto 0);
		begin
			-- single level of carry-increment tree
			-- d=n+2
			tree : CIA_Row
				generic map (WIDTH => left + 1)
				port map (
					Gi => Gm(level)(left downto 0),
					Pi => Pm(level)(left downto 0),
					Co => Ct,
					Io => It,
					Go => Gm(level+1)(left/4 downto 0),
					Po => Pm(level+1)(left/4 downto 0)
				);

			-- intermediate result
			-- d=n+4
			res : for i in 0 to WIDTH-1 generate
				tmp : AND2 port map (Cm(level)(i), Ct(i/step), t(i));
				sum : XOR2 port map (Ym(level)(i), t(i), Ym(level+1)(i));
				inc : AND2 port map (Cm(level)(i), It(i/step), Cm(level+1)(i));
			end generate;
		end block;
	end generate;

	-- outputs
	Y <= Ym(LEVELS);
	C <= Cm(LEVELS);
	G <= Gm(LEVELS)(0);
	P <= Pm(LEVELS)(0);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
