--------------------------------------------------------------------------
-- ROP2_testbench.vhdl : ROP2's tesbench for the F-CPU project
-- Copyright (C) 2000 Yann GUIDON (whygee@f-cpu.org) 10/22/2000@20h
-- 
-- This file is adapted from the file testbench_template.vhdl.
--
-- v0.2 : YG cleaned the integer() cast then added support for testing
-- the COMBINE instructions.
-- v0.3 : Erik Hansen <erikh@cs.tu-berlin.de> set get_hexa as impure
-- and has cast some constant strings for output with write().
--
--   current release : 11/17/2000
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
--------------------------------------------------------------------------
-- Only 5 commands are understood yet : A: set the data_in A bus,
-- B: set the data_in B bus (A and B are "static"), F: set the ROP function.
-- O (for "combine or") and D ("combine and") are not memorized.
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.std_logic_textio.all;
    USE ieee.numeric_std.all;
LIBRARY std;
    USE std.textio.ALL;
    USE work.FCPU_config.ALL;

-- the declaration of the design unit under test.
    USE work.EU_ROP2;

entity ROP2_testbench is
  GENERIC(
    filename : string := "testbenches\rop2vectors.txt" -- name of the vector file
  );
end ROP2_testbench;

architecture test of ROP2_testbench is
-- Copy of the design's port declaration :
  signal Din_A, Din_B  : F_VECTOR;                       -- the 2 operands
  signal ROP_function  : Std_ulogic_vector(3 downto 0);  -- 3 function bits
  signal Combine_AND, Combine_OR : Std_ulogic;           -- additional funtion
  signal Combine_size : Std_ulogic_vector(1 downto 0);   -- unused ATM. Byte chuncks only.
  signal Dout         : F_VECTOR;                        -- the result

-- internal counter :
  shared variable cycle : natural :=0;
  shared variable lout : line ;

-- For a dumb parser in VHDL :
  shared variable temp_char : Character := NUL;  -- parsed character
  shared variable buff : line;       -- line from which we parse
  shared variable line_number : natural:=0;
  shared variable i : natural;  -- index in buff

  procedure print is
  begin
    WRITE(lout,string'("* cycle # : "));
    WRITE(lout,cycle);
-- print all the signals here :
    WRITE(lout,string'("   ROP2 function:"));
    HWRITE(lout,ROP_function);
    if (Combine_AND='1') then
      WRITE(lout,string'("   Combine : AND"));
    else
      if (Combine_OR='1') then
        WRITE(lout,string'("   Combine : OR"));
      end if;
    end if;
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'(" Din_A="));
    HWRITE(lout,Din_A);
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'(" Din_B="));
    HWRITE(lout,Din_B);
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'("  Dout="));
    HWRITE(lout,Dout);
    WRITELINE(OUTPUT, lout);
  end;

  procedure getch is -- "read()" does not have "unread()"...
  begin
    temp_char:=buff(i);
    i:=i+1;
  end;

  -- warning !!! no overflow checks !!!
  impure function get_hexa(s:Std_ulogic_vector) return Std_ulogic_vector is
    variable t:Std_ulogic_vector(3 downto 0):=(others=>'0');
    variable u:Std_ulogic_vector(s'high downto 0):=(others=>'0');
  begin
    hexa_loop : while i<=buff'high loop
      getch;
      case temp_char is
        when '0' => t:=X"0";
        when '1' => t:=X"1";
        when '2' => t:=X"2";
        when '3' => t:=X"3";
        when '4' => t:=X"4";
        when '5' => t:=X"5";
        when '6' => t:=X"6";
        when '7' => t:=X"7";
        when '8' => t:=X"8";
        when '9' => t:=X"9";
        when 'A' => t:=X"A";
        when 'B' => t:=X"B";
        when 'C' => t:=X"C";
        when 'D' => t:=X"D";
        when 'E' => t:=X"E";
        when 'F' => t:=X"F";
        when 'a' => t:=X"A";
        when 'b' => t:=X"B";
        when 'c' => t:=X"C";
        when 'd' => t:=X"D";
        when 'e' => t:=X"E";
        when 'f' => t:=X"F";
        when others => i:=i-1; return u;
      end case;
-- shift left :
      u(u'high downto 4):=u((u'high)-4 downto 0);
      u(3 downto 0):=t;
    end loop;
-- exits here only if EOL :
    return u;
  end;

begin

 -- Instantiate the tested circuit
 ROP2_instance : entity EU_ROP2
   port map(
     Din_A => Din_A,
     Din_B => Din_B,
     ROP_function => ROP_function(2 downto 0),
     Combine_AND => Combine_AND,
     Combine_OR => Combine_OR,
     Combine_size => Combine_size,
     Dout => Dout
   );

 -- the testbench routine :
 process
   file test_vector : TEXT OPEN READ_MODE IS filename;
   variable temp_OR,temp_AND : Std_ulogic:='0';
 begin
-- simulation body :
   while (not endfile(test_vector)) loop
     -- init temps :
     temp_AND:='0';
     temp_OR:='0';

     -- read one vector line from the file :
     readline(test_vector,buff);
     line_number:=line_number+1;
     write(lout,string'("*** decoding line #"));
     write(lout,line_number);
     writeline(output,lout);

     If buff'length/=0 then 
     -- parsing "buff" here ...
       i:=1;
parse_loop:
       while i<=buff'high loop
         getch;
         case temp_char is

           when ';' => -- it's a comment
             exit parse_loop;

           when 'Q' =>
             write(lout,string'(LF&"   ### End of vector file ###."&LF));
             writeline(output,lout);
             wait;

           -- warning : no redundancy checks.
           when 'A' => -- set Din_A
             Din_A<=get_hexa(Din_A);

           when 'B' => -- set Din_B
             Din_B<=get_hexa(Din_B);

           when 'F' => -- set the function
             ROP_function<=get_hexa(ROP_function);

           when 'O' => -- combine OR
             if (temp_AND or temp_OR)/='0' then
               write(lout,string'("Warning : multiple COMBINE in vector !"&BEL));
             else
               temp_OR:='1';
             end if;

           when 'D' => -- combine AND
             if (temp_AND or temp_OR)/='0' then
               write(lout,string'("Warning : multiple COMBINE in vector !"&BEL));
             else
               temp_AND:='1';
             end if;

           when others =>
             write(lout,string'(LF&"  # unknown command : "));
             write(lout,temp_char&LF);
             writeline(output,lout);
         end case;
  
-- expect a coma/separator/EOL :
         if (i-1>=buff'High) then   -- do not expect coma if already at end of line
           exit parse_loop;
         end if;
         getch;
         if (temp_char = ';') then
           exit parse_loop;
         end if;
         if (temp_char /= ',') then
           write(lout,string'("',' expected, got "));
           write(lout,temp_char);
           write(lout,string'(" instead"));
           writeline(output,lout);
         end if;

       end loop parse_loop;
     end if; -- void string

     -- perform the assignation :
     Combine_OR<=temp_OR;
     Combine_AND<=temp_AND;

     -- cycle :
     cycle:=cycle+1;   -- advance the counter
     wait for 10 ns;   -- let the circuit do the work
     print;            -- be happy
   end loop;

   file_close(test_vector);

   wait; -- stop the simulation
 end process;

end test;
