-- ciastage.vhdl -- Carry-Increment Adder Stage
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id$

library IEEE;
use IEEE.std_logic_1164.all;

entity CIA_Stage is
	generic (
		WIDTH : natural := 64;
		STEP : natural := 4
	);
	port (
		-- generate/propagate inputs
		Gi : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
		Pi : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
		-- sum/increment inputs (d<=2 possible)
		Yi : in std_ulogic_vector(WIDTH-1 downto 0);
		Ci : in std_ulogic_vector(WIDTH-1 downto 0);
	--
		-- generate/propagate outputs, d=2
		Go : out std_ulogic_vector((WIDTH-1)/STEP/4 downto 0);
		Po : out std_ulogic_vector((WIDTH-1)/STEP/4 downto 0);
		-- sum/increment outputs, d=4
		Yo : out std_ulogic_vector(WIDTH-1 downto 0);
		Co : out std_ulogic_vector(WIDTH-1 downto 0)
	);
end CIA_Stage;

architecture Struct_1 of CIA_Stage is
	component AND2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component XOR2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component CIA_Row
		generic (WIDTH : natural := 64);
		port (
			Gi, Pi : in std_ulogic_vector(WIDTH-1 downto 0);
			Co, Io : out std_ulogic_vector(WIDTH-1 downto 0);
			Go, Po : out std_ulogic_vector((WIDTH-1)/4 downto 0)
		);
	end component;

	constant left : natural := (WIDTH - 1) / STEP;

	signal Ct, It : std_ulogic_vector(left downto 0);
	signal t : std_ulogic_vector(WIDTH-1 downto 0);
begin
	-- single level of carry-increment tree
	-- d=2
	tree : CIA_Row
		generic map (WIDTH => left + 1)
		port map (
			Gi => Gi, Pi => Pi, Co => Ct, Io => It, Go => Go, Po => Po
		);

	-- intermediate result
	-- d=4
	res : for i in 0 to WIDTH-1 generate
		tmp : AND2 port map (Ci(i), Ct(i/STEP), t(i));
		sum : XOR2 port map (Yi(i), t(i), Yo(i));
		inc : AND2 port map (Ci(i), It(i/STEP), Co(i));
	end generate;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
