-- reduce_tree.vhdl -- reduce number of partial products in a tree
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: reduce_tree.vhdl,v 1.5 2000/11/27 18:36:31 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity ReduceTree is
	generic (
		WIDTH : natural := 128;
		ILINES : natural := 4;
		OLINES : natural := 3
	);
	port (
		A : in std_ulogic_vector(WIDTH*ILINES-1 downto 0);
	--
		Y : out std_ulogic_vector(WIDTH*OLINES-1 downto 0)
	);
begin
	-- only some combinations are valid:
	assert (4 * OLINES = 3 * ILINES)	-- 4:3 reducer
		or (3 * OLINES = 2 * ILINES)	-- 3:2 reducer
		or (4 * OLINES = 2 * ILINES)	-- 4:2 reducer
		report "can only reduce 4->3, 3->2 or 4->2"
		severity failure;
end ReduceTree;

architecture Struct_1 of ReduceTree is
	component XOR3
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
	component MAJ23
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
begin
	-- 4:2 reducer
	-- d=4
	red_4_2 : if 4 * OLINES = 2 * ILINES generate
		assert ILINES mod 4 = 0
			report "ILINES is not a multiple of 4"
			severity failure;

		bl : block
			constant N : natural := ILINES / 4;
			signal t : std_ulogic_vector(WIDTH*3*N-1 downto 0);
		begin
			lp1 : for i in 0 to WIDTH*N-1 generate
				init : if i < N generate
					t(3*i) <= '0';
				end generate;
				t(3*i+1) <= A(4*i+0);
				x1 : XOR3 port map (A(4*i+1), A(4*i+2), A(4*i+3), t(3*i+2));
				carry : if i < (WIDTH-1)*N generate
					m1 : MAJ23 port map (A(4*i+1), A(4*i+2), A(4*i+3), t(3*i+3*N));
				end generate;
			end generate;
			lp2 : for i in 0 to WIDTH*N-1 generate
				init : if i < N generate
					Y(2*i) <= '0';
				end generate;
				x1 : XOR3 port map (t(3*i+0), t(3*i+1), t(3*i+2), Y(2*i+1));
				carry : if i < (WIDTH-1)*N generate
					m1 : MAJ23 port map (t(3*i+0), t(3*i+1), t(3*i+2), Y(2*i+2*N));
				end generate;
			end generate;
		end block;
	end generate;

	-- 4:3 reducer
	-- d=2
	red_4_3 : if 4 * OLINES = 3 * ILINES generate
		assert ILINES mod 4 = 0
			report "ILINES is not a multiple of 4"
			severity failure;

		bl : block
			constant N : natural := ILINES / 4;
		begin
			lp : for i in 0 to WIDTH*N-1 generate
				init : if i < N generate
					Y(3*i) <= '0';
				end generate;
				Y(3*i+1) <= A(4*i+0);
				x1 : XOR3 port map (A(4*i+1), A(4*i+2), A(4*i+3), Y(3*i+2));
				carry : if i < (WIDTH-1)*N generate
					m1 : MAJ23 port map (A(4*i+1), A(4*i+2), A(4*i+3), Y(3*i+3*N));
				end generate;
			end generate;
		end block;
	end generate;

	-- 3:2 reducer
	-- d=2
	red_3_2 : if 3 * OLINES = 2 * ILINES generate
		assert ILINES mod 3 = 0
			report "ILINES is not a multiple of 3"
			severity failure;

		bl : block
			constant N : natural := ILINES / 3;
		begin
			lp : for i in 0 to WIDTH*N-1 generate
				init : if i < N generate
					Y(2*i) <= '0';
				end generate;
				x1 : XOR3 port map (A(3*i+0), A(3*i+1), A(3*i+2), Y(2*i+1));
				carry : if i < (WIDTH-1)*N generate
					m1 : MAJ23 port map (A(3*i+0), A(3*i+1), A(3*i+2), Y(2*i+2*N));
				end generate;
			end generate;
		end block;
	end generate;
end Struct_1;

architecture Behave_1 of ReduceTree is
	function xor3 (a, b, c : std_ulogic) return std_ulogic is
	begin return a xor b xor c;
	end xor3;

	function maj23 (a, b, c : std_ulogic) return std_ulogic is
	begin return (a and b) or (b and c) or (c and a);
	end maj23;
begin
	-- 4:2 reducer
	-- d=4
	red_4_2 : if 4 * OLINES = 2 * ILINES generate
		assert ILINES mod 4 = 0
			report "ILINES is not a multiple of 4"
			severity failure;

		process (A)
			constant N : natural := ILINES / 4;
			variable t : std_ulogic_vector(WIDTH*3*N-1 downto 0) := (others => '0');
		begin
			Y <= (others => '0');
			for i in 0 to WIDTH*N-1 loop
				t(3*i+1) := A(4*i+0);
				t(3*i+2) := xor3(A(4*i+1), A(4*i+2), A(4*i+3));
				if i < (WIDTH-1)*N then
					t(3*i+3*N) := maj23(A(4*i+1), A(4*i+2), A(4*i+3));
				end if;
			end loop;
			for i in 0 to WIDTH*N-1 loop
				Y(2*i+1) <= xor3(t(3*i+0), t(3*i+1), t(3*i+2));
				if i < (WIDTH-1)*N then
					Y(2*i+2*N) <= maj23(t(3*i+0), t(3*i+1), t(3*i+2));
				end if;
			end loop;
		end process;
	end generate;

	-- 4:3 reducer
	red_4_3 : if 4 * OLINES = 3 * ILINES generate
		assert ILINES mod 4 = 0
			report "ILINES is not a multiple of 4"
			severity failure;

		process (A)
			constant N : natural := ILINES / 4;
		begin
			Y <= (others => '0');
			for i in 0 to WIDTH*N-1 loop
				Y(3*i+1) <= A(4*i+0);
				Y(3*i+2) <= xor3(A(4*i+1), A(4*i+2), A(4*i+3));
				if i < (WIDTH-1)*N then
					Y(3*i+3*N) <= maj23(A(4*i+1), A(4*i+2), A(4*i+3));
				end if;
			end loop;
		end process;
	end generate;

	-- 3:2 reducer
	red_3_2 : if 3 * OLINES = 2 * ILINES generate
		assert ILINES mod 3 = 0
			report "ILINES is not a multiple of 3"
			severity failure;

		process (A)
			constant N : natural := ILINES / 3;
		begin
			Y <= (others => '0');
			for i in 0 to WIDTH*N-1 loop
				Y(2*i+1) <= xor3(A(3*i+0), A(3*i+1), A(3*i+2));
				if i < (WIDTH-1)*N then
					Y(2*i+2*N) <= maj23(A(3*i+0), A(3*i+1), A(3*i+2));
				end if;
			end loop;
		end process;
	end generate;
end Behave_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
