-- pciadd2.vhdl -- Generic Carry-Increment Adder w/ Optional Pipelining
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: pciadd2.vhdl,v 1.3 2000/12/12 15:21:24 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity Piped_CIAdd is
	generic (
		WIDTH : natural := 64;
		PIPE_AFTER : natural := 0;	-- gates per stage; 0 means no pipelining
		PIPE_DELAY : natural := 0	-- additional delay before 1st stage
	);
	port (
		-- summands
		A : in std_ulogic_vector(WIDTH-1 downto 0);
		B : in std_ulogic_vector(WIDTH-1 downto 0);
		-- clock/reset (optional)
		Clk : in std_ulogic := '0';
		Rst : in std_ulogic := '0';
	--
		-- sum output
		Y : out std_ulogic_vector(WIDTH-1 downto 0);
		-- increment output
		C : out std_ulogic_vector(WIDTH-1 downto 0);
		-- generate output
		G : out std_ulogic;
		-- propagate output
		P : out std_ulogic
	);
begin
	-- NOTE: Pipeline timing is restricted.  This unit can only be
	-- split at d=1,3,5,7...  To match the timing of higher-level
	-- entities, increase PIPE_DELAY and/or decrease PIPE_AFTER
	-- appropriately.
	assert PIPE_AFTER mod 2 = 0
		report "PIPE_AFTER must be an even number"
		severity failure;
	assert (PIPE_AFTER = 0) or (PIPE_DELAY mod 2 = 1)
		report "PIPE_DELAY must be an odd number"
		severity failure;
end Piped_CIAdd;

-- This is a variable-width carry-increment adder.  It adds the `A'
-- and `B' inputs and provides the result on the `Y' output.  The `C'
-- output provides an `increment' vector that can be used to add 1 to
-- the result (`Y' xor `C' = `A' + `B' + 1); note that this could
-- also be derived from `Y' itself, but that would take longer :)

-- If you need a carry input, use `Y xor (C and (C'range => Cin))'
-- to calculate the final result.  A carry output can be derived
-- from the `P' (propagate) and `G' (generate) outputs, if necessary:
-- `Cout <= G or (P and Cin)'.

architecture Struct_1 of Piped_CIAdd is
	component AND2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component XOR2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component Piped_CIA_Row
		generic (WIDTH : natural := 64; PIPELINED : boolean := false);
		port (
			Gi : in std_ulogic_vector(WIDTH-1 downto 0);
			Pi : in std_ulogic_vector(WIDTH-1 downto 0);
			Clk : in std_ulogic := '0';
			Rst : in std_ulogic := '0';
			Co : out std_ulogic_vector(WIDTH-1 downto 0);
			Io : out std_ulogic_vector(WIDTH-1 downto 0);
			Go : out std_ulogic_vector((WIDTH-1)/4 downto 0);
			Po : out std_ulogic_vector((WIDTH-1)/4 downto 0)
		);
	end component;
	component Piped_CIA_Inc
		generic (
			WIDTH : natural := 64;
			STEP : natural := 4;
			PIPELINED : boolean := false
		);
		port (
			Yi : in std_ulogic_vector(WIDTH-1 downto 0);
			Ci : in std_ulogic_vector(WIDTH-1 downto 0);
			Ii : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
			Cs : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
			Clk : in std_ulogic := '0';
			Rst : in std_ulogic := '0';
			Yo : out std_ulogic_vector(WIDTH-1 downto 0);
			Co : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;
	component PipeReg
		generic (WIDTH : natural := 64; DOREG : boolean := false);
		port (
			Clk, Rst : in std_ulogic;
			D : in std_ulogic_vector(WIDTH-1 downto 0);
			Q : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	function do_pipe (d : natural) return boolean is
	begin
		if PIPE_AFTER /= 0 then
			return ((PIPE_DELAY + d) mod PIPE_AFTER) = 0;
		end if;
		return false;
	end do_pipe;

	-- integer logarithm (rounded up)
	function ilog (x : natural; base : natural := 2) return natural is
		variable y : natural := 1;
	begin
		while x > base ** y loop
			y := y + 1;
		end loop;
		return y;
	end ilog;

	constant LEVELS : natural := ilog(WIDTH, 4);

	subtype matrix_row is std_ulogic_vector(WIDTH-1 downto 0);
	type matrix is array (natural range <>) of matrix_row;

	signal Ym, Cm : matrix(LEVELS downto 0);
	-- these are actually narrower in higher levels, but...
	signal Gm, Pm : matrix(LEVELS downto 0);
begin
	assert (WIDTH <= 4 ** LEVELS) and (4 * WIDTH > 4 ** LEVELS)
		report "ilog failure"
		severity failure;

	-- a row of half adders
	-- d=1
	input : for i in 0 to WIDTH-1 generate
		sum   : XOR2 port map (A(i), B(i), Pm(0)(i));
		carry : AND2 port map (A(i), B(i), Gm(0)(i));
	end generate;

	-- first-level inputs
	-- d=1
	reg_0 : PipeReg
		generic map (WIDTH => WIDTH, DOREG => (LEVELS > 0) and do_pipe(1))
		port map (D => Pm(0), Q => Ym(0), Clk => Clk, Rst => Rst);
	Cm(0) <= (others => '1');

	-- carry-increment tree
	-- d=3+2*LEVELS (4 bit: d=5, 16 bit: d=7, 64 bit: d=9, 256 bit: d=11)
	all_levels : for level in 0 to LEVELS-1 generate
		level_n : block
			constant step : natural := 4 ** level;
			constant left : natural := (WIDTH - 1) / step;

			constant do_pipe_1 : boolean := do_pipe(2*level+1);
			constant do_pipe_2 : boolean := do_pipe(2*level+3);

			signal Ct, It : std_ulogic_vector(left downto 0);
			signal t : std_ulogic_vector(WIDTH-1 downto 0);
		begin
			-- single level of carry-increment tree
			-- d=n+2
			tree : Piped_CIA_Row
				generic map (WIDTH => left + 1, PIPELINED => do_pipe_1)
				port map (
					Gi => Gm(level)(left downto 0),
					Pi => Pm(level)(left downto 0),
					Clk => Clk,
					Rst => Rst,
					Co => Ct,
					Io => It,
					Go => Gm(level+1)(left/4 downto 0),
					Po => Pm(level+1)(left/4 downto 0)
				);

			-- intermediate result
			-- d=n+4
			res : Piped_CIA_Inc
				generic map (
					WIDTH => WIDTH, STEP => step, PIPELINED => do_pipe_2
				)
				port map (
					Yi => Ym(level),
					Ci => Cm(level),
					Ii => It,
					Cs => Ct,
					Clk => Clk,
					Rst => Rst,
					Yo => Ym(level+1),
					Co => Cm(level+1)
				);
		end block;
	end generate;

	-- outputs
	Y <= Ym(LEVELS);
	C <= Cm(LEVELS);
	G <= Gm(LEVELS)(0);
	P <= Pm(LEVELS)(0);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
