-- imu.vhdl - Integer Multiply Execution Unit for the F-CPU.
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: imu.vhdl,v 1.4 2000/12/11 04:56:35 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

library work;
use work.FCPU_config.all;

entity EU_IMU is
	port(
		-- operands
		Din_0    : in F_VECTOR;	-- multiplicand
		Din_1    : in F_VECTOR;	-- multiplicator
		Din_2    : in F_VECTOR;	-- summand (optional)
		-- MAC flags (should be derived from opcode)
		MacLo    : in std_ulogic;
		MacHi    : in std_ulogic;
		MacAlt   : in std_ulogic;
		-- flag bits (directly copied from instruction word)
		Flags    : in std_ulogic_vector(13 downto 8);
		-- SIMD mode bits (decoded)
		Size     : in std_ulogic_vector(LOGMAXSIZE-1 downto 0);
		-- clock/reset inputs
		Clk, Rst : in std_ulogic;
	--
		-- 8-bit result
		Dout_0   : out F_VECTOR;
		-- 8-bit carry output
		Dout_1   : out F_VECTOR;
		-- 16-bit result
		Dout_2   : out F_VECTOR;
		-- 16-bit carry output
		Dout_3   : out F_VECTOR;
		-- 32-bit result
		Dout_4   : out F_VECTOR;
		-- 32-bit carry output
		Dout_5   : out F_VECTOR;
		-- 64-bit result
		Dout_6   : out F_VECTOR;
		-- 64-bit carry output
		Dout_7   : out F_VECTOR
	);
end EU_IMU;

architecture Struct_1 of EU_IMU is
	component IMul64
		generic (
			PIPE_AFTER : natural := 0;	-- gates per stage; 0 means no pipelining
			PIPE_DELAY : natural := 0	-- additional delay before 1st stage
		);
		port (
			A : in std_ulogic_vector(63 downto 0);
			B : in std_ulogic_vector(63 downto 0);
			X : in std_ulogic_vector(63 downto 0) := (others => '0');
			SignedMode : in std_ulogic := '0';
			MacLo : in std_ulogic := '0';
			MacHi : in std_ulogic := '0';
			MacAlt : in std_ulogic := '0';	-- alternative (same-size) MAC version
			U08, U16, U32 : in std_ulogic := '1';
			Clk : in std_ulogic := '0';
			Rst : in std_ulogic := '0';
		--
			Y08l : out std_ulogic_vector(63 downto 0);
			Y08h : out std_ulogic_vector(63 downto 0);
			Y16l : out std_ulogic_vector(63 downto 0);
			Y16h : out std_ulogic_vector(63 downto 0);
			Y32l : out std_ulogic_vector(63 downto 0);
			Y32h : out std_ulogic_vector(63 downto 0);
			Y64l : out std_ulogic_vector(63 downto 0);
			Y64h : out std_ulogic_vector(63 downto 0)
		);
	end component;

	-- internal unit width (only 64 is supported right now)
	constant w : natural := (UMAX - 1) mod 64 + 1;
begin
	assert w = 64
		report "width must be a multiple of 64"
		severity failure;

	-- many multipliers in a row...
	instantiate : for i in UMAX/w-1 downto 0 generate
		core : IMul64
			generic map (
				-- standard timing: 6 gates per stage
				PIPE_AFTER => 6, PIPE_DELAY => 0
			)
			port map (
				A => Din_0(w*i+w-1 downto w*i),
				B => Din_1(w*i+w-1 downto w*i),
				X => Din_2(w*i+w-1 downto w*i),
				SignedMode => Flags(12),	-- as defined in the manual
				MacLo => MacLo,
				MacHi => MacHi,
				MacAlt => MacAlt,
				U08 => Size(0),
				U16 => Size(1),
				U32 => Size(2),		-- BEWARE! not available when UMAX < 64!
				Clk => Clk,
				Rst => Rst,
				Y08l => Dout_0(w*i+w-1 downto w*i),
				Y08h => Dout_1(w*i+w-1 downto w*i),
				Y16l => Dout_2(w*i+w-1 downto w*i),
				Y16h => Dout_3(w*i+w-1 downto w*i),
				Y32l => Dout_4(w*i+w-1 downto w*i),
				Y32h => Dout_5(w*i+w-1 downto w*i),
				Y64l => Dout_6(w*i+w-1 downto w*i),
				Y64h => Dout_7(w*i+w-1 downto w*i)
			);
	end generate;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
