-- asu.vhdl - Add/Subtract Execution Unit for the F-CPU.
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: asu.vhdl,v 1.4 2000/12/11 04:54:44 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

library work;
use work.FCPU_config.all;

entity EU_ASU is
	port(
		-- operands
		Din_0    : in F_VECTOR;
		Din_1    : in F_VECTOR;
		-- subtract flag (should be derived from opcode)
		Subtract : in std_ulogic;
		-- flag bits (directly copied from instruction word)
		Flags    : in std_ulogic_vector(13 downto 8);
		-- SIMD mode bits (decoded)
		Size     : in std_ulogic_vector(LOGMAXSIZE-1 downto 0);
		-- clock/reset inputs
		Clk      : in std_ulogic;
		Rst      : in std_ulogic;
	--
		-- 8-bit result
		Dout_0   : out F_VECTOR;
		-- 8-bit carry output
		Dout_1   : out F_VECTOR;
		-- 64-bit result
		Dout_2   : out F_VECTOR;
		-- 64-bit carry output
		Dout_3   : out F_VECTOR
	);
end EU_ASU;

architecture Struct_1 of EU_ASU is
	component IAdd
		generic (WIDTH : natural := 64);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
			B : in std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
			Sub : in std_ulogic := '0';
			Sat : in std_ulogic := '0';
			U08, U16, U32 : in std_ulogic := '0';
			Y8l : out std_ulogic_vector(WIDTH-1 downto 0);
			Y8h : out std_ulogic_vector(WIDTH-1 downto 0);
			Yl : out std_ulogic_vector(WIDTH-1 downto 0);
			Yh : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	-- internal unit width (only 64 is supported right now)
	constant w : natural := (UMAX - 1) mod 64 + 1;
begin
	assert w = 64
		report "width must be a multiple of 64"
		severity failure;

	-- many adders in a row...
	instantiate : for i in UMAX/w-1 downto 0 generate
		adder : IAdd
			generic map (WIDTH => w)
			port map (
				A => Din_0(w*i+w-1 downto w*i),
				B => Din_1(w*i+w-1 downto w*i),
				Sub => Subtract,
				Sat => Flags(12),	-- as defined in the manual
				U08 => Size(0),
				U16 => Size(1),
				U32 => Size(2),		-- BEWARE! not available when UMAX < 64!
				Y8l => Dout_0(w*i+w-1 downto w*i),
				Y8h => Dout_1(w*i+w-1 downto w*i),
				Yl => Dout_2(w*i+w-1 downto w*i),
				Yh => Dout_3(w*i+w-1 downto w*i)
			);
	end generate;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
