--------------------------------------------------------------------------
-- rop2_testbench.vhdl : ROP2's tesbench for the F-CPU project
-- Copyright (C) 2000 Yann GUIDON (whygee@f-cpu.org) 10/22/2000@20h
-- 
-- This file is adapted from the file testbench_template.vhdl.
--
-- v0.2 : YG cleaned the integer() cast then added support for testing
-- the COMBINE instructions.
-- v0.3 : Erik Hansen <erikh@cs.tu-berlin.de> set get_hexa as impure
-- and has cast some constant strings for output with write().
-- v0.4 : Mon Sep  3 08:59:49 2001 : YG updates the new signal names,
-- interface and commands. Further tests will be performed with the BIST unit,
-- so this is only a proof of concept.
-- 
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- Command summary :
-- data buses : ("static")
--  A: set the data_in A bus
--  B: set the data_in B bus
--  C: set the data_in C bus
--  F: set the ROP function (0 to 7)
-- Modes :  (not memorized)
--  O: "combine or" mode
--  D: "combine and" mode
--  M: MUX mode
--  nothing: normal mode
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.std_logic_textio.all;
    USE ieee.numeric_std.all;
LIBRARY std;
    USE std.textio.ALL;
    USE work.FCPU_config.ALL;

-- the declaration of the design units under test.
    USE work.EU_ROP2;
    USE work.EU_ROP2_XBAR;

entity ROP2_testbench is
  GENERIC(
    filename : string := "eu_rop2\rop2vectors.txt" -- name of the vector file
  );
end ROP2_testbench;

architecture test of ROP2_testbench is
-- Copy of the design's port declaration :
  signal Din_A, Din_B, Din_C : F_VECTOR;                -- the 3 operands
  signal ROP_function : Std_ulogic_vector(3 downto 0);  -- 3 function bits
  signal ROP2_mode    : Std_ulogic_vector(1 downto 0);  -- operation mode
--signal Combine_size : Std_ulogic_vector(1 downto 0);  -- unused ATM. Byte chuncks only.
  signal Dout         : F_VECTOR;                       -- the result

-- temporary variables between EU_ROP2 and EU_ROP2_XBAR :
  signal
    ROP2_function_bit0,
    ROP2_function_bit1,
    ROP2_function_bit2,
    ROP2_function_bit3 : Std_ulogic_vector((MAXSIZE *2) downto 0);
-- please note : these signal are pipelined in the FC0 but are directly
-- connected in this testbench for simplicity/convenience.

-- internal variables :
  shared variable cycle : natural :=0;
  shared variable lout : line ;

-- For a dumb parser in VHDL :
  shared variable temp_char : Character := NUL;  -- parsed character
  shared variable buff : line;                   -- line from which we parse
  shared variable line_number : natural:=0;
  shared variable i : natural;                   -- index in buff

  procedure print is
  begin
    WRITE(lout,string'("* cycle # : "));
    WRITE(lout,cycle);
-- print all the signals here :
    WRITE(lout,string'("   ROP2 function:"));
    HWRITE(lout,ROP_function);
    if (ROP2_mode=ROP2_AND_MODE) then
      WRITE(lout,string'(" MODE : Combine AND"));
    else
      if (ROP2_mode=ROP2_OR_MODE) then
        WRITE(lout,string'(" MODE : Combine OR"));
      else
        if (ROP2_mode=ROP2_MUX_MODE) then
          WRITE(lout,string'(" MODE : MUX"));
        else
          WRITE(lout,string'(" MODE : Normal"));
        end if;
      end if;
    end if;
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'(" Din_A="));
    HWRITE(lout,Din_A);
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'(" Din_B="));
    HWRITE(lout,Din_B);
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'(" Din_C="));
    HWRITE(lout,Din_C);
    WRITELINE(OUTPUT, lout);
    WRITE(lout,string'("  Dout="));
    HWRITE(lout,Dout);
    WRITELINE(OUTPUT, lout);
  end;

  procedure getch is -- "read()" does not have "unread()"...
  begin
    temp_char:=buff(i);
    i:=i+1;
  end;

  -- warning !!! no overflow checks !!!
  -- and "impure" is not well managed by vanilla HDL.
  impure function get_hexa(s:Std_ulogic_vector) return Std_ulogic_vector is
    variable t:Std_ulogic_vector(3 downto 0):=(others=>'0');
    variable u:Std_ulogic_vector(s'high downto 0):=(others=>'0');
  begin
    hexa_loop : while i<=buff'high loop
      getch;
      case temp_char is
        when '0' => t:=X"0";
        when '1' => t:=X"1";
        when '2' => t:=X"2";
        when '3' => t:=X"3";
        when '4' => t:=X"4";
        when '5' => t:=X"5";
        when '6' => t:=X"6";
        when '7' => t:=X"7";
        when '8' => t:=X"8";
        when '9' => t:=X"9";
        when 'A' => t:=X"A";
        when 'B' => t:=X"B";
        when 'C' => t:=X"C";
        when 'D' => t:=X"D";
        when 'E' => t:=X"E";
        when 'F' => t:=X"F";
        when 'a' => t:=X"A";
        when 'b' => t:=X"B";
        when 'c' => t:=X"C";
        when 'd' => t:=X"D";
        when 'e' => t:=X"E";
        when 'f' => t:=X"F";
        when others => i:=i-1; return u;
      end case;
-- shift left :
      u(u'high downto 4):=u((u'high)-4 downto 0);
      u(3 downto 0):=t;
    end loop;
-- exits here only if EOL :
    return u;
  end;

begin

 -- Instantiate the tested circuit
 ROP2_instance_1 : entity EU_ROP2_XBAR
   port map(
     ROP2_function => ROP_function(2 downto 0),
     ROP2_function_bit0 => ROP2_function_bit0,
     ROP2_function_bit1 => ROP2_function_bit1,
     ROP2_function_bit2 => ROP2_function_bit2,
     ROP2_function_bit3 => ROP2_function_bit3
   );

 ROP2_instance_2 : entity EU_ROP2
   port map(
     ROP2_in_A => Din_A,
     ROP2_in_B => Din_B,
     ROP2_in_C => Din_C,
     ROP2_function_bit0 => ROP2_function_bit0,
     ROP2_function_bit1 => ROP2_function_bit1,
     ROP2_function_bit2 => ROP2_function_bit2,
     ROP2_function_bit3 => ROP2_function_bit3,
     ROP2_mode => ROP2_mode,
     ROP2_out => Dout
   );

 -- the testbench routine :
 process
   file test_vector : TEXT OPEN READ_MODE IS filename;
   variable temp_mode : Std_ulogic_vector(1 downto 0);
 begin
-- simulation body :
   while (not endfile(test_vector)) loop
     -- init temps :
     temp_mode:=(others=>'0');

     -- read one vector line from the file :
     readline(test_vector,buff);
     line_number:=line_number+1;
     write(lout,string'("*** decoding line #"));
     write(lout,line_number);
     writeline(output,lout);

     If buff'length/=0 then 
     -- parsing "buff" here ...
       i:=1;
parse_loop:
       while i<=buff'high loop
         getch;
         case temp_char is

           when ';' => -- it's a comment
             exit parse_loop;

           when 'Q' =>
             write(lout,string'(LF&"   ### End of vector file ###."&LF));
             writeline(output,lout);
             wait;

           -- warning : no redundancy checks.
           when 'A' => -- set Din_A
             Din_A<=get_hexa(Din_A);

           when 'B' => -- set Din_B
             Din_B<=get_hexa(Din_B);

           when 'C' => -- set Din_B
             Din_C<=get_hexa(Din_C);

           when 'F' => -- set the function
             ROP_function<=get_hexa(ROP_function);

           when 'O' => -- combine OR
             if (temp_mode /= ROP2_DIRECT_MODE) then
               write(lout,string'("Warning : multiple COMBINE in test vector !"&BEL));
             else
               temp_mode:=ROP2_OR_MODE;
             end if;

           when 'D' => -- combine AND
             if (temp_mode /= ROP2_DIRECT_MODE) then
               write(lout,string'("Warning : multiple COMBINE in test vector !"&BEL));
             else
               temp_mode:=ROP2_AND_MODE;
             end if;

           when 'M' => -- MUX
             if (temp_mode /= ROP2_DIRECT_MODE) then
               write(lout,string'("Warning : multiple COMBINE in test vector !"&BEL));
             else
               temp_mode:=ROP2_MUX_MODE;
             end if;

           when others =>
             write(lout,string'(LF&"  # unknown command : "));
             write(lout,temp_char&LF);
             writeline(output,lout);
         end case;
  
-- expect a coma/separator/EOL :
         if (i-1>=buff'High) then   -- do not expect coma if already at end of line
           exit parse_loop;
         end if;
         getch;
         if (temp_char = ';') then
           exit parse_loop;
         end if;
         if (temp_char /= ',') then
           write(lout,string'("',' expected, got "));
           write(lout,temp_char);
           write(lout,string'(" instead"));
           writeline(output,lout);
         end if;

       end loop parse_loop;
     end if; -- void string

     -- perform the assignation :
     ROP2_mode<=temp_mode;

     -- cycle :
     cycle:=cycle+1;   -- advance the counter
     wait for 10 ns;   -- let the circuit do the work
     print;            -- be happy
   end loop;

   file_close(test_vector);

   wait; -- stop the simulation
 end process;

end test;
