--------------------------------------------------------------------------
-- f-cpu/vhdl/registers/row.vhdl - Part of Register Set
-- Copyright (C) 2002 Etienne Labarre (etienne.labarre@gadz.org)
-- Tue Mar 12 14:16:44 CET 2002
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- Base part of R7 unit : 
--  Number of registers : REG_NUMBER-1 registers + 1 register NULL
--  Size of registers   : ROW_WIDTH
--
-- Properties :
-- # 3 read ports
--   read operation when control signal = 0
-- # 2 write ports
--   write operation when control signal = 0
--   no write when two access at the same adress. 
-- # Registrer n0 :
--   write : no effect
--   read  : = NULL
-- # No reset signal
-- # in each read access, a signal row_null :
--   0 if all bits of register are null.
--   Made by OR operation to all bits of the data to write 
--   (BEFORE write operation. Signal clocked with datas at write access)
--   ==> signal NULL coherent with readed data when by-pass operation
--   (write at clock n, read at clock n+1 at the same adress)
--   
--------------------------------------------------------------------------

-- include standard libraries
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.all;

-- include mux/demux libraries
LIBRARY work;
USE work.mux;
USE work.demux;

-- entity interface definition :
ENTITY R7_row is
generic(
   -- Width of unit. 
   ROW_WIDTH : natural := 8;
   -- Number of registers (log)
   REG_LOG   : natural := 6
   );
port (
   -- 3 read adress
   row_read_adress_0,
   row_read_adress_1,
   row_read_adress_2 	: in std_ulogic_vector(REG_LOG-1 downto 0);

   -- 3 read ports
   row_read_port_0,
   row_read_port_1,
   row_read_port_2 	: out std_ulogic_vector(ROW_WIDTH-1 downto 0);
	
   -- 2 write adress
   row_write_adress_0,
   row_write_adress_1 	: in std_ulogic_vector(REG_LOG-1 downto 0);

   -- 2 write ports
   row_write_port_0,
   row_write_port_1 	: in std_ulogic_vector(ROW_WIDTH-1 downto 0);

   -- NULL signals for read ports :
   row_null_0,
   row_null_1,
   row_null_2 		: out std_ulogic;

   -- control signals :
   -- 3 read  : Read data when rising edge and null read signal
   -- 2 write : Write data when rising edge and null write signal
   -- 1 clock 
   row_control_read_0,
   row_control_read_1,
   row_control_read_2 	: in std_ulogic;
   row_control_write_0,
   row_control_write_1 	: in std_ulogic;
   row_clock 		: in std_ulogic
   );
end R7_row;

Architecture simu_row of R7_row is

   -- Definition of register number
   constant REG_NUMBER	: natural := 2**REG_LOG;

   -- internal signals for read access
   signal i_read_port_0,
          i_read_port_1,
          i_read_port_2 	: std_ulogic_vector(ROW_WIDTH-1 downto 0);

   -- REG_NUMBER x ROW_WIDTH bits to maintain (flip-flop)
   signal reg_out	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);

   -- internal multiplexer signals for read access
   signal read_mux_0	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);
   signal read_mux_1	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);
   signal read_mux_2	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);

   -- internal demultiplexer signals for write access
   signal write_demux_0	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);
   signal write_demux_1	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);

   signal write_ctl_0	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);
   signal write_ctl_1	: std_ulogic_vector(ROW_WIDTH*REG_NUMBER-1 downto 0);

   signal write_demux_others : std_ulogic;
   signal write_ctl_others   : std_ulogic;

   -- internal signals for NULL flags
   signal reg_null : std_ulogic_vector(REG_NUMBER-1 downto 0);

   signal i_row_null_0 : std_ulogic;
   signal i_row_null_1 : std_ulogic;
   signal i_row_null_2 : std_ulogic;

   signal row_write_port  : std_ulogic_vector(ROW_WIDTH-1 downto 0);
   signal register_null   : std_ulogic;
   signal sv2             : std_ulogic_vector(1 downto 0);

begin

   read_bit_loop : for i in ROW_WIDTH-1 downto 0 generate
      -- read loop : for each bit ( 0 to ROW_WIDTH-1 ), there is 
      -- a multiplexer REG_NUMBER --> 1 :
      -- input  : register 0 bit i
      --          register 1 bit i
      --          ...
      --          register REG_NUMBER-1 bit i
      -- 
      -- output : i_read_port bit i

      -- mux of data signals
      mux_0_bit_i : entity mux
         generic map (
	    width => REG_LOG
	    )
         port map (
	    mux_input  => reg_out((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	    mux_output => i_read_port_0(i), 
	    mux_select => row_read_adress_0 
	    );

      mux_1_bit_i : entity mux
         generic map (
	    width => REG_LOG
	    )
         port map (
	    mux_input  => reg_out((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	    mux_output => i_read_port_1(i), 
	    mux_select => row_read_adress_1 
	    );

      mux_2_bit_i : entity mux
         generic map (
	    width => REG_LOG
	    )
         port map (
	    mux_input  => reg_out((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	    mux_output => i_read_port_2(i), 
	    mux_select => row_read_adress_2 
	    );

   end generate read_bit_loop;

   -- mux of NULL signals 
   -- one bit per register --> signal null
   mux_0_null : entity mux
      generic map (
	 width => REG_LOG
	 )
      port map (
         mux_input  => reg_null(REG_NUMBER-1 downto 0),
	 mux_output => i_row_null_0,
	 mux_select => row_read_adress_0
	 );
	 
   mux_1_null : entity mux
      generic map (
	 width => REG_LOG
	 )
      port map (
         mux_input  => reg_null(REG_NUMBER-1 downto 0),
	 mux_output => i_row_null_1,
	 mux_select => row_read_adress_1
	 );
	 
   mux_2_null : entity mux
      generic map (
	 width => REG_LOG
	 )
      port map (
         mux_input  => reg_null(REG_NUMBER-1 downto 0),
	 mux_output => i_row_null_2,
	 mux_select => row_read_adress_2
	 );

   -- init of "demux_others" signal for mux components
   write_demux_others <= '0';
   write_ctl_others   <= '1';

   demux_loop : for i in ROW_WIDTH-1 downto 0 generate

      -- demux of write data
      demux_0_bit_i : entity demux
         generic map (
	    width => REG_LOG
	    )
         port map (
	    demux_output => write_demux_0((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	    demux_input  => row_write_port_0(i), 
	    demux_select => row_write_adress_0, 
	    demux_others => write_demux_others
	    );

      demux_1_bit_i : entity demux
         generic map (
	    width => REG_LOG
	    )
         port map (
	    demux_output => write_demux_1((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	    demux_input  => row_write_port_1(i), 
	    demux_select => row_write_adress_1, 
	    demux_others => write_demux_others
	    );

      -- Make one select bit per bit to memorize :
      -- 0 if to write, 1 if no change
      demux_ctl_0 : entity demux
            generic map (
	       width => REG_LOG
	       )
            port map (
	       demux_output => write_ctl_0((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	       demux_input  => row_control_write_0,
	       demux_select => row_write_adress_0, 
	       demux_others => write_ctl_others
	       );

      demux_ctl_1 : entity demux
            generic map (
	       width => REG_LOG
	       )
            port map (
	       demux_output => write_ctl_1((i+1)*REG_NUMBER-1 downto i*REG_NUMBER), 
	       demux_input  => row_control_write_1,
	       demux_select => row_write_adress_1, 
	       demux_others => write_ctl_others
	       );

   end generate demux_loop;

   -- Make NULL signal for each register (not register 0)
   -- NULL signal has no signification if the both write
   -- operations are at the same adress.
   -- NULL signal must be generated when clock event.
   sv2 <= row_control_write_0 & row_control_write_1;
   with sv2 select
      row_write_port <= 
         row_write_port_0 when "01",
         row_write_port_1 when "10",
          (others => '0') when others;

   or_loop : process(row_write_port) is 
      variable temp : std_ulogic;
   begin
      temp := '0';
      or_loop : for i in ROW_WIDTH-1 downto 0 loop
         temp := temp or row_write_port(i);
      end loop;
      register_null <= temp;
   end process;
   

   -- For each register (without register 0),
   --  update memory when rising edge
   -- and write_ctl signal is null
   --  update read ports when rising edge 
   -- and read_ctl signal is null
   update_reg : process(row_clock)
   begin
      -- Detect of rising edge
      if (row_clock'event and row_clock = '1') then

         -- Update read ports
         if (row_control_read_0 = '0') then
            row_read_port_0 <= i_read_port_0;
	    row_null_0      <= i_row_null_0;
         end if;
         if (row_control_read_1 = '0') then
            row_read_port_1 <= i_read_port_1;
	    row_null_1      <= i_row_null_1;
         end if;
         if (row_control_read_2 = '0') then
            row_read_port_2 <= i_read_port_2;
	    row_null_2      <= i_row_null_2;
         end if;

         -- Register 0 is NULL register by hardware
         R0_loop : for i in ROW_WIDTH-1 downto 0 loop
            reg_out(i*REG_NUMBER) <= '0';
         end loop;

         -- Update memory (ROW_WIDTH*(REG_NUMBER-1) bits for data and REG_NUMBER-1 NULL bits)
         D_bit_loop : for i in ROW_WIDTH-1 downto 0 loop
            D_reg_loop : for j in REG_NUMBER-1 downto 1 loop
               if (write_ctl_0(i*REG_NUMBER+j) = '0') then
                  reg_out(i*REG_NUMBER+j) <= write_demux_0(i*REG_NUMBER+j);
		  reg_null(j) <= register_null;
               end if;
               if (write_ctl_1(i*REG_NUMBER+j) = '0') then
                  reg_out(i*REG_NUMBER+j) <= write_demux_1(i*REG_NUMBER+j);
		  reg_null(j) <= register_null;
               end if;
            end loop;
         end loop;

      end if;
   end process;

end;

