-----------------------------------------------------------------------
-- f-cpu/vhdl/scheduler/scheduler_definitions.vhdl
-- The instruction decoder's definition package for the FC0
-- Copyright (C) 2001 Yann GUIDON (whygee@f-cpu.org)
-- created lun dec 17 20:44:52 GMT 2001 by whygee@f-cpu.org
-- version mar dec 18 01:43:23 GMT 2001 by whygee@f-cpu.org
-- version jeu dec 20 22:56:41 GMT 2001
--
--------------------------BEGIN-VHDL-LICENSE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENSE------------------------------
--
-- First revision : including EU output port numbering, SQ vector format ...
--
-----------------------------------------------------------------------


LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
-- LIBRARY work;
--     USE work.FCPU_config.ALL;


package scheduler_definitions is
  subtype t_reg is Std_ulogic_vector(5 downto 0);  -- should be moved to FCPU_config
  
  -- number of slot pairs in the Scheduling Queue
  constant SQ_depth : natural := 8;


  -- format for the writeback to the register set :
  subtype st_write_type is Std_ulogic_vector(1 downto 0);

  -- mode :   0-7  8-15  16-31  32-47  48-63
  -- SIMD      X    X      X      X      X
  -- u8        X
  -- u16       X    X
  -- u32       X    X      X
  -- u64       X    X      X      X      X
  -- imm-0     X    X
  -- imm-16                X
  -- imm-32                       X
  -- imm-64                              X
  -- the immediate part is however not needed inside the pipeline
  -- so we store only 2 bits that describe u8 to u64.

  constant SQ_write_u8  : st_write_type := "00";
  constant SQ_write_u16 : st_write_type := "01";
  constant SQ_write_u32 : st_write_type := "10";
  constant SQ_write_u64 : st_write_type := "11";

  -- defines the EU handles :
  subtype st_EU_type is Std_ulogic_vector(3 downto 0);
  constant EU_type_ASU  : st_EU_type := X"0";
  constant EU_type_INC  : st_EU_type := X"2";
  constant EU_type_SHL  : st_EU_type := X"4";
  constant EU_type_IDU  : st_EU_type := X"6";
  constant EU_type_ROP2 : st_EU_type := X"7";
  constant EU_type_POPC : st_EU_type := X"8";
  constant EU_type_LSU  : st_EU_type := X"9";
  constant EU_type_SR   : st_EU_type := X"A";
  constant EU_type_IMU  : st_EU_type := X"C";
  -- not all values are necessary : ASU, ROP2, SHL, POPC, IMU and INC
  -- are pipelined and there is no contention. SR, DIV and IDU are
  -- "blocking" in certain circumstances.

  -- defines the output ports :
  subtype st_EU_port is Std_ulogic_vector(3 downto 0);
  constant EU_port_ASU0   : st_EU_port := X"0";
  constant EU_port_ASU1   : st_EU_port := X"1";
  constant EU_port_INC0   : st_EU_port := X"2";
  constant EU_port_INC1   : st_EU_port := X"3";
  constant EU_port_SHL0   : st_EU_port := X"4";
  constant EU_port_SHL1   : st_EU_port := X"5";
  constant EU_port_IDU    : st_EU_port := X"6";
  constant EU_port_ROP2   : st_EU_port := X"7";
  constant EU_port_POPC   : st_EU_port := X"8";
  constant EU_port_LSU    : st_EU_port := X"9";
  constant EU_port_SR     : st_EU_port := X"A";
  constant EU_port_IMM    : st_EU_port := X"B";
  constant EU_port_IMU0   : st_EU_port := X"C";
  constant EU_port_IMU1   : st_EU_port := X"D";
  constant EU_port_IMU2   : st_EU_port := X"E";
  constant EU_port_IMU3   : st_EU_port := X"F";
  -- remark : not all ports are used now. It's a "large" allocation
  -- which should become configurable later.


  -- the destination register is inserted from one of the following sources :
  subtype st_mux_port is Std_ulogic_vector(1 downto 0);
  -- constant MUX_R1 : st_mux_port := "00";  -- should be removed ! but it's
  -- possible to use it in extreme cases.
  constant MUX_R2 : st_mux_port := "01";
  constant MUX_R3 : st_mux_port := "10";
  constant MUX_R4 : st_mux_port := "11";


  -- this is the information that is contained in every Scheduling Queue slot :
  type t_SQ_slot is
    record
      R1  : t_reg;      -- the register number
      EU1 : st_EU_port; -- the EU from which to read the value
      V1  : Std_ulogic; -- the "valid" bit, '1' if the slot is reserved
      WM1 : st_write_type; -- write mask (a copy of the "size" field in the opcode)
      R2  : t_reg;
      EU2 : st_EU_port;
      V2  : Std_ulogic;
      WM2 : st_write_type;
    end record;

  constant slot_cleared : t_SQ_slot :=
    (V1 | V2 => '0', R1 | R2 => (others=>'0'), WM1 | WM2 => (others=>'0'), EU1 | EU2 => (others=>'0'));


  -- defines the interface between the scheduler and the opcode decoder :
  type t_dec_interface is
    record
      -- concerning the scheduling :
      bad_opcode,     -- trigger a special trap
      nop,            -- more useful than it seems
      conditional,    -- cjump and cmove (relegate the issue to the next stage)
      -- necessary ressources :
      requires_src1,  -- also : condition
      requires_src2,  -- also : pointer
      requires_src3,  -- auxiliary source, such as mux or for L/S
      writes_dest1,
      writes_dest2,   -- dest1 xor 1
      writes_ptr,     -- for use with the memory instructions with post-increment
      write_0reg,
      write_1reg,     -- number of required write slots
      write_2reg,
      imm8,
      imm16,
      -- pointer verification :
      check_ptr_data,  -- load/store instructions
      check_ptr_instruction  -- jumps
      -- to be continued
        : Std_ulogic;

      latency1, latency2 : Std_ulogic_vector(SQ_depth-1 downto 0);
      -- 1 cycle : rop2, inc, asu8
      -- 2 cycles : asu, shl ?
      -- others : multiplier64

      -- target execution units :
      EU_type : st_EU_type;
      EU_port1, EU_port2 : st_EU_port;  -- this goes directly into the scheduling queue.
      write_mask1, write_mask2 : st_write_type;
      -- If only one register is needed, they have the same value. (?)
    end record;

end scheduler_definitions;

package body scheduler_definitions is

  -- empty.

end scheduler_definitions;
