--------------------------------------------------------------------------
-- f-cpu/vhdl/eu_inc/test_find_lsb.vhdl - Testbench for the find_lsb
-- Copyright (C) 2002 Etienne LABARRE (etienne.labarre@gadz.org)
--
-- Sun Jun 30 13:09:23 CEST 2002
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- 
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
-- text I/O
    use IEEE.std_logic_textio.all;
    use std.textio.all;
-- local functions
LIBRARY work;
    USE work.random.all;
    USE work.find_lsb;

Entity test_find_lsb is
  -- empty
end test_find_lsb;

Architecture testbench_find_lsb of test_find_lsb is

  constant WIDTH : natural := 8;
  constant UMX   : natural := 2**WIDTH;
  constant n_test : natural := 100;

  signal find_lsb_in_vector   : std_ulogic_vector(2**WIDTH-1 downto 0);
  signal find_lsb_out_vector  : std_ulogic_vector(2**WIDTH-1 downto 0);
  signal find_lsb_simd_vector : std_ulogic_vector(WIDTH-1 downto 0);

begin

   -- Component under test :
   find_lsb_component : entity find_lsb
      generic map( find_lsb_width => WIDTH )
      port map(
	 find_lsb_input => find_lsb_in_vector,
	 find_lsb_output => find_lsb_out_vector,
	 find_lsb_simd => find_lsb_simd_vector
	 );

   -- test bench :
   testbench : process is
      variable lout  : line;
      variable nb_chunk : natural;
      variable input_vector    : std_ulogic_vector(2**WIDTH-1 downto 0);
      variable output_vector   : std_ulogic_vector(2**WIDTH-1 downto 0);
      variable expected_vector : std_ulogic_vector(2**WIDTH-1 downto 0);

      -- procedure for debug
      procedure debug_result is
         variable lout  : line;
      begin
         write(lout, string'("input  : "));
         write(lout, input_vector);
         writeline(output, lout);
         write(lout, string'("output : "));
         write(lout, output_vector);
         writeline(output, lout);
         write(lout, string'("expected "));
         write(lout, expected_vector);
         writeline(output, lout);
      end procedure;

      -- function for find LSB0
      function lsb0 ( value : in std_ulogic_vector )
         return std_ulogic_vector is

	 variable carry : std_ulogic;
	 variable temp  : std_ulogic_vector(value'range);

      begin
         carry := '0';
         for i in value'low to value'high loop
	    temp(i) := carry;
	    if carry = '0' then
	       if value(i) = '0' then
	          carry := '1';
	       end if;
	    end if;
	 end loop;
	 return not temp;
      end function;

      -- function for find MSB0
      function msb0 ( value : in std_ulogic_vector )
         return std_ulogic_vector is

	 variable carry : std_ulogic;
	 variable temp  : std_ulogic_vector(value'range);

      begin
         carry := '0';
         for i in value'high downto value'low loop
	    temp(i) := carry;
	    if carry = '0' then
	       if value(i) = '0' then
	          carry := '1';
	       end if;
	    end if;
	 end loop;
	 return not temp;
      end function;

      procedure test_lsb0 ( value : in integer ) is
      begin
         for i in 0 to value loop
            -- input of test
            input_vector  := rand(input_vector);
	    find_lsb_in_vector <= input_vector;
	    -- perform expected value
            for i in 1 to nb_chunk loop
	       expected_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk) := 
	          lsb0(input_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk));
            end loop;
	    wait for 1 ns;
	    -- output of test
	    output_vector := find_lsb_out_vector;
	    assert output_vector = expected_vector
	       report "erreur sur LSB0"
	       severity warning;
	    if output_vector /= expected_vector then
	       debug_result;
	    end if;
	 end loop;
      end procedure;
	 
  begin    -- process
     write(lout, string'("******************* START OF TEST *******************"));
     writeline(output, lout);
     write(lout, string'("*      TEST WITH "));
     write(lout, 2**WIDTH);
     write(lout, string'(" BITS"));
     writeline(output, lout);
     write(lout, string'("*****************************************************"));
     writeline(output, lout);

     find_lsb_simd_vector <= "00000000";
     nb_chunk := 1;
     write(lout, string'("Test of LSB0 SIMD 256 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     find_lsb_simd_vector <= "00000001";
     nb_chunk := 2;
     write(lout, string'("Test of LSB0 SIMD 128 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     find_lsb_simd_vector <= "00000011";
     nb_chunk := 4;
     write(lout, string'("Test of LSB0 SIMD 64 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     find_lsb_simd_vector <= "00000111";
     nb_chunk := 8; 
     write(lout, string'("Test of LSB0 SIMD 32 bits"));
     writeline(output, lout);
     test_lsb0(n_test);
     
     find_lsb_simd_vector <= "00001111";
     nb_chunk := 16;
     write(lout, string'("Test of LSB0 SIMD 16 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     find_lsb_simd_vector <= "00011111";
     nb_chunk := 32;
     write(lout, string'("Test of LSB0 SIMD 8 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     find_lsb_simd_vector <= "00111111";
     nb_chunk := 64;
     write(lout, string'("Test of LSB0 SIMD 4 bits"));
     writeline(output, lout);
     test_lsb0(n_test);

     write(lout, string'("*******************  END OF TEST ********************"));
     writeline(output, lout);
     wait;
  end process;

end; 
