/*
 * fromfs.h
 * created Wed Mar 13 20:32:30 CET 2002 by whygee@f-cpu.org
 *
 * This file describes the data structures of the F-romfs file
 * that is appended and used by whygee's boot routine.
 *
 * a "directory" is composed of :
 *   - a header (type, size, number of elements and misc.)
 *   - a directory listing (256-bit entries : name, size, index, attributes)
 *   - a data block region
 *
 * F-CPU is a RISC 64-bit CPU with 256-bit cache lines :
 *  - all data must be aligned on natural boundaries
 *  - the start of the data blocks must be aligned (at least)
 *     to 64-bit boundaries
 *  - executable blocks must be copied to RAM and aligned to
 *     256 bits before execution.
 * 
 *
 * ***** i will assume that the file uses the target's endianness ! *****
 * This is because the target (the CPU that boots) doesn't need endianness
 * correction, or the boot routine would bloat... If the target CPU is not
 * the same endian as the CPU that generated the file, then one has to add
 * a "swap" routine in the mkfromfs. An endian flag can also be put in the
 * "root" directory by mkfromfs to check whether the target can read the
 * data.
 *
 * 
 */ 

#ifndef FROMFS_H
#define FROMFS_H


/* helps distinguish between host and target endians */
#include <endian.h>

#define FROMFS_SIGNATURE "FROMFS0"
/* The 8th char is either B for Big Endian or L for Little endian : */

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define CURRENT_ENDIANNESS_CHAR 'L'
#else
#if __BYTE_ORDER == __BIG_ENDIAN
#define CURRENT_ENDIANNESS_CHAR 'B'
#else
#error "unsupported endianness"
#endif
#endif

typedef struct {
  unsigned char signature[8];  /* equal to FROMFS_SIGNATURE */
  unsigned long int
    dir_size,     /* in bytes, can be rounded up to the next 8 bytes */
    dir_entries;  /* number of "files" */                
  /* i'd put a CRC here, and other stuff but that's not necessary yet;
     as long as it boots... and another version with "fromfs%n" will solve
     the problem. */
} fromfs_header;

#define offset_dir_size 8
#define offset_dir_entries 12
#define size_fromfs_header 16

/*
  offset    0  1  2  3  4  5  6  7
    0      'F''R''O''M''F''S''0' EndianChar
    8       block_size | entry count
*/

typedef struct {
  unsigned char
    name_size,   /* like a Pascal string, where the name size helps accelerate the search */
    name[19];    /* i hope it's a good compromise for the size ... */
  unsigned long int
    flags,
    block_size,
    block_index;
} fromfs_entry;

/* remark : the name_size field is useful if we want larger file names, several
   entries will then be used to store the full string. */

#define offset_name_size     0
#define offset_name          1
#define offset_flags        20
#define offset_block_size   24
#define offset_block_index  28
#define size_fromfs_entry   32

/*
  offset    0  1  2  3  4  5  6  7
    0      siz|        name
    8            name
    16         name   |      Flags   
    24     block_size | block_index
*/

/* The two used flags (in this version) */
#define FROMFS_FLAG_VALID 1<<0  /* entry is used (later used for long file names) */
#define FROMFS_FLAG_EXEC  1<<1  /* executable code */

#ifdef FROMFS_USES_COMPR
#define FROMFS_FLAG_COMPR 1<<2  /* compressed data block */
#endif

#ifdef FROMFS_USES_SUBDIR
#define FROMFS_FLAG_DIR   1<<3  /* the data block is actually an embedded (sub)directory */
#endif

#ifdef FROMFS_USES_LINK
#define FROMFS_FLAG_LINK  1<<4  /* we do hard links instead, now */
#endif

#endif /* FROMFS_H */
