-----------------------------------------------------------------------
-- f-cpu/vhdl/scheduler/scheduler_queue.vhdl
-- FC0's scheduling queue
-- Copyright (C) 2001 Yann GUIDON (whygee@f-cpu.org)
-- created lun dec 17 21:03:25 GMT 2001 by whygee@f-cpu.org
-- version jeu dec 20 15:55:03 GMT 2001 adapted from scheduler_slot.vhdl
-- version sam dec 22 07:15:51 GMT 2001
-- version dim dec 23 01:48:47 GMT 2001 : re-swapped the comparison to
--  avoid the issue where two slots are busy when only one is actually busy
--------------------------BEGIN-VHDL-LICENSE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENSE------------------------------
--
-- First revision : let's see if it works :-)
--
-----------------------------------------------------------------------


LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
LIBRARY work;
    USE work.scheduler_definitions.ALL;


entity scheduler_queue is
  port (
    -- in
    clk, reset : in Std_ulogic;
    -- from the instruction word
    R1, R2, R3 : in t_reg;
    -- from the issue logic :
    EU_port1, EU_port2 : st_EU_port;
    issue_level1, issue_level2 : in Std_ulogic_vector(SQ_depth-1 downto 0);
    issue_source1, issue_source2 : in st_mux_port;
    write_mask : st_write_type;
    -- out
    -- goes to the Xbar command lines
    data_out : out t_SQ_slot;
    -- goes to the issue logic
    slot1_busy, slot2_busy : out Std_ulogic_vector(SQ_depth-1 downto 0);
    R1_busy, R2_busy, R3_busy, R4_busy : out Std_ulogic);
end scheduler_queue;

architecture first_try of scheduler_queue is
  -- the queue itself :
  type t_queue is array (SQ_depth downto 0) of t_SQ_slot;  -- off-by-one on purpose
  signal queue : t_queue;
  -- some temporary variables for the comparisons :
  signal t1, t2, busy_1, busy_2, busy_3, busy_4 :
    Std_ulogic_vector(SQ_depth-1 downto 0);
  constant busy_zero : Std_ulogic_vector(SQ_depth-1 downto 0) := (others => '0');
  -- the register number MUX output
  signal reg_mux1, reg_mux2, new_R1, new_R2, new_R3 : t_reg;

begin  -- first_try

  -- one pipeline stage :
  new_register: process (clk)
  begin  -- process
    if rising_edge(clk) then
      new_R1 <= R1;
      new_R2 <= R2;
      new_R3 <= R3;
    end if;
  end process;

  -- mux
  with issue_source1 select
    reg_mux1 <=
      new_R2 when MUX_R2,
      new_R3 when MUX_R3,
      t_reg'(new_R3(new_R3'high downto 1) & not new_R3(0)) when others;  -- MUX_R4
-- could be a mux2 with a xor on the LSB of R3

  -- copy-paste-modify :
  with issue_source2 select
    reg_mux2 <=
      new_R2 when MUX_R2,
      new_R3 when MUX_R3,
      t_reg'(new_R3(new_R3'high downto 1) & not new_R3(0)) when others;  -- MUX_R4

  -- initialize the FIFO's top to "empty" :
  queue(SQ_depth) <= slot_cleared;
  -- this line will be changed when the IDU will be used.

  -- implement the FIFO :
  queue_body: for i in SQ_depth-1 downto 0 generate

    ------------------------
    -- data storage + mux :
    ------------------------

    fifo: process (clk) is
    begin
      if reset='1' then
        queue(i) <= slot_cleared;
      elsif rising_edge(clk) then
        queue(i).V1 <= queue(i+1).V1 or issue_level1(i);
        -- there is a 2-mux in front of the ff :
        if issue_level1(i)='1' then
          queue(i).R1  <= reg_mux1;
          queue(i).EU1 <= EU_port1;
          queue(i).WM1 <= write_mask;
        else
          queue(i).R1  <= queue(i+1).R1;
          queue(i).EU1 <= queue(i+1).EU1;
          queue(i).WM1 <= queue(i+1).WM1;
        end if;
        queue(i).V2 <= queue(i+1).V2 or issue_level2(i);
        if issue_level2(i)='1' then
          queue(i).R2  <= reg_mux2;
          queue(i).EU2 <= EU_port2;
          queue(i).WM2 <= write_mask;
        else
          queue(i).R2  <= queue(i+1).R2;
          queue(i).EU2 <= queue(i+1).EU2;
          queue(i).WM2 <= queue(i+1).WM2;
        end if;
      end if;
    end process fifo;

    -- comparisons
    busy_1(i) <= '1' when ((queue(i).V1='1') and (R1=queue(i).R1))
                       or ((queue(i).V2='1') and (R1=queue(i).R2)) else '0';
    busy_2(i) <= '1' when ((queue(i).V1='1') and (R2=queue(i).R1))
                       or ((queue(i).V2='1') and (R2=queue(i).R2)) else '0';
    t1(i)     <= '1' when ((queue(i).V1='1') and (R3(5 downto 1)=queue(i).R1(5 downto 1))) else '0';
    t2(i)     <= '1' when ((queue(i).V2='1') and (R3(5 downto 1)=queue(i).R2(5 downto 1))) else '0';
    busy_3(i) <= '1' when ((t1(i)='1') and (R3(0)= queue(i).R1(0)))
                       or ((t2(i)='1') and (R3(0)= queue(i).R2(0))) else '0';
    busy_4(i) <= '1' when ((t1(i)='1') and (R3(0)= not queue(i).R1(0)))
                       or ((t2(i)='1') and (R3(0)= not queue(i).R2(0))) else '0';

    -- report the slot's availability :
    slot1_busy(i) <= queue(i).V1;
    slot2_busy(i) <= queue(i).V2;

  end generate queue_body;

  -----------------------------------------------------------------------------
  -----------------------------------------------------------------------------
  --                          MISSING :
  --  * checcking all the off-by-ones
  --  * comparison with reg_mux1 and reg_mux2 (?)
  --  * no comparison for the last stage (because some data are ready during the next cycle)
  --  * register bypasses (2x)
  -----------------------------------------------------------------------------
  -----------------------------------------------------------------------------

  -- connexion of the output
  data_out <= queue(0);

  -- big ORs :
  R1_busy <= '1' when (busy_1 /= busy_zero) -- or ( cmp with reg_mux1 and reg_mux1 )
                 else '0';
  R2_busy <= '1' when (busy_2 /= busy_zero) else '0';
  R3_busy <= '1' when (busy_3 /= busy_zero) else '0';
  R4_busy <= '1' when (busy_4 /= busy_zero) else '0';

end first_try;
