--------------------------------------------------------------------------
-- File vhdl/eu_inc/find_lsb.vhdl - Binary tree for find LSB
-- Copyright (C) 2002 Etienne Labarre (etienne.labarre@gadz.org)
--
-- Sun Jun 30 11:37:34 CEST 2002
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- Binary tree for find lsb in a word. Size 2**N, with N = max 8.
--
-- fbit_simd : Perform SIMD operation. 
--
-- fbit_simd = "000000" => 256 bits operations
-- fbit_simd = "000001" => 128 bits operations
-- fbit_simd = "000011" => 64  bits operations
-- fbit_simd = "000111" => 32  bits operations
-- fbit_simd = "001111" => 16  bits operations
-- fbit_simd = "011111" => 8   bits operations
-- fbit_simd = "111111" => 4   bits operations
-- 
--------------------------------------------------------------------------

-- include standard libraries
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.all;

LIBRARY work;
USE work.bloc_and;

-- Definition of find_lsb entity
entity find_lsb is
   generic( find_lsb_width : natural := 6 );
   port (
      find_lsb_input  : in  std_ulogic_vector(2**find_lsb_width-1 downto 0);
      find_lsb_output : out std_ulogic_vector(2**find_lsb_width-1 downto 0);
      find_lsb_simd   : in  std_ulogic_vector(find_lsb_width-1 downto 0)
      );
end find_lsb;


-- Description of find_lsb entity
architecture simple of find_lsb is

   constant BITNB   : natural := 2**find_lsb_width;
   constant LEVELNB : natural := (find_lsb_width+1)/2;

   -- Definition of internal variables
   type natural_array is array(7 downto 0) of natural;
   type vector_array  is array(7 downto 0) of std_ulogic_vector(BITNB-1 downto -1);

   constant level_size : natural_array := (
      BITNB/49536,
      BITNB/12384,
      BITNB/4096,
      BITNB/1024,
      BITNB/256,
      BITNB/64,
      BITNB/16,
      BITNB/4
      );

   signal simd_bits    : std_ulogic_vector(find_lsb_width+1 downto 0);

   signal carry        : vector_array;
   signal level        : vector_array;
   signal temp_simd    : vector_array;
   signal temp         : vector_array;

begin

   -- control of width validity domain
   assert LEVELNB < 5
      report "bad data width"
      severity failure;

   -- Input of tree
   carry(0)(BITNB-1 downto 0) <= find_lsb_input(BITNB-1 downto 0);
   
   -- tree of and-gates
   second_loop : for i in 0 to LEVELNB-1 generate
      third_loop : for j in 0 to level_size(i)-1 generate
         second_level : entity bloc_and
	    port map (
	       input  => carry(i)(4*j+3 downto 4*j),
	       output => level(i+1)(4*j+3 downto 4*j),
	       carry  => carry(i+1)(j)
	       );
      end generate;
   end generate;

   -- Final AND bloc
   loop_inter : for i in 1 to BITNB-1 generate
      temp(0)(i) <= level(1)(i-1);
      temp(1)(i) <= level(1)(i-1);
      temp(2)(i) <= level(1)(i-1) and carry(1)(i/4-1);
      temp(3)(i) <= level(1)(i-1) and level(2)(i/4-1);
      temp(4)(i) <= level(1)(i-1) and level(2)(i/4-1) and carry(2)(i/16-1);
      temp(5)(i) <= level(1)(i-1) and level(2)(i/4-1) and level(3)(i/16-1);
      temp(6)(i) <= level(1)(i-1) and level(2)(i/4-1) and level(3)(i/16-1) and carry(3)(i/32-1);
      temp(7)(i) <= level(1)(i-1) and level(2)(i/4-1) and level(3)(i/16-1) and level(4)(i/32-1);
   end generate;

   -- Output for SIMD
   loop_simd : for i in 2 to find_lsb_width generate
      loop_simda : for j in 0 to (BITNB/(2**i))-1 generate
         temp_simd(i-2)(j*(2**i)) <= '1';
         loop_simdb : for k in 0 to i-1 generate
            loop_simdc : for l in (2**k) to (2*(2**k))-1 generate
               temp_simd(i-2)(j*(2**i)+l) <= temp(k)(j*(2**i)+l);
            end generate;
         end generate;
      end generate;
   end generate;


   with find_lsb_simd(5 downto 0) select
      find_lsb_output <= 
         temp_simd(6)(BITNB-1 downto 0) when "000000",
         temp_simd(5)(BITNB-1 downto 0) when "000001",
         temp_simd(4)(BITNB-1 downto 0) when "000011",
         temp_simd(3)(BITNB-1 downto 0) when "000111",
         temp_simd(2)(BITNB-1 downto 0) when "001111",
         temp_simd(1)(BITNB-1 downto 0) when "011111",
         temp_simd(0)(BITNB-1 downto 0) when "111111",
	 (others => '0') when others;

end;
