/* 
  f-cpu/c/scheduler/scheduler.c
  c simulation of the F-CPU decoder unit
  Copyright (C) 2002 Jaap Stolk (JWS) jwstolk@yahoo.com
  version: 19 July 2002 13:30

 ------------------------BEGIN-LICENSE------------------------------------
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ---------------------------END-LICENSE-----------------------------------
--
--   sch_need_r0_nr
--   sch_need_r1_nr
--   sch_need_r2_nr
--   sch_need_cycles
--   sch_need_w0_nr
--   sch_need_w1_nr
--        |       | | |                    | |
--        |       | | |                    | |
--        v       v v v                    v v
-- %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
-- %%                         scheduler                                %%
-- %% (scheduler-stage is executed at the same time as the xbar-stage) %%
-- %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
--                              |
--                              v
--                       scheduler_stalled
--
--  make borh write regusters the same if we only need 1 ?? (0 for nop?)
--
--  if a register is marked as unused, a baypass or deleayed bypass may
--  still be needed !! (check the register numbers on the xbar).
--
--  scoreboard needs to fit 0-cycle instructions ! (nop, reg move ??) (or less?)
--
--   sch_need_w0_nr = 10  may be chanced into: sch_need_w0_nr = 0   !!
--   sch_need_w1_nr = 0   (find free W slot)   sch_need_w1_nr = 10  !!
--
--  writing to register number 0 must be catched by the decoder, so we
--  don't spill a write slot on it.
--
--  show the the exact reason for the stall, for dibugging and optimizing !!!
*/

/* defines the constants, inputs and outputs : */
#include <scheduler.h>

static inline void scheduler_cycle (void) {

 
  int level_loop;
  int reg_loop;
  int tmp_swap;

  /* %%%%%%%% check scoreboard: %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

  /* we can look at the scoreboard BEFORE we add the IMUL and IDIV counters, */
  /* bet then we need an extra level in the scoreboard !! */

  scheduler_stalled = false; /* hardwired for now */

  /* check if all the input registers are free: */
  if ( sch_need_r0_nr ) {
    if ( register_in_use[0][sch_need_r0_nr] ) { scheduler_stalled = true; }
  }
  if ( sch_need_r1_nr ) {
    if ( register_in_use[0][sch_need_r1_nr] ) { scheduler_stalled = true; }
  }
  if ( sch_need_r2_nr ) {
    if ( register_in_use[0][sch_need_r2_nr] ) { scheduler_stalled = true; }
  }

  /* check if the needed write ports are free: */
  /* the nop instruction needs 0 free write ports !? */
  if ( sch_need_w0_nr ){   
    if ( sch_need_w1_nr ){ /* need two write ports: */
      if ( write_bus_in_use[0+sch_need_cycles][0] |
           write_bus_in_use[0+sch_need_cycles][1] ) {
        scheduler_stalled = true;
      }else{ /* at least 1 free write port, swap ports if needed: */
        if ( write_bus_in_use[0+sch_need_cycles][0] ) {
          tmp_swap       = sch_need_w0_nr;
          sch_need_w0_nr = sch_need_w1_nr;
          sch_need_w1_nr = tmp_swap;
        }
      }
    }else{                 /* need one write port: */
      if ( write_bus_in_use[0+sch_need_cycles][0] &
           write_bus_in_use[0+sch_need_cycles][1] ) {
        scheduler_stalled = true;
      }
    }
  } // else{   }                /* don't need any write port: */

  /* %%%%%%%% extend the scoreboard: %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

  /* fill the top level with "unused": */
  for (reg_loop=1; reg_loop<64; reg_loop++){
    register_in_use[7][reg_loop] = false;
  }
  for (reg_loop=1; reg_loop<2; reg_loop++){
    write_bus_in_use[7][reg_loop] = false;
  }

  /* add IMUL and IDV counters to toplevel when they reach level 7. */
  /* ............ (also for write busses) */

  /* %%%%%%%% isue instruction in scoreboard: %%%%%%%%%%%%%%%%%%%%%%% */

  /* (no set counters for IMUL / IDIV yet!!) */

  if ( !scheduler_stalled ){

    /* set the output register(s) to "used": */
    if (sch_need_w0_nr){
      for (level_loop=0; level_loop<=sch_need_cycles; level_loop++){
        register_in_use[level_loop][sch_need_w0_nr] = true; 
      }
    }
    if (sch_need_w1_nr){
      for (level_loop=0; level_loop<=sch_need_cycles; level_loop++){
        register_in_use[level_loop][sch_need_w0_nr] = true; 
      }
    }

    /* set the write bus slot(s) to "used": */
    if (sch_need_w0_nr){ write_bus_in_use[sch_need_cycles][0]= true; }
    if (sch_need_w1_nr){ write_bus_in_use[sch_need_cycles][1]= true; }

  }

  /* %%%%%%%% clock scoreboard: %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

  /* (this is done by the stage clock after each cycle) */

  /* clock the scoreboard FIFO */
  /* rotatong ptr's to levels could be used if this simulates to slow !! */
  for (level_loop=0; level_loop<7; level_loop++){
    for (reg_loop=1; reg_loop<64; reg_loop++){
      register_in_use[level_loop][reg_loop] = register_in_use[level_loop+1][reg_loop];
    }
  }

  /* clock the write_bus_in_use FIFO */
  for (level_loop=0; level_loop<7; level_loop++){
    for (reg_loop=0; reg_loop<2; reg_loop++){
      write_bus_in_use[level_loop][reg_loop] = write_bus_in_use[level_loop+1][reg_loop];
    }
  }

}







