/*
 * eapars.y -- YACC Parser for ELF Assembler
 * Copyright (C) 2001 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%{

static const char rcsid[] = "@(#) $Id: eapars.y,v 1.32 2001/09/04 02:48:58 michael Exp $";

#include <stdlib.h>
#include <string.h>
#include <libelf.h>
#include <assert.h>

#include <eatypes.h>

/* scanner interface */

extern int yylex(void);		/* most skeleton files seem to lack this */

/* target-specific data */

typedef struct target_props target_t;

typedef struct reloc reloc_t;

struct target_props {
	long		(*reloc_type)(const target_t*, reloc_t *rel, long size);
	long		(*encode_register)(const target_t*, const char *name);
	void		(*emit)(const target_t*, const char *name, expr_t *expr);
	void		(*emit_stab)(const target_t*, const char*, word_t, word_t, word_t, expr_t*);
	size_t		text_alignment;		/* instruction size <= x <= cache line size */
	size_t		data_alignment;		/* best guess: register size */
	size_t		max_alignment;		/* best guess: the CPU's page size */
	unsigned	all_start_symbols : 1;	/* create section start symbols unconditionally? */
	/* ELF stuff */
	unsigned	elf_class;
	unsigned	elf_data;
	unsigned	elf_machine;
	size_t		elf_reloc_size;
	size_t		elf_symbol_size;
	size_t		elf_reloc_type;
	size_t		elf_reloc_section;
	unsigned	elf_alloc_reloc : 1;
	unsigned	elf_alloc_symtab : 1;
};

extern target_t fcpu_target;		/* defined later */
target_t *target = &fcpu_target;

/* error handling */

extern unsigned long current_line;
extern unsigned long lineno;

extern void warn(const char *fmt, ...);
extern void error(const char *fmt, ...);
extern void fatal(const char *fmt, ...);	/* does not return */

extern void yyerror(const char *msg);

/* generic lookup tables */

typedef struct {
	const char*	name;
	long		value;
} lookup_table_t;

long table_lookup(lookup_table_t *lt, const char *item, long def, const char *category);

extern lookup_table_t object_types[];	/* defined later */
extern lookup_table_t section_types[];	/* defined later */
extern lookup_table_t section_attrs[];	/* defined later */

/* conditional assembly and include */

static unsigned cond_state = 2;

#define start_cond(cc)	(cond_state = (cond_state & 2) ? ((cc) ? 2 : 1) : 0)
#define change_cond(cc)	(cond_state = (cond_state & 1) ? ((cc) ? 2 : 1) : 0)
#define cond()			(cond_state & 2)

void include_file(const char *name);

/* standard malloc */

void *xrealloc(void *p, size_t len);
#define xmalloc(len) (xrealloc(NULL, (len)))
void xfree(void *p);

/* simple & fast pool-based "throw-away" memory allocator */

void *palloc(size_t len);
void pfree(void);

/* XXX: avoid this silliness */
char*
combine_strings(const char *s1, const char *s2, const char *s3) {
	char *s = palloc(strlen(s1) + strlen(s2) + strlen(s3) + 1);
	char *t = s;

	while ((*t = *s1++)) t++;
	while ((*t = *s2++)) t++;
	while ((*t = *s3++)) t++;
	return s;
}

/* sections */

extern scn_t *current_section;	/* defined later */
extern scn_t abs_section[];		/* defined later */

void init_sections(void);
long scn_flags_from_string(const char *str);
scn_t *find_section(const char *name, const long *flags, const long *type, long sub);
void default_section_attrs(scn_t *scn, const long *flags, const long *type);
scn_t *prev_section(void);
void change_section(scn_t *newscn);
char *extend_section(scn_t *scn, word_t bytes);
sym_t *dummy_symbol(const char *name, scn_t *scn, long type);
sym_t *section_start_symbol(scn_t *scn);
word_t add_string(scn_t *scn, const char *str);
void file_symbol(const char *name);
void add_comment(const char *text);

/* expressions */

extern expr_t errexpr[];	/* defined later */

expr_t *unary(int w, expr_t *r);
expr_t *binary(int w, expr_t *l, expr_t *r);
expr_t *check_abs(expr_t *e);
expr_t *numeric_expr(const char *val);
expr_t *address(sym_t *sym, scn_t *scn, word_t off);
expr_t *current_addr(void);
expr_t *register_operand(const char *reg);
expr_t *make_arglist(expr_t *l, expr_t *r);
expr_t *symbol_expr(sym_t *sym);

word_t to_int(expr_t *e);

int is_abs(expr_t *e);
int is_true(expr_t *e);

/* allocation */

void alloc_bytes(expr_t *size, expr_t *fill);
void alloc_data(long size, expr_t *data);
void alloc_fdata(long size, expr_t *data);
void alloc_string(int addz, const char *val);
void alloc_leb128(int signd, expr_t *data);
void alloc_fill(expr_t *repeat, expr_t *size, expr_t *data);
void align(int mode, expr_t *alignment, expr_t *fill, expr_t *max);
void new_org(expr_t *off, expr_t *fill);

/* symbols */

sym_t *make_symbol(const char *name);
sym_t *intern_symbol(const char *name);
void define_symbol(long unique, sym_t *sym, expr_t *val);
void define_common(long local, sym_t *sym, expr_t *size, expr_t *align);
void set_symbol_binding(sym_t *sym, long bind);
void set_symbol_size(sym_t *sym, word_t size);
void set_symbol_type(sym_t *sym, long type);
void set_symbol_value(sym_t *sym, expr_t *val);
int locally_defined(sym_t *sym);

/* the hardest part... */

void emit_instruction(const char *name, expr_t *args);
void emit_stab(const char *name, expr_t *type, expr_t *other, expr_t *desc, expr_t *value);

/* relocations */

word_t add_relocation(expr_t *val, long size);

%}

%union {
	long		lval;
	const char*	sval;
	expr_t*		expr;
	scn_t*		sect;
	sym_t*		symb;
}

/* asm directives */
%token ERROR EXTERN FILL FNAME IDENT ORG PREVSCN SECTION SIZE
%token SPACE SYMVER TYPE VERSION WEAKEXT
%token <lval> ALIGN ASSIGN COMMON EXPORT FDATA IDATA LEB128 SDATA
%token <sval> NAMESCN	/* .text | .data | .bss | probably others */

/* stabs debugging */
%token STABS STABN STABD

/* conditional assembly and include */
%token IF IFDEF IFNDEF ELIF ELSE ENDIF INCLUDE

/* other tokens */
%token <lval> CHARACTER
%token <sval> IDENTIFIER
%token <sval> NUMBER
%token <sval> STRING

/* "||" operator */
%token PAR

%type <expr> expr unary primary abs_expr opt_abs_expr operand operand_list
%type <expr> register_name
%type <lval> if_statement if_block conditional elif
%type <lval> integer_storage floating_storage string_storage leb128_storage
%type <lval> object_type section_type solaris_scn_flag scn_flag_list scn_flags
%type <sect> section_designator
%type <sval> section_name alias_name ident_or_string
%type <symb> symbol

/* operator precedence */
%left '|'
%left '^'
%left '&'
%left EQL NEQ
%left '<' '>' LEQ GEQ
%left SHL SHR
%left '+' '-'
%left '*' '/' '%'

%start file

%%

file : initialize group

initialize :
	{ cond_state = 2; init_sections(); }

group :
group : group set_location eol
group : group set_location line eol
group : group set_location conditional
	{ cond_state = $3; }
group : group set_location INCLUDE ident_or_string eol
	{ include_file($4); }

eol : '\n'
eol : error '\n'
	{ yyerrok; }

set_location :
	{ current_line = lineno; }

/*
	=> Conditional Assembly <=
	Since conditional assembly is done at the grammar level, the body
	of a .if (.ifdef, .ifndef, .elif, .else) directive must still
	be syntactically correct.
*/
conditional : if_block endif_statement
conditional : if_block else_statement group endif_statement

if_block : if_statement group
if_block : if_block elif_statement group

if_statement : IF abs_expr eol
	{ $$ = cond_state; start_cond(is_true($2)); }
if_statement : IFDEF symbol eol
	{ $$ = cond_state; start_cond(locally_defined($2)); }
if_statement : IFNDEF symbol eol
	{ $$ = cond_state; start_cond(!locally_defined($2)); }

symbol : IDENTIFIER
	{ $$ = intern_symbol($1); }

elif_statement : elif abs_expr eol
	{ cond_state = $1; change_cond(is_true($2)); }

elif : set_location ELIF
	{ $$ = cond_state; cond_state = 2; }

else_statement : set_location ELSE eol
	{ change_cond(1); }

endif_statement : set_location ENDIF eol

line : instruction
line : instruction ';'			/* last instruction may be empty */
line : instruction ';' line

instruction : label
instruction : parallel_instructions
instruction : pseudo_instruction
instruction : label instruction

label : symbol ':'
	{ if (cond()) define_symbol(1, $1, current_addr()); }

/*
	=> Parallel Execution <=
	Instructions separated by "||" may be executed simultaneously
	(if supported by the hardware).  The CPU (or the assembler) may
	reorder them as it sees fit.  Labels, if present, apply to the
	group as a whole.  If the so-parallelized instructions depend on
	each other, the behaviour is undefined.  Future versions of this
	program may detect that case and issue a warning, terminate with
	or without an error message, silently "sequentialize" the code,
	or simply ignore the error, at the implementors discretion.
	You've been warned.
*/
parallel_instructions : machine_instruction
parallel_instructions : parallel_instructions PAR machine_instruction
parallel_instructions : parallel_instructions PAR '\n' machine_instruction

machine_instruction : IDENTIFIER
	{ if (cond()) emit_instruction($1, NULL); }
machine_instruction : IDENTIFIER operand_list
	{ if (cond()) emit_instruction($1, $2); }

operand_list : operand
	{ $$ = make_arglist($1, NULL); }
operand_list : operand ',' operand_list
	{ $$ = make_arglist($1, $3); }

/*
	=> Implementor's Choice <=
	An identifier may be the name of an instruction, a register name
	or an ordinary symbol, which causes some ambiguities.  While
	instruction names can be easily recognized by position (after
	a newline, semicolon or "||" operator, with no ":" following),
	there is no way to tell the difference between a register name
	and a symbol used as an immediate operand.  Prefixing one of
	them with an otherwise unused character fixes the problem.
	Some assemblers use "$" or "#" for immediates, others use "%"
	for registers, some even prefix *both* (like gas for Intel IA32).
	I decided to prefix only immediate operands because register names
	are probably more often used than immediate operands.  I chose
	"$" as the prefix character (which may appear in an identifier,
	but not at its beginning) because most other characters (including
	the popular "#", "%", "&", and "@") already serve other purposes.
*/
operand : '$' expr
	{ $$ = $2; }
operand : register_name

/* ia32 operands:
	operand : ia32_operand
	operand : '*' ia32_operand

	ia32_operand : '%' register_name
	ia32_operand : '(' ia32_indirect ')'
	ia32_operand : expr '(' ia32_indirect ')'

	ia32_indirect : '%' register_name
	ia32_indirect : '%' register_name ',' '%' register_name
	ia32_indirect : '%' register_name ',' '%' register_name ',' abs_expr
	ia32_indirect : ',' '%' register_name ',' abs_expr
*/

register_name : IDENTIFIER
	{ $$ = register_operand($1); }

abs_expr : expr			{ $$ = check_abs($1); }

expr : expr '*' expr	{ $$ = binary('*', $1, $3); }
expr : expr '/' expr	{ $$ = binary('/', $1, $3); }
expr : expr '%' expr	{ $$ = binary('%', $1, $3); }
expr : expr '+' expr	{ $$ = binary('+', $1, $3); }
expr : expr '-' expr	{ $$ = binary('-', $1, $3); }
expr : expr SHL expr	{ $$ = binary(SHL, $1, $3); }
expr : expr SHR expr	{ $$ = binary(SHR, $1, $3); }
expr : expr '<' expr	{ $$ = binary('<', $1, $3); }
expr : expr '>' expr	{ $$ = binary('>', $1, $3); }
expr : expr LEQ expr	{ $$ = binary(LEQ, $1, $3); }
expr : expr GEQ expr	{ $$ = binary(GEQ, $1, $3); }
expr : expr EQL expr	{ $$ = binary(EQL, $1, $3); }
expr : expr NEQ expr	{ $$ = binary(NEQ, $1, $3); }
expr : expr '&' expr	{ $$ = binary('&', $1, $3); }
expr : expr '|' expr	{ $$ = binary('|', $1, $3); }
expr : expr '^' expr	{ $$ = binary('^', $1, $3); }
expr : unary			{ $$ = $1; }

unary : '+' primary		{ $$ = unary('+', $2); }
unary : '-' primary		{ $$ = unary('-', $2); }
unary : '~' primary		{ $$ = unary('~', $2); }
unary : '!' primary		{ $$ = unary('!', $2); }
unary : primary			{ $$ = $1; }

primary : '.'			{ $$ = current_addr(); }
primary : symbol		{ $$ = symbol_expr($1); }
primary : NUMBER		{ $$ = numeric_expr($1); }
primary : CHARACTER		{ $$ = address(NULL, abs_section, $1); }
primary : '(' expr ')'	{ $$ = $2; }
primary : '(' error ')'	{ $$ = errexpr; yyerrok; }

opt_abs_expr :			{ $$ = NULL; }
opt_abs_expr : abs_expr	{ $$ = $1; }

pseudo_instruction : '.' '=' expr	/* same as `.org expr' */
	{ if (cond()) new_org($3, NULL); }
pseudo_instruction : ORG expr
	{ if (cond()) new_org($2, NULL); }
pseudo_instruction : ORG expr ',' abs_expr
	{ if (cond()) new_org($2, $4); }
pseudo_instruction : symbol '=' expr	/* same as `.set symbol, expr' */
	{ if (cond()) define_symbol(0, $1, $3); }
pseudo_instruction : ASSIGN symbol ',' expr
	{ if (cond()) define_symbol($1, $2, $4); }
pseudo_instruction : COMMON symbol ',' abs_expr
	{ if (cond()) define_common($1, $2, $4, NULL); }
pseudo_instruction : COMMON symbol ',' abs_expr ',' abs_expr
	{ if (cond()) define_common($1, $2, $4, $6); }
pseudo_instruction : EXPORT symbol
	{ if (cond()) set_symbol_binding($2, $1); }
pseudo_instruction : WEAKEXT symbol		/* same as `.weak symbol' */
	{ if (cond()) set_symbol_binding($2, STB_WEAK); }
pseudo_instruction : WEAKEXT symbol ',' expr
	{ if (cond()) { set_symbol_binding($2, STB_WEAK); set_symbol_value($2, $4); } }
pseudo_instruction : section_designator
	{ if (cond()) change_section($1); }
pseudo_instruction : integer_storage
	{ /* handled elsewhere */ }
pseudo_instruction : floating_storage
	{ /* handled elsewhere */ }
pseudo_instruction : string_storage
	{ /* handled elsewhere */ }
pseudo_instruction : leb128_storage
	{ /* handled elsewhere */ }
pseudo_instruction : ALIGN abs_expr
	{ if (cond()) align($1, $2, NULL, NULL); }
pseudo_instruction : ALIGN abs_expr ',' abs_expr
	{ if (cond()) align($1, $2, $4, NULL); }
pseudo_instruction : ALIGN abs_expr ',' opt_abs_expr ',' abs_expr
	{ if (cond()) align($1, $2, $4, $6); }
pseudo_instruction : SPACE abs_expr
	{ if (cond()) alloc_bytes($2, NULL); }
pseudo_instruction : SPACE abs_expr ',' abs_expr
	{ if (cond()) alloc_bytes($2, $4); }
pseudo_instruction : FILL abs_expr
	{ if (cond()) alloc_fill($2, NULL, NULL); }
pseudo_instruction : FILL abs_expr ',' abs_expr
	{ if (cond()) alloc_fill($2, $4, NULL); }
pseudo_instruction : FILL abs_expr ',' opt_abs_expr ',' abs_expr
	{ if (cond()) alloc_fill($2, $4, $6); }
pseudo_instruction : ERROR
	{ if (cond()) error("encountered .err directive"); }
pseudo_instruction : EXTERN symbol
	{ /* always ignore */ }
pseudo_instruction : FNAME ident_or_string
	{ if (cond()) file_symbol($2); }
pseudo_instruction : IDENT ident_or_string
	{ if (cond()) add_comment($2); }
pseudo_instruction : VERSION ident_or_string
	{ /* ignore for now */ }
pseudo_instruction : SIZE symbol ',' abs_expr
	{ if (cond()) set_symbol_size($2, to_int($4)); }
pseudo_instruction : TYPE symbol ',' object_type
	{ if (cond()) set_symbol_type($2, $4); }
pseudo_instruction : SYMVER symbol ',' alias_name
	{ if (cond()) define_symbol(1, intern_symbol($4), symbol_expr($2)); }
pseudo_instruction : STABS STRING ',' abs_expr ',' abs_expr ',' abs_expr ',' expr
	{ if (cond()) emit_stab($2, $4, $6, $8, $10); }
pseudo_instruction : STABN abs_expr ',' abs_expr ',' abs_expr ',' expr
	{ if (cond()) emit_stab(NULL, $2, $4, $6, $8); }
pseudo_instruction : STABD abs_expr ',' abs_expr ',' abs_expr
	{ if (cond()) emit_stab(NULL, $2, $4, $6, current_addr()); }

integer_storage : IDATA expr
	{ if (cond()) alloc_data($$ = $1, $2); }
integer_storage : integer_storage ',' expr
	{ if (cond()) alloc_data($$ = $1, $3); }

floating_storage : FDATA abs_expr
	{ if (cond()) alloc_fdata($$ = $1, $2); }
floating_storage : floating_storage ',' abs_expr
	{ if (cond()) alloc_fdata($$ = $1, $3); }

string_storage : SDATA STRING
	{ if (cond()) alloc_string($$ = $1, $2); }
string_storage : string_storage ',' STRING
	{ if (cond()) alloc_string($$ = $1, $3); }

leb128_storage : LEB128 abs_expr
	{ if (cond()) alloc_leb128($$ = $1, $2); }
leb128_storage : leb128_storage ',' abs_expr
	{ if (cond()) alloc_leb128($$ = $1, $3); }

section_designator : NAMESCN
	{ $$ = find_section($1, NULL, NULL, 0); }
section_designator : NAMESCN abs_expr
	{ $$ = find_section($1, NULL, NULL, to_int($2)); }
section_designator : PREVSCN
	{ $$ = prev_section(); }
section_designator : section_name
	{ $$ = find_section($1, NULL, NULL, 0); }
section_designator : section_name ',' section_type
	{ $$ = find_section($1, NULL, &($3), 0); }
section_designator : section_name ',' scn_flags
	{ $$ = find_section($1, &($3), NULL, 0); }
section_designator : section_name ',' scn_flags ',' section_type
	{ $$ = find_section($1, &($3), &($5), 0); }
section_designator : section_name ',' scn_flag_list
	{ $$ = find_section($1, &($3), NULL, 0); }

section_name : SECTION ident_or_string
	{ $$ = $2; }

scn_flags : ident_or_string			/* awx */
	{ $$ = scn_flags_from_string($1); }

scn_flag_list : solaris_scn_flag
	{ $$ = $1; }
scn_flag_list : scn_flag_list ',' solaris_scn_flag
	{ $$ = $1 | $3; }

ident_or_string : IDENTIFIER
ident_or_string : STRING

/* ELF-specific stuff */

object_type : '@' IDENTIFIER	/* function / object */
	{ $$ = table_lookup(object_types, $2, STT_NOTYPE, "symbol type"); }
object_type : '#' IDENTIFIER	/* function / object */
	{ $$ = table_lookup(object_types, $2, STT_NOTYPE, "symbol type"); }

section_type : '@' IDENTIFIER	/* progbits / nobits */
	{ $$ = table_lookup(section_types, $2, SHT_PROGBITS, "section type"); }

solaris_scn_flag : '#' IDENTIFIER	/* write | alloc | execinstr */
	{ $$ = table_lookup(section_attrs, $2, 0, "section attribute"); }

/* XXX: avoid this silliness */
alias_name : IDENTIFIER '@' IDENTIFIER
	{ $$ = combine_strings($1, "@", $3); }
alias_name : IDENTIFIER '@' '@' IDENTIFIER
	{ $$ = combine_strings($1, "@@", $4); }

%%

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

struct sym {
	sym_t		*next;
	const char	*name;
	scn_t		*scn;	/* section this symbol belongs to */
	expr_t		*val;
	long		type;
	long		bind;
	word_t		size;
	word_t		align;
	size_t		relocs;
	/* ELF stuff */
	size_t		elf_index;
};

/* generic lookup tables */

long
table_lookup(lookup_table_t *lt, const char *item, long def, const char *category) {
	int i;

	if (!cond()) return def;

	assert(lt);
	assert(item);
	assert(category);
	for (; lt->name; lt++) {
		i = strcmp(item, lt->name);
		if (i < 0) {
			/* assume ordered table */
			break;
		}
		if (i == 0) {
			return lt->value;
		}
	}
	error("invalid %s \"%s\"", category, item);
	return def;
}

lookup_table_t object_types[] = {
	{ "function", STT_FUNC   },
	{ "object",   STT_OBJECT },
	{ NULL,       STT_NOTYPE }
};

lookup_table_t section_types[] = {
	{ "nobits",   SHT_NOBITS   },
	{ "progbits", SHT_PROGBITS },
	{ NULL,       SHT_NULL     }
};

lookup_table_t section_attrs[] = {
	{ "alloc",     SHF_ALLOC     },
	{ "execinstr", SHF_EXECINSTR },
	{ "write",     SHF_WRITE     },
	{ NULL,        0             }
};

/* standard malloc */

void*
xrealloc(void *p, size_t len) {
	len = len ? len : 1;
	p = p ? realloc(p, len) : malloc(len);
	if (!p) fatal("out of menory");
	return p;
}

void
xfree(void *p) {
	if (p) free(p);
}

/* pool alloc */

struct pool {
	struct pool	*next;
	unsigned	free;
};

static struct pool *palloc_pool;

#define PALLOC_POOL_SIZE	65536U
#define PALLOC_MAX_SIZE		256U
#define PALLOC_ALIGN		8U

void*
palloc(size_t len) {
	struct pool *p = palloc_pool;
	char *result;

	if (len > PALLOC_MAX_SIZE) {
		/* palloc is for small memory blocks only */
		fatal("cannot palloc %u bytes", len);
	}
	len += PALLOC_ALIGN - 1;
	len -= len % PALLOC_ALIGN;
	if (!p || p->free + len > PALLOC_POOL_SIZE) {
		p = xmalloc(PALLOC_POOL_SIZE);
		p->next = palloc_pool;
		palloc_pool = p;
		p->free = sizeof(struct pool) + PALLOC_ALIGN - 1;
		p->free -= p->free % PALLOC_ALIGN;
	}
	result = (char*)p + p->free;
	p->free += len;
	return (void*)result;
}

void
pfree(void) {
	struct pool *p, *q;

	assert(cond());
	for (p = palloc_pool; p; p = q) {
		q = p->next;
		xfree(p);
	}
	palloc_pool = NULL;
}

/* sections */

struct scn {
	scn_t		*next;
	const char	*name;
	char		*data;
	long		flags;
	long		type;
	word_t		size;
	word_t		align;
	sym_t		*dummy;
	reloc_t		*relocs;
	size_t		nrelocs;
	/* ELF stuff */
	Elf_Scn		*elf_scn;
	Elf_Scn		*elf_rel;
};

struct reloc {
	sym_t		*sym;
	long		type;
	word_t		off;
	word_t		addend;
};

scn_t *all_scns = NULL;

scn_t*
make_section(const char *name) {
	scn_t *scn;

	scn = palloc(sizeof(scn_t) + strlen(name) + 1);
	assert(scn);
	scn->next = NULL;
	scn->name = strcpy((char*)(scn + 1), name);
	scn->data = NULL;
	scn->flags = 0;
	scn->type = 0;
	scn->size = 0;
	scn->align = 0;
	scn->dummy = NULL;
	scn->relocs = NULL;
	scn->nrelocs = 0;
	scn->elf_scn = NULL;
	scn->elf_rel = NULL;
	return scn;
}

scn_t*
find_section(const char *name, const long *flags, const long *type, long sub) {
	scn_t **p, *scn;

	assert(cond());
	if (sub) {
		error("subsections not supported");
		sub = 0;
	}
	for (p = &all_scns; (scn = *p); p = &scn->next) {
		if (strcmp(name, scn->name) == 0) {
			if ((flags != NULL && *flags != scn->flags)
			 || (type != NULL && *type != scn->type)) {
				error("attempt to change section attributes for %s", name);
			}
			return scn;
		}
	}
	scn = make_section(name);
	assert(scn);
	default_section_attrs(scn, flags, type);
	if (target->all_start_symbols) {
		scn->dummy = dummy_symbol("", scn, STT_SECTION);
	}
	scn->next = *p, *p = scn;
	return scn;
}

static int
nobits_section(scn_t *scn) {
	assert(scn);
	return scn->type == SHT_NOBITS;
}

char*
extend_section(scn_t *scn, word_t bytes) {
	const unsigned PAGESZ = 1024;
	word_t off = scn->size;
	word_t n;

	assert(cond());
	assert(scn);
	scn->size += bytes;
	if (nobits_section(scn)) {
		return NULL;
	}
	n = scn->size + PAGESZ - 1;
	if (n / PAGESZ > (off + PAGESZ - 1) / PAGESZ) {
		scn->data = xrealloc(scn->data, n - n % PAGESZ);
	}
	return scn->data + off;
}

static struct scntab {
	const char*	name;
	long		type;
	long		flags;
} well_known_sections[] = {
	{ ".bss",            SHT_NOBITS,      SHF_ALLOC+SHF_WRITE     },
	{ ".comment",        SHT_PROGBITS,    0                       },
	{ ".data",           SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     },
	{ ".data1",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     },
	{ ".debug",          SHT_PROGBITS,    0                       },
	{ ".dynamic",        SHT_DYNAMIC,     SHF_ALLOC /* +WRITE */  }, /*ld*/
	{ ".dynstr",         SHT_STRTAB,      SHF_ALLOC               }, /*ld*/
	{ ".dynsym",         SHT_DYNSYM,      SHF_ALLOC               }, /*ld*/
	{ ".fini",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
	{ ".got",            SHT_PROGBITS,    0 /* ALLOC + WRITE */   }, /*ld*/
	{ ".hash",           SHT_HASH,        SHF_ALLOC               }, /*ld*/
	{ ".init",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
	{ ".interp",         SHT_PROGBITS,    0 /* SHF_ALLOC */       }, /*ld*/
	{ ".line",           SHT_PROGBITS,    0                       },
	{ ".note",           SHT_NOTE,        0                       }, /*auto*/
	{ ".plt",            SHT_PROGBITS,    0 /* ALLOC + EXEC */    }, /*ld*/
	{ ".rel.bss",        SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.data",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.data1",      SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.debug",      SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.fini",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.got",        SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.init",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.line",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.plt",        SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.rodata",     SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.rodata1",    SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.sbss",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.sdata",      SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.stab",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rel.text",       SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.bss",       SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.data",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.data1",     SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.debug",     SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.fini",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.got",       SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.init",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.line",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.plt",       SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.rodata",    SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.rodata1",   SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.sbss",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.sdata",     SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.stab",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela.text",      SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rodata",         SHT_PROGBITS,    SHF_ALLOC               },
	{ ".rodata1",        SHT_PROGBITS,    SHF_ALLOC               },
	{ ".sbss",           SHT_NOBITS,      SHF_ALLOC+SHF_WRITE     }, /*short*/
	{ ".sdata",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*short*/
	{ ".shstrtab",       SHT_STRTAB,      0                       }, /*auto*/
	{ ".stab",           SHT_PROGBITS,    0                       },
	{ ".stabstr",        SHT_STRTAB,      0                       },
	{ ".strtab",         SHT_STRTAB,      0 /* SHF_ALLOC */       }, /*auto*/
	{ ".symtab",         SHT_SYMTAB,      0 /* SHF_ALLOC */       }, /*auto*/
	{ ".text",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
#if 0 /* Linux/GNU special (LSB 1.0) */
	{ ".ctors",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".dtors",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".eh_frame",       SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".gnu.version_d",  SHT_GNU_verdef,  SHF_ALLOC               }, /*LSB*/
	{ ".gnu.version_r",  SHT_GNU_verneed, SHF_ALLOC               }, /*LSB*/
	{ ".gnu.version",    SHT_GNU_versym,  SHF_ALLOC               }, /*LSB*/
	{ ".note.ABI-tag",   ?,               ?,                      }, /*LSB*/
#endif
#if 0 /* GNU junk */
	{ ".gnu.warning.*",  SHT_PROGBITS,    0                       },
#endif
};

void
default_section_attrs(scn_t *scn, const long *flags, const long *type) {
	size_t l, r, m;
	int i;

	assert(cond());
	assert(scn);
	if (type == NULL || flags == NULL) {
		/* get default attributes */
		l = 0;
		r = sizeof(well_known_sections) / sizeof(well_known_sections[0]);
		while (l < r) {
			m = l + ((r - l) >> 1);
			i = strcmp(scn->name, well_known_sections[m].name);
			if (i < 0) {
				r = m;
			}
			else if (i > 0) {
				l = m + 1;
			}
			else {
				/* found */
				if (type == NULL) {
					type = &well_known_sections[m].type;
				}
				if (flags == NULL) {
					flags = &well_known_sections[m].flags;
				}
				break;
			}
		}
	}
	scn->type = type ? *type : SHT_PROGBITS;
	scn->flags = flags ? *flags : 0;
	assert(scn->type != SHT_NULL);
}

scn_t *bss_section = NULL;

scn_t *current_section = NULL;
scn_t *previous_section = NULL;

scn_t abs_section[1];
scn_t undef_section[1];
scn_t common_section[1];

/* XXX: add section stack? */
scn_t*
prev_section(void) {
	if (previous_section) {
		return previous_section;
	}
	if (cond()) {
		error("no previous section");
	}
	return current_section;
}

void
change_section(scn_t *newscn) {
	assert(cond());
	assert(newscn);
	if (newscn != current_section) {
		previous_section = current_section;
		current_section = newscn;
	}
}

word_t
add_string(scn_t *scn, const char *str) {
	word_t off;
	char *s;

	assert(cond());
	assert(scn);
	assert(str);
	off = scn->size;
	if (nobits_section(scn)) {
		error("directive is invalid in NOBITS sections");
		/* sanitize */
		return off;
	}
	s = extend_section(scn, strlen(str) + 1);
	strcpy(s, str);
	return off;
}

static scn_t *comment_section = NULL;

void
add_comment(const char *text) {
	if (!comment_section) {
		comment_section = find_section(".comment", NULL, NULL, 0);
	}
	if (comment_section->size == 0) {
		add_string(comment_section, "");
	}
	add_string(comment_section, text);
}

void
init_sections(void) {
	long flags, type;
	scn_t *scn;

	assert(cond());
	all_scns = NULL;
	/*
	 * create text section
	 */
	type = SHT_PROGBITS;
	flags = SHF_ALLOC | SHF_EXECINSTR;
	scn = find_section(".text", &flags, &type, 0);
	scn->align = target->text_alignment;
	/*
	 * assembly starts in `.text 0'
	 */
	current_section = scn;
	previous_section = NULL;
	comment_section = NULL;
	/*
	 * create data sections
	 */
	flags = SHF_ALLOC | SHF_WRITE;
	scn = find_section(".data", &flags, &type, 0);
	scn->align = target->data_alignment;
	type = SHT_NOBITS;
	scn = find_section(".bss", &flags, &type, 0);
	scn->align = target->data_alignment;
	bss_section = scn;
}

long
scn_flags_from_string(const char *str) {
	long flags = 0;
	int err = 0;
	char c;

	if (!cond()) return 0;

	while ((c = *str++)) {
		switch (c) {
			case 'A': case 'a': flags |= SHF_ALLOC; break;
			case 'W': case 'w': flags |= SHF_WRITE; break;
			case 'X': case 'x': flags |= SHF_EXECINSTR; break;
			default: err++;
		}
	}
	if (err) {
		error("invalid section attribute");
	}
	return flags;
}

/* expressions */

union expr {
	struct {
		int		op;
#define op(expr)	((expr)->any.op)
	} any;
	struct {
		int		op;		/* unary/binary */
		expr_t	*left;
		expr_t	*right;
#define left(expr)	((expr)->bin.left)
#define right(expr)	((expr)->bin.right)
	} bin;
	struct {			/* absolute, address */
		int		op;		/* 'a' */
		sym_t	*sym;
		scn_t	*scn;
		word_t	off;
#define sym(expr)	((expr)->addr.sym)
#define scn(expr)	((expr)->addr.scn)
#define off(expr)	((expr)->addr.off)
	} addr;
	struct {			/* bignum */
		int		op;		/* 'b' */
		size_t	len;
		word_t	*val;
#define blen(expr)	((expr)->big.len)
#define bval(expr)	((expr)->big.val)
	} big;
	/* XXX: add 'f' (floating-point number) */
	struct {			/* register */
		int		op;		/* 'r' */
		long	rnum;
#define rnum(expr)	((expr)->reg.rnum)
	} reg;
};

expr_t errexpr[] = {{{ 'E' }}};

expr_t*
make_arglist(expr_t *l, expr_t *r) {
	expr_t *e;

	if (!cond()) return NULL;
	e = palloc(sizeof(expr_t));
	assert(e);
	op(e) = ',';
	left(e) = l;
	right(e) = r;
	return e;
}

expr_t*
symbol_expr(sym_t *sym) {
	if (!cond()) return errexpr;
	assert(sym);
	if (!sym->scn) {
		sym->scn = undef_section;
		sym->val = address(sym, undef_section, 0);
	}
	/* XXX: really? */
	return sym->val;
}

expr_t*
address(sym_t *sym, scn_t *scn, word_t off) {
	expr_t *e;

	if (!cond()) return errexpr;

	assert(scn);
	e = palloc(sizeof(expr_t));
	assert(e);
	op(e) = 'a';
	sym(e) = sym;
	scn(e) = scn;
	off(e) = off;
	return e;
}

expr_t*
current_addr(void) {
	if (!cond()) return errexpr;
	return address(NULL, current_section, current_section->size);
}

int
is_abs(expr_t *e) {
	assert(cond());
	assert(e);
	switch (op(e)) {
		case 'a':
			return scn(e) == abs_section;
		case 'b':
		case 'f':
			return 1;
	}
	return 0;
}

expr_t*
check_abs(expr_t *e) {
	assert(e);
	if (cond()) {
		if (e != errexpr && !is_abs(e)) {
			error("absolute expression expected");
			e = errexpr;
		}
	}
	return e;
}

word_t
to_int(expr_t *e) {
	if (!cond()) return 0;
	assert(e);
	if (op(e) != 'a' || scn(e) != abs_section) {
		error("integer value expected");
		return 0;
	}
	return off(e);
}

/*
 * Evaluation rules:
 *
 * {abs a} x {abs b} -> {abs (a x b)}
 * {*** a}  {abs b} -> {scn (a  b)}	# + keep reference to symbol, if any
 * {abs a} + {*** b} -> {scn (a + b)}	# + keep reference to symbol, if any
 * {scn a} - {scn b} -> {abs (a - b)}	# same scn (!= und) only!
 * {*** a} x {*** b} -> error
 */
expr_t*
binary(int w, expr_t *l, expr_t *r) {
	word_t v;

	if (!cond()) return errexpr;

	if (l == errexpr || r == errexpr) {
		return errexpr;
	}
	if (op(l) == 'a' && op(r) == 'a') {
		if (scn(l) == abs_section && scn(r) == abs_section) {
			switch (w) {
				case '*': v = off(l)  * off(r); break;
				case '/': v = off(l)  / off(r); break;
				case '%': v = off(l)  % off(r); break;
				case '+': v = off(l)  + off(r); break;
				case '-': v = off(l)  - off(r); break;
				case SHL: v = off(l) << off(r); break;
				case SHR: v = off(l) >> off(r); break;
				case '<': v = off(l)  < off(r); break;
				case '>': v = off(l)  > off(r); break;
				case LEQ: v = off(l) <= off(r); break;
				case GEQ: v = off(l) >= off(r); break;
				case EQL: v = off(l) == off(r); break;
				case NEQ: v = off(l) != off(r); break;
				case '&': v = off(l)  & off(r); break;
				case '^': v = off(l)  ^ off(r); break;
				case '|': v = off(l)  | off(r); break;
				default: assert(0);
			}
			return address(NULL, abs_section, v);
		}
		if (w == '+') {
			if (scn(r) == abs_section) {
				return address(sym(l), scn(l), off(l) + off(r));
			}
			if (scn(l) == abs_section) {
				return address(sym(r), scn(r), off(l) + off(r));
			}
		}
		else if (w == '-') {
			if (scn(r) == abs_section) {
				return address(sym(l), scn(l), off(l) - off(r));
			}
			if (scn(l) == scn(r) && scn(l)->name) {
				/* pseudo-sections are not allowed */
				assert(scn(l) != undef_section);
				assert(scn(l) != common_section);
				return address(NULL, abs_section, off(l) - off(r));
			}
		}
	}
	error("absolute expression expected");
	return errexpr;
}

expr_t*
unary(int w, expr_t *r) {
	word_t v;

	if (!cond()) return errexpr;

	if (r == errexpr) {
		return errexpr;
	}
	if (op(r) != 'a' || scn(r) != abs_section) {
		error("absolute expression expected");
		return errexpr;
	}
	v = off(r);
	switch (w) {
		case '+': return r;
		case '-': v = -v; break;
		case '~': v = ~v; break;
		case '!': v = !v; break;
		default: assert(0);
	}
	return address(NULL, abs_section, v);
}

static unsigned
digit(unsigned c) {
	if (c - '0' <= '9' - '0') return c - '0';
	if (c - 'A' <= 'Z' - 'A') return c - 'A' + 10;
	if (c - 'a' <= 'z' - 'a') return c - 'a' + 10;
	return ~0u;
}

expr_t*
numeric_expr(const char *val) {
	int check_overflow = 1;
	int base = 10;
	word_t v = 0;
	unsigned x;

	if (!cond()) return errexpr;

	assert(val);
	if (*val == '0') {
		switch (*++val) {
			case 'X': case 'x': base = 16; val++; break;
			case 'B': case 'b': base = 2; val++; break;
			case '\0': return address(NULL, abs_section, 0);
			default: base = 8;
		}
	}
	while (*val) {
		x = digit(*val++ & 0xff);
		if (x >= base) {
			/* the scanner should prevent this */
			fatal("invalid digit `%c' in number", val[-1]);
		}
		if (check_overflow && (~(word_t)0 - x) / base < v) {
			error("numeric value out of range, truncating");
			check_overflow = 0;
		}
		v = base * v + x;
	}
	return address(NULL, abs_section, v);
}

int
is_true(expr_t *e) {
	if (e != errexpr) {
		switch (op(e)) {
			case 'a': return off(e) != 0;
			case 'b': return 1;	/* bignums are never zero, by definition */
			case 'f': fatal("floating-point numbers not supported");
		}
		error("absolute expression expected");
	}
	return 0;
}

expr_t*
register_operand(const char *reg) {
	long rnum;
	expr_t *e;

	if (!cond()) return errexpr;

	rnum = target->encode_register(target, reg);
	if (rnum != -1L) {
		e = palloc(sizeof(expr_t));
		assert(e);
		op(e) = 'r';
		rnum(e) = rnum;
		return e;
	}
	error("register name expected");
	return errexpr;
}

/* allocation */

void
alloc_bytes(expr_t *size, expr_t *fill) {
	word_t len, val;
	char *dest;

	assert(cond());
	len = to_int(size);
	val = fill ? to_int(fill) : 0;
	dest = extend_section(current_section, len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	memset(dest, val, len);
}

static void
put_data(char *dest, size_t len, word_t val, int big_endian) {
	size_t i;

	assert(cond());
	assert(dest);
	if (big_endian) {
		for (i = len; i > 0; ) {
			dest[--i] = val & 0xff;
			val >>= 8;
		}
	}
	else {
		for (i = 0; i < len; ) {
			dest[i++] = val & 0xff;
			val >>= 8;
		}
	}
}

void
alloc_data(long size, expr_t *data) {
	char *dest;
	word_t val;
	size_t len;

	assert(cond());
	assert(size);
	assert(data);
	if (data == errexpr) {
		return;
	}
	assert(op(data) == 'a');
	len = size < 0 ? -size : size;
	if (nobits_section(current_section)) {
		if (scn(data) != abs_section || off(data) != 0) {
			error("initializer must be zero");
		}
		dest = extend_section(current_section, len);
		return;
	}
	if (scn(data) != abs_section) {
		/* try to relocate (may fail) */
		/* otherwise, returns value to store: 0 for RELA, >=0 for REL */
		/* note that add_relocation() may support either endian */
		val = add_relocation(data, size);
	}
	else {
		val = off(data);
	}
	dest = extend_section(current_section, len);
	put_data(dest, len, val, size < 0);
}

void
alloc_fdata(long size, expr_t *data) {
	assert(cond());
	fatal("floating-point numbers not supported");
}

void
alloc_string(int addz, const char *val) {
	char *dest;
	size_t len;

	assert(cond());
	assert(val);
	if (nobits_section(current_section)) {
		error("directive is invalid in NOBITS section");
		return;
	}
	len = strlen(val) + (addz != 0);
	if (len) {
		dest = extend_section(current_section, len);
		memcpy(dest, val, len);
	}
}

void
alloc_fill(expr_t *repeat, expr_t *size, expr_t *fill) {
	word_t count, len, val;
	char *dest;

	assert(cond());
	assert(repeat);
	count = to_int(repeat);
	len = size ? to_int(size) : 1;
	val = fill ? to_int(fill) : 0;
	if (len > 8) len = 8;	/* compatibility */
	if (!count || !len) {
		return;
	}
	dest = extend_section(current_section, count * len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	while (count > 0) {
		/* only support native-endian mode */
		put_data(dest, len, val, target->elf_data == ELFDATA2MSB);
		dest += len;
		count--;
	}
}

/* variable-length encoding used by DWARF */
void
alloc_leb128(int signd, expr_t *data) {
	char temp[2 * sizeof(word_t)];
	word_t val;
	size_t len;
	char *dest;

	assert(cond());
	assert(data);
	val = to_int(data);
	if (nobits_section(current_section)) {
		if (val) {
			error("initializer must be zero");
		}
		/* sanitize */
		extend_section(current_section, 1);
		return;
	}
	len = 0;
	if (signd) {
		/* XXX: is this correct? */
		while (val + 0x40 >= 0x80) {
			temp[len++] = 0x80 | (val & 0x7f);
			val = (sword_t)val >> 7;
		}
	}
	else {
		while (val >= 0x80) {
			temp[len++] = 0x80 | (val & 0x7f);
			val >>= 7;
		}
	}
	temp[len++] = val & 0x7f;
	dest = extend_section(current_section, len);
	memcpy(dest, temp, len);
}

void
align(int mode, expr_t *alignment, expr_t *fill, expr_t *max) {
	word_t how, val, lim, off;
	char *dest;

	assert(cond());
	assert(alignment);
	how = to_int(alignment);
	val = fill ? to_int(fill) : 0;
	lim = max ? to_int(max) : 0;
	if (mode < 0) {
		/* logarithmic mode */
		how = 1ULL << how;
	}
	else if (how & (how - 1)) {
		error("alignment must be a power of 2");
		return;
	}
	if (how > target->max_alignment) {
		error("alignment too large");
		return;
	}
	if (how <= 1) {
		return;
	}
	off = current_section->size % how;
	if (!off || (lim && (how - off) > lim)) {
		return;
	}
	if (current_section->align < how) {
		current_section->align = how;
	}
	dest = extend_section(current_section, how -= off);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("initializer must be zero");
		}
		return;
	}
	if (mode < 0) {
		mode = -mode;
	}
	if (mode > 1) {
		/* multi-byte fill */
		if (how % mode) {
			error("alignment padding (%u bytes) not a multiple of %u",
				(unsigned)how, (unsigned)mode);
			/* sanitize */
			memset(dest, 0, how % mode);
			dest += how % mode;
		}
		while (how >= mode) {
			put_data(dest, mode, val, target->elf_data == ELFDATA2MSB);
			dest += mode;
			how -= mode;
		}
	}
	else {
		memset(dest, val, how);
	}
}

void
new_org(expr_t *off, expr_t *fill) {
	word_t org, val, cur, len;
	char *dest;

	assert(cond());
	assert(off);
	if (off == errexpr) {
		return;
	}
	if (scn(off) == undef_section) {
		assert(sym(off));
		error("undefined symbol \"%s\"", sym(off)->name);
	}
	if (scn(off) != current_section && scn(off) != abs_section) {
		error("new location must be in current (sub)section");
		return;
	}
	org = off(off);
	val = fill ? to_int(fill) : 0;
	cur = current_section->size;
	if (org < cur) {
		error("cannot move location counter backwards");
	}
	if (org <= cur) {
		return;
	}
	len = org - cur;
	dest = extend_section(current_section, len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	memset(dest, val, len);
}

#undef op
#undef left
#undef right
#undef scn
#undef off
#undef blen
#undef bval
#undef sym
#undef rnum

/* symbols */

sym_t *all_syms = NULL;
sym_t *file_syms = NULL;

sym_t*
make_symbol(const char *name) {
	sym_t *sym;

	assert(cond());
	assert(name);
	sym = palloc(sizeof(sym_t) + strlen(name) + 1);
	sym->next = NULL;
	sym->name = strcpy((char*)(sym + 1), name);
	sym->scn = NULL;
	sym->val = NULL;
	sym->type = STT_NOTYPE;
	sym->bind = -1;
	sym->size = 0;
	sym->align = 0;
	sym->relocs = 0;
	sym->elf_index = STN_UNDEF;
	return sym;
}

sym_t*
dummy_symbol(const char *name, scn_t *scn, long type) {
	sym_t *sym;

	assert(cond());
	assert(name);
	assert(scn);
	sym = make_symbol(name);
	sym->scn = scn;
	sym->val = address(sym, scn, 0);
	sym->type = type;
	sym->bind = STB_LOCAL;
	return sym;
}

void
file_symbol(const char *name) {
	sym_t **p;

	assert(cond());
	assert(name);
	for (p = &file_syms; *p; p = &(*p)->next);
	*p = dummy_symbol(name, abs_section, STT_FILE);
	assert((*p)->next == NULL);
}

sym_t*
section_start_symbol(scn_t *scn) {
	assert(scn);
	/* Pseudo sections (ABS, UNDEF, COMMON) don't have names */
	/* They should not appear in this place */
	assert(scn->name);
	if (!scn->dummy) {
		scn->dummy = dummy_symbol("", scn, STT_SECTION);
	}
	return scn->dummy;
}

sym_t*
intern_symbol(const char *name) {
	static unsigned table[10] = { 0 };
	char buf[3*sizeof(*table)+5];
	sym_t **p, *sym;

	if (!cond()) return NULL;
	assert(name);
	/* handle local labels */
	if (*name >= '0' && *name <= '9') {
		unsigned n;

		n = name[0] - '0';
		assert(n < 10);
		switch (name[1]) {
			case 'b': n = table[n]; break;
			case 'f': n = table[n] + 1; break;
			case '\0': n = table[n] += 1; break;
			default: assert(0);
		}
		sprintf(buf, ".L%c-%u", name[0], n);
		name = buf;
	}
	for (p = &all_syms; (sym = *p); p = &sym->next) {
		/* XXX: sort? */
		if (strcmp(name, sym->name) == 0) {
			return sym;
		}
	}
	sym = make_symbol(name);
	if (name == buf) {
		sym->bind = STB_LOCAL;
	}
	sym->next = *p;
	*p = sym;
	return sym;
}

void
set_symbol_value(sym_t *sym, expr_t *val) {
	assert(cond());
	assert(sym);
	assert(val);
	if (val != errexpr && val->any.op != 'a') {
		error("absolute expression expected");
		/* sanitize */
		sym->scn = abs_section;
		sym->val = address(NULL, abs_section, 0);
		return;
	}
	if (val->addr.scn == undef_section || val->addr.scn == common_section) {
		error("value unknown at compile time");
		return;
	}
	if (val->addr.sym) {
		/* copy symbol type, size and alignment */
		sym->type = val->addr.sym->type;
		sym->size = val->addr.sym->size;
		sym->align = val->addr.sym->align;
		set_symbol_binding(sym, val->addr.sym->bind);
	}
	/* tag value with myself */
	/* XXX: really override original symbol? */
	sym->scn = val->addr.scn;
	sym->val = address(sym, val->addr.scn, val->addr.off);
}

void
define_symbol(long unique, sym_t *sym, expr_t *val) {
	assert(cond());
	assert(sym);
	assert(val);
	if (unique && sym->scn && sym->scn != undef_section) {
		error("redefinition of symbol \"%s\"", sym->name);
		return;
	}
	/* XXX: fix relocations? */
	set_symbol_value(sym, val);
}

void
define_common(long local, sym_t *sym, expr_t *size, expr_t *alignment) {
	word_t val, len, algn;

	assert(cond());
	assert(sym);
	assert(size);
	len = to_int(size);
	if (len == 0) {
		error("common symbol \"%s\" is zero bytes long", sym->name);
		/* sanitize */
		len = 1;
	}
	if (alignment) {
		algn = to_int(alignment);
		if (algn & (algn - 1)) {
			error("alignment must be a power of 2");
			return;
		}
		if (algn > target->max_alignment) {
			error("alignment too large");
			return;
		}
	}
	else {
		algn = target->data_alignment;
		assert((algn & (algn - 1)) == 0);
		while (algn > 1 && len % algn) {
			algn >>= 1;
		}
	}
	if (sym->scn) {
		error("redefinition of symbol \"%s\"", sym->name);
		return;
	}
	if (local || sym->bind == STB_LOCAL || sym->bind == STB_WEAK) {
		/* allocate space in .bss section */
		val = bss_section->size;
		if (algn > 1) {
			val += algn - 1;
			val -= val % algn;
		}
		sym->scn = bss_section;
		sym->val = address(sym, bss_section, val);
		sym->type = STT_OBJECT;		/* gas uses STT_NOTYPE ?! */
		if (sym->bind == -1) {
			sym->bind = STB_LOCAL;
		}
		sym->size = len;
		bss_section->size = val + len;
		if (bss_section->align < algn) {
			bss_section->align = algn;
		}
	}
	else {
		/* put symbol in dummy section */
		sym->scn = common_section;
		sym->val = address(sym, common_section, 0);
		sym->type = STT_OBJECT;
		sym->bind = STB_GLOBAL;
		sym->size = len;
		sym->align = algn;
	}
}

void
set_symbol_binding(sym_t *sym, long bind) {
	assert(cond());
	assert(sym);
	if (sym->bind == -1) {
		sym->bind = bind;
	}
}

void
set_symbol_type(sym_t *sym, long type) {
	assert(cond());
	assert(sym);
	sym->type = type;
}

void
set_symbol_size(sym_t *sym, word_t size) {
	assert(cond());
	assert(sym);
	sym->size = size;
}

int
locally_defined(sym_t *sym) {
	return cond() && sym->scn && sym->scn != undef_section;
}

/* the hardest part... */

void
emit_instruction(const char *name, expr_t *args) {
	assert(cond());
	target->emit(target, name, args);
}

void
emit_stab(const char *name, expr_t *type, expr_t *other, expr_t *desc, expr_t *value) {
	assert(cond());
	if (!target->emit_stab) {
		error("target does not support stabs");
		return;
	}
	target->emit_stab(target,
		name, to_int(type), to_int(other), to_int(desc), value);
}

/* relocations */

void
add_reloc_entry(scn_t *scn, reloc_t *rel) {
	size_t n;

	assert(cond());
	assert(scn);
	assert(rel);
	assert(rel->sym);
	rel->sym->relocs = 1;
	if (scn->nrelocs % 16u == 0) {
		n = scn->nrelocs + 16u;
		scn->relocs = xrealloc(scn->relocs, n * sizeof(reloc_t));
	}
	scn->relocs[scn->nrelocs++] = *rel;
}

word_t
add_relocation(expr_t *val, long size) {
	sym_t *sym;
	reloc_t rel;
	word_t addend;

	assert(cond());
	assert(val);
	assert(val->any.op == 'a');
	assert(val->addr.scn);
	assert(val->addr.scn != abs_section);
	if (!(sym = val->addr.sym)) {
		sym = section_start_symbol(val->addr.scn);
		assert(sym);
	}
	assert(val->addr.scn == sym->scn);
	addend = val->addr.off;
	/* XXX: what happens if the symbol is defined later? */
	if (sym->scn != undef_section && sym->scn != common_section) {
		assert(sym->scn->name[0]);
		assert(sym->val->any.op == 'a');
		addend = addend - sym->val->addr.off;
	}
	rel.sym = sym;
	rel.off = current_section->size;
	rel.addend = addend;
	/* must come last */
	rel.type = target->reloc_type(target, &rel, size);
	if (rel.type == -1) {
		/* target does not support this */
		error("absolute expression expected");
		return 0;
	}
	add_reloc_entry(current_section, &rel);
	return target->elf_reloc_type == ELF_T_RELA ? 0 : addend;
}

/* F-CPU specific stuff */

#include <fcpu_opcodes.h>
#include <elf_FCPU.h>

int loadcons_warn = 1;
int keep_local_symbols = 0;

static long
fcpu_reloc_type(const target_t *tp, reloc_t *rel, long size) {
	assert(cond());
	assert(tp);
	assert(rel);
	switch (size) {
		case  0: return R_FCPU_NONE;
		case  8: return R_FCPU_LE_64;
		case -8: return R_FCPU_BE_64;
		/* XXX: PC-relative stuff */
	}
	return -1L;
}

static long
fcpu_regnum(const target_t *tp, const char *reg) {
	unsigned long rn;
	char *s;

	assert(cond());
	assert(tp);
	assert(reg);
	if (*reg == 'R' || *reg == 'r') {
		rn = strtoul(reg + 1, &s, 10);
		assert(s);
		if (s != reg + 1 && !*s && rn < 64) {
			return (long)rn;
		}
	}
	return -1L;
}

#define more_args(args)	((args) != NULL)

static expr_t*
unchain_arg(expr_t **list) {
	expr_t *e;

	assert(list);
	if (more_args(*list)) {
		assert((*list)->any.op == ',');
		e = (*list)->bin.left;
		*list = (*list)->bin.right;
		assert(e->any.op != ',');
		return e;
	}
	return NULL;
}

static int
expect_register(expr_t **list, int *r) {
	expr_t *e;

	if (!(e = unchain_arg(list)) || e->any.op != 'r') {
		error("register expected");
		return -1;
	}
	if (r) *r = e->reg.rnum & 0x3f;
	return 0;
}

static int
expect_expr(expr_t **list, expr_t **resp) {
	expr_t *e;

	assert(resp);
	if (!(e = unchain_arg(list))) {
		error("integer expression expected");
		return -1;
	}
	if (e == errexpr) {
		return -1;
	}
	if (e->any.op != 'a') {
		error("integer expression expected");
		return -1;
	}
	*resp = e;
	return 0;
}

static int
expect_integer(expr_t **list, int *r, int bits) {
	word_t v, w;
	expr_t *e;

	assert(r);
	assert(bits);
	if (expect_expr(list, &e)) {
		return -1;
	}
	if (e->addr.scn != abs_section) {
		error("absolute expression expected");
		return -1;
	}
	v = e->addr.off;
	/* check range */
	if (bits < 0) {
		w = 1ULL << -bits;
		if (v + (w >> 1) >= w) {
			warn("integer constant out of range [-%x;%x]",
				(int)(w >> 1), (int)(w >> 1) - 1);
		}
	}
	else {
		w = 1ULL << bits;
		if (v >= w) {
			warn("integer constant out of range [0;%x]",
				(unsigned)w - 1);
		}
	}
	*r = v & (w - 1);
	return 0;
}

static void
put_insn(unsigned opcode, unsigned flags) {
	char *dest;

	opcode = (opcode << 24) | flags;
	dest = extend_section(current_section, 4);
	put_data(dest, 4, opcode, INSTRUCTION_BIG_ENDIAN);
}

static void
fcpu_emit_loadaddri(unsigned code, int sig, expr_t *args) {
	expr_t *e;
	word_t v;
	int r;

	if (expect_expr(&args, &e)) return;
	if (expect_register(&args, &r)) return;
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	if (e->addr.scn == abs_section) {
		v = e->addr.off;
		if (v + 0x10000 >= 0x20000) {
			error("displacement out of range");
			/* sanitize */
			v = 0;
		}
		code |= ((unsigned)v & 0x1ffff) << 6;
	}
	else {
		reloc_t rel;
		sym_t *sym;

		if (!(sym = e->addr.sym)) {
			assert(e->addr.scn->name[0]);
			sym = section_start_symbol(e->addr.scn);
			assert(sym);
		}
		assert(sym->scn == e->addr.scn);
		v = e->addr.off;
		/* XXX: what happens if the symbol is defined later? */
		if (sym->scn != undef_section && sym->scn != common_section) {
			assert(sym->scn->name[0]);
			assert(sym->val);
			assert(sym->val->any.op == 'a');
			v = v - sym->val->addr.off;
		}
		/* add relocation entry */
		rel.sym = sym;
		rel.off = current_section->size;
		rel.type = R_FCPU_PC17;
		rel.addend = v - 4;		/* addr = . + 4 + disp */
		add_reloc_entry(current_section, &rel);
	}
	put_insn(0, code | r);
}

static void
fcpu_emit_loadcons(unsigned code, int sig, expr_t *args) {
	expr_t *e;
	word_t v;
	int r;
	int i;

	if (expect_expr(&args, &e)) return;
	if (expect_register(&args, &r)) return;
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	if (e->addr.scn == abs_section) {
		/* absolute expression */
		/* XXX: optimize this further? */
		v = e->addr.off;
		if (code == (OP_LOADCONSX << 24)) {
			/* loadconsx */
			for (i = 0; i < 3 && v + 0x8000 >= 0x10000; i++) {
				put_insn(OP_LOADCONS, (i << 22) | (v & 0xffff) << 6 | r);
				v = (sword_t)v >> 16;
			}
			put_insn(OP_LOADCONSX, (i++ << 22) | (v & 0xffff) << 6 | r);
		}
		else {
			assert(code == (OP_LOADCONS << 24));
			/* loadcons */
			/* XXX: what else can we do? */
			for (i = 0; i < 4; i++) {
				put_insn(OP_LOADCONS, (i << 22) | (v & 0xffff) << 6 | r);
				v = v >> 16;
			}
		}
	}
	else {
		/* XXX: make a difference between loadcons and loadconsx? */
		static const int reloc_types[] = {
			R_FCPU_U_16, R_FCPU_U_32, R_FCPU_U_48, R_FCPU_U_64,
		};
		reloc_t rel;
		sym_t *sym;

		if (!(sym = e->addr.sym)) {
			sym = section_start_symbol(e->addr.scn);
			assert(sym);
		}
		assert(sym->scn == e->addr.scn);
		v = e->addr.off;
		/* XXX: what happens if the symbol is defined later? */
		if (sym->scn != undef_section && sym->scn != common_section) {
			assert(sym->scn->name[0]);
			assert(sym->val);
			assert(sym->val->any.op == 'a');
			v = v - sym->val->addr.off;
		}
		for (i = 0; i < 4; i++) {
			/* add relocation entry */
			rel.sym = sym;
			rel.off = current_section->size + 4 * i;
			rel.type = reloc_types[i];
			rel.addend = v;
			add_reloc_entry(current_section, &rel);
		}
		/* emit dummy loads (values are filled in by ld) */
		put_insn(OP_LOADCONS, (0 << 22) | r);
		put_insn(OP_LOADCONS, (1 << 22) | r);
		put_insn(OP_LOADCONS, (2 << 22) | r);
		put_insn(0, code | (3 << 22) | r);
	}
	if (loadcons_warn && i > 1) {
		warn("emitted %d instructions", i);
	}
}

static void
fcpu_emit(const target_t *tp, const char *name, expr_t *args) {
	extern int fcpu_encode_instruction(const char *name);
	int code, sig, tmp, r;

	assert(tp);
	assert(name);
	/* get opcode + instruction `signature' */
	code = fcpu_encode_instruction(name);
	if (code == -1 || code == -2) {
		/* invalid or unsupported instruction */
		return;
	}
	/* check arguments */
	sig = code & 0x3f;
	code &= ~0x3f;
	switch (sig) {
		default: assert(0);
		case ARG_NONE:	/* (halt, rfe, srb_save, srb_restore, serialize, nop) */
			break;
		case ARG_R:		/* reg (loopentry) */
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_RO:	/* reg[,reg] (jmpa) */
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (more_args(args)) {
				if (expect_register(&args, &r)) return;
				code |= r << 0;
			}
			break;
		case ARG_RR:	/* reg,reg (common) */
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_U16R:	/* uimm16,reg (loadcons.n, geti, puti, syscall, trap) */
			if (expect_integer(&args, &r, 16)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S16R:	/* simm16,reg (loadconsx.n) */
			if (expect_integer(&args, &r, -16)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S17R:	/* simm17,reg (loadaddri) */
			fcpu_emit_loadaddri(code, sig, args);
			return;
		case ARG_U64R:	/* uimm64,reg (generic loadcons) */
		case ARG_S64R:	/* simm64,reg (generic loadconsx) */
			fcpu_emit_loadcons(code, sig, args);
			return;
		case ARG_ORR:	/* [reg,]reg,reg (popcount, load[f], store[f]) */
			if (expect_register(&args, &r)) return;
			if (expect_register(&args, &tmp)) return;
			r = (r << 6) | tmp;
			if (more_args(args)) {
				if (expect_register(&args, &tmp)) return;
				r = (r << 6) | tmp;
			}
			code |= r;
			break;
		case ARG_RRO:	/* reg,reg[,reg] (jmpa{cc}) */
			if (expect_register(&args, &r)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (more_args(args)) {
				if (expect_register(&args, &r)) return;
				code |= r << 0;
			}
			break;
		case ARG_RRR:	/* reg,reg,reg (common) */
			if (expect_register(&args, &r)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_U8RR:	/* uimm8,reg,reg (common) */
			if (expect_integer(&args, &r, 8)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S8RR:	/* simm8,reg,reg (muli, divi, modi?) */
			if (expect_integer(&args, &r, -8)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S9RR:	/* simm9,reg,reg (loadi[f], storei[f]) */
			if (expect_integer(&args, &r, -9)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
	}
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	put_insn(0, code);
}

target_t
fcpu_target = {
	/* reloc_type        */ fcpu_reloc_type,
	/* encode_register   */ fcpu_regnum,
	/* emit              */ fcpu_emit,
	/* emit_stab         */ NULL,			/* not supported yet */
	/* text_alignment    */ 4,
	/* data_alignment    */ 8,
	/* max_alignment     */ 4096,
	/* all_start_symbols */ 1,
	/* elf_class         */ ELFCLASS64,
	/* elf_data          */ ELFDATA2LSB,
	/* elf_machine       */ EM_FCPU,
	/* elf_reloc_size    */ sizeof(Elf64_Rela),
	/* elf_symbol_size   */ sizeof(Elf64_Sym),
	/* elf_reloc_type    */ ELF_T_RELA,
	/* elf_reloc_section */ SHT_RELA,
	/* elf_alloc_reloc   */ 0,
	/* elf_alloc_symtab  */ 0,
};

/* ELF stuff */

#include <gelf.h>

typedef struct strtab {
	char	*data;
	size_t	size;
} strtab_t;

size_t
addstr(strtab_t *st, const char *str) {
	size_t off, len, n;

	assert(st);
	assert(str);
	off = st->size;
	if (off == 0) {
		st->data = xmalloc(1024);
		*st->data = '\0';
		st->size = off = 1;
	}
	len = strlen(str);
	if (len == 0) {
		return 0;
	}
	st->size = off + len + 1;
	if ((n = st->size + 1023) / 1024 > (off + 1023) / 1024) {
		st->data = xrealloc(st->data, n - n % 1024);
	}
	strcpy(st->data + off, str);
	return off;
}

size_t
add2str(strtab_t *st, const char *str1, const char *str2) {
	size_t off, len, n;

	assert(st);
	assert(str1);
	assert(str2);
	off = st->size;
	if (off == 0) {
		st->data = xmalloc(1024);
		*st->data = '\0';
		st->size = off = 1;
	}
	len = strlen(str1) + strlen(str2);
	if (len == 0) {
		return 0;
	}
	st->size = off + len + 1;
	if ((n = st->size + 1023) / 1024 > (off + 1023) / 1024) {
		st->data = xrealloc(st->data, n - n % 1024);
	}
	strcpy(st->data + off, str1);
	strcat(st->data + off, str2);
	return off;
}

void
elf_error(void) {
	fatal("libelf: %s", elf_errmsg(-1));
}

void
copy_symbol(strtab_t *names, Elf_Data *data, size_t n, sym_t *sym) {
	const size_t NSYMS = 256;
	GElf_Sym esym;

	/*
	 * make space
	 */
	if (n % NSYMS == 0) {
		data->d_buf = xrealloc(data->d_buf,
			(n + NSYMS) * target->elf_symbol_size);
	}
	data->d_size = (n + 1) * target->elf_symbol_size;

	/*
	 * copy symbol structure
	 */
	if (n == 0) {
		/* first member is always empty */
		assert(!sym);
		esym.st_name = addstr(names, "");
		assert(esym.st_name == 0);
		esym.st_info = 0;
		esym.st_other = 0;
		esym.st_shndx = SHN_UNDEF;
		esym.st_value = 0;
		esym.st_size = 0;
	}
	else {
		assert(sym);
		assert(sym->bind != -1);
		esym.st_name = addstr(names, sym->name);
		esym.st_other = 0;
		assert(sym->scn);
		if (sym->scn == abs_section) {
			esym.st_shndx = SHN_ABS;
			assert(sym->val);
			esym.st_value = to_int(sym->val);
		}
		else if (sym->scn == undef_section) {
			esym.st_shndx = SHN_UNDEF;
			esym.st_value = 0;
			assert(sym->bind == STB_GLOBAL || sym->bind == STB_WEAK);
		}
		else if (sym->scn == common_section) {
			esym.st_shndx = SHN_COMMON;
			esym.st_value = sym->align;
			assert(sym->bind == STB_GLOBAL);
		}
		else {
			assert(sym->scn->elf_scn);
			esym.st_shndx = elf_ndxscn(sym->scn->elf_scn);
			assert(sym->val);
			assert(sym->val->any.op == 'a');
			assert(sym->val->addr.scn == sym->scn);
			esym.st_value = sym->val->addr.off;
		}
		esym.st_info = GELF_ST_INFO(sym->bind, sym->type);
		esym.st_size = sym->size;
		sym->elf_index = n;
	}
	if (!gelf_update_sym(data, n, &esym)) elf_error();
}

static int
local_label(sym_t *sym) {
	return sym->name[0] == '.' && sym->name[1] == 'L'
		&& (unsigned)(sym->name[2] - '0') < 10u
		&& sym->name[3] == '-';
}

void
fix_symbols(void) {
	sym_t *sym;

	for (sym = all_syms; sym; sym = sym->next) {
		if (sym->scn == undef_section) {
			if (sym->bind == STB_LOCAL) {
				if (local_label(sym)) {
					error("instance %s of local label \"%c\" is undefined",
						&sym->name[4], sym->name[2]);
				}
				else {
					error("undefined symbol \"%s\"", sym->name);
				}
			}
			if (sym->bind < STB_GLOBAL) {
				sym->bind = STB_GLOBAL;
			}
		}
		else if (sym->bind < STB_LOCAL) {
			sym->bind = STB_LOCAL;
		}
	}
}

void
fix_relocs(void) {
	scn_t *scn;
	reloc_t *r;
	size_t n;

	/* XXX: skip this if we keep local symbols? */
	for (scn = all_scns; scn; scn = scn->next) {
		for (n = 0; n < scn->nrelocs; n++) {
			r = scn->relocs + n;
			assert(r->sym);
			assert(r->sym->scn);
			if (r->sym->scn == undef_section) continue;
			if (r->sym->scn == common_section) continue;
			if (r->sym->bind > STB_LOCAL) continue;
			/* change symbol reference */
			assert(r->sym->val);
			assert(r->sym->val->any.op == 'a');
			assert(r->sym->val->addr.scn == r->sym->scn);
			r->addend += r->sym->val->addr.off;	/* XXX: really? */
			r->sym = section_start_symbol(r->sym->scn);
			assert(r->sym);
		}
	}
}

void
write_elf_file(const char *fname) {
	strtab_t scn_names = { NULL, 0 };
	strtab_t sym_names = { NULL, 0 };
	GElf_Ehdr etmp, *ehdr;
	GElf_Shdr stmp, *shdr;
	Elf *elf;
	Elf_Data *data;
	Elf_Scn *escn;
	Elf_Scn *symscn;
	Elf_Scn *strscn;
	scn_t *scn;
	sym_t *sym;
	int fd;
	size_t n;

	assert(fname);
	if ((fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0666)) == -1) {
		fatal("%s: %s", fname, strerror(errno));
	}
	if (!(elf = elf_begin(fd, ELF_C_WRITE, NULL))) elf_error();
	if (!gelf_newehdr(elf, target->elf_class)) elf_error();
	if (!(ehdr = gelf_getehdr(elf, &etmp))) elf_error();
	ehdr->e_ident[EI_DATA] = target->elf_data;
	ehdr->e_type = ET_REL;
	ehdr->e_machine = target->elf_machine;
	ehdr->e_version = EV_CURRENT;
	ehdr->e_entry = 0;
	ehdr->e_flags = 0;
	//ehdr->e_shstrndx = 0;	/* filled in later */
	if (!gelf_update_ehdr(elf, ehdr)) elf_error();
	for (scn = all_scns; scn; scn = scn->next) {
		/* create data section */
		assert(scn->name);
		assert(!scn->elf_scn);
		assert(scn->type == SHT_NOBITS || scn->type == SHT_PROGBITS);
		if (!(escn = elf_newscn(elf))) elf_error();
		if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
		shdr->sh_name = addstr(&scn_names, scn->name);
		shdr->sh_type = scn->type;
		shdr->sh_flags = scn->flags;
		shdr->sh_addr = 0;
		shdr->sh_link = 0;
		shdr->sh_info = 0;
		shdr->sh_entsize = 0;
		if (!gelf_update_shdr(escn, shdr)) elf_error();
		if (!(data = elf_newdata(escn))) elf_error();
		data->d_buf = scn->data;
		assert(scn->type != SHT_NOBITS || data->d_buf == NULL);
		data->d_type = ELF_T_BYTE;
		data->d_size = scn->size;
		data->d_align = scn->align;
		data->d_version = EV_CURRENT;
		scn->elf_scn = escn;
		if (scn->relocs && scn->nrelocs) {
			static const char *pfx[] = { ".rel", ".rela" };

			/* create a relocation section */
			if (!(escn = elf_newscn(elf))) elf_error();
			if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
			shdr->sh_name = add2str(&scn_names,
				pfx[target->elf_reloc_type == ELF_T_RELA], scn->name);
			shdr->sh_type = target->elf_reloc_section;
			shdr->sh_flags = target->elf_alloc_reloc ? SHF_ALLOC : 0;
			shdr->sh_addr = 0;
			shdr->sh_link = 0;		/* symbol table index added later */
			shdr->sh_info = elf_ndxscn(scn->elf_scn);
			shdr->sh_entsize = gelf_fsize(elf, target->elf_reloc_type, 1, EV_CURRENT);
			if (!gelf_update_shdr(escn, shdr)) elf_error();
			scn->elf_rel = escn;
		}
	}
	/*
	 * create .symtab / .strtab
	 */
	if (!(symscn = elf_newscn(elf))) elf_error();
	if (!(strscn = elf_newscn(elf))) elf_error();
	if (!(shdr = gelf_getshdr(symscn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".symtab");
	shdr->sh_type = SHT_SYMTAB;
	shdr->sh_flags = target->elf_alloc_symtab ? SHF_ALLOC : 0;
	shdr->sh_addr = 0;
	shdr->sh_link = elf_ndxscn(strscn);
	shdr->sh_info = 0;	/* index of first public symbol */
	shdr->sh_entsize = gelf_fsize(elf, ELF_T_SYM, 1, EV_CURRENT);
	if (!gelf_update_shdr(symscn, shdr)) elf_error();
	if (!(data = elf_newdata(symscn))) elf_error();
	data->d_buf = NULL;
	data->d_type = ELF_T_SYM;
	data->d_size = 0;
	data->d_align = gelf_fsize(elf, ELF_T_ADDR, 1, EV_CURRENT);
	data->d_version = EV_CURRENT;
	/*
	 * copy symbols
	 */
	n = 0;
	copy_symbol(&sym_names, data, n++, NULL);
	for (sym = file_syms; sym; sym = sym->next) {
		assert(sym->type == STT_FILE);
		copy_symbol(&sym_names, data, n++, sym);
	}
	for (scn = all_scns; scn; scn = scn->next) {
		if ((sym = scn->dummy)) {
			assert(sym->type == STT_SECTION);
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	for (sym = all_syms; sym; sym = sym->next) {
		if (!sym->scn) continue;
		if (!keep_local_symbols) {
			/* detect local symbols by name */
			if (sym->name[0] == '.' && sym->name[1] == 'L') continue;
		}
		if (sym->bind <= STB_LOCAL) {
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	shdr->sh_info = n;
	for (sym = all_syms; sym; sym = sym->next) {
		if (!sym->scn) continue;
		if (sym->bind > STB_LOCAL) {
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	/* update shdr->sh_info */
	if (!gelf_update_shdr(symscn, shdr)) elf_error();
	/*
	 * create .strtab
	 */
	if (!(shdr = gelf_getshdr(strscn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".strtab");
	shdr->sh_type = SHT_STRTAB;
	shdr->sh_flags = target->elf_alloc_symtab ? SHF_ALLOC : 0;
	shdr->sh_addr = 0;
	shdr->sh_link = 0;
	shdr->sh_info = 0;
	shdr->sh_entsize = 0;
	if (!gelf_update_shdr(strscn, shdr)) elf_error();
	if (!(data = elf_newdata(strscn))) elf_error();
	data->d_buf = sym_names.data;
	data->d_type = ELF_T_BYTE;
	data->d_size = sym_names.size;
	data->d_align = 0;
	data->d_version = EV_CURRENT;
	/*
	 * create .shstrtab
	 */
	if (!(escn = elf_newscn(elf))) elf_error();
	if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".shstrtab");
	shdr->sh_type = SHT_STRTAB;
	shdr->sh_flags = 0;
	shdr->sh_addr = 0;
	shdr->sh_link = 0;
	shdr->sh_info = 0;
	shdr->sh_entsize = 0;
	if (!gelf_update_shdr(escn, shdr)) elf_error();
	if (!(data = elf_newdata(escn))) elf_error();
	data->d_buf = scn_names.data;
	data->d_type = ELF_T_BYTE;
	data->d_size = scn_names.size;
	data->d_align = 0;
	data->d_version = EV_CURRENT;
	ehdr->e_shstrndx = elf_ndxscn(escn);
	if (!gelf_update_ehdr(elf, ehdr)) elf_error();
	/*
	 * copy relocs
	 */
	for (scn = all_scns; scn; scn = scn->next) {
		if (!(escn = scn->elf_rel)) continue;
		/* update header */
		if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
		shdr->sh_link = elf_ndxscn(symscn);
		if (!gelf_update_shdr(escn, shdr)) elf_error();
		/* copy data */
		if (!(data = elf_newdata(escn))) elf_error();
		data->d_type = target->elf_reloc_type;
		data->d_size = scn->nrelocs * target->elf_reloc_size;
		data->d_buf = xmalloc(data->d_size);
		data->d_align = gelf_fsize(elf, ELF_T_ADDR, 1, EV_CURRENT);
		data->d_version = EV_CURRENT;
		for (n = 0; n < scn->nrelocs; n++) {
			reloc_t *r = scn->relocs + n;
			union {
				GElf_Rela rela;
				GElf_Rel rel;
			} u;

			/* assume GElf_Rel is contained in GElf_Rela */
			/* ISO C says it should be, but check anyway */
			assert(&u.rel.r_offset == &u.rela.r_offset);
			assert(&u.rel.r_info   == &u.rela.r_info);

			assert(r->sym);
			assert(r->sym->elf_index);
			u.rela.r_offset = r->off;
			u.rela.r_info = GELF_R_INFO((GElf_Xword)r->sym->elf_index,
										(GElf_Xword)r->type);
			u.rela.r_addend = r->addend;
			if (target->elf_reloc_type == ELF_T_RELA) {
				if (!gelf_update_rela(data, n, &u.rela)) elf_error();
			}
			else {
				if (!gelf_update_rel(data, n, &u.rel)) elf_error();
			}
		}
	}
	/*
	 * and finally, write the file
	 */
	if (!elf_update(elf, ELF_C_WRITE)) elf_error();
	if (elf_end(elf) != 0) {
		fatal("libelf: elf_{begin,end} nesting error");
	}
	if (close(fd)) {
		fatal("%s: %s", fname, strerror(errno));
	}
	xfree(scn_names.data);
	xfree(sym_names.data);
	/* XXX: free symbol table / section data */
}

/* dumb main program */

extern int errors, warnings;

extern void set_input_file(const char *name, FILE *fp);
extern void push_input_file(const char *name, FILE *fp);

void
include_file(const char *name) {
	FILE *fp;

	if ((fp = fopen(name, "r"))) {
		push_input_file(name, fp);
	}
	else {
		error("%s: %s", name, strerror(errno));
	}
}

int
main(int argc, char **argv) {
	FILE *fp;
	int i;

	if (elf_version(EV_CURRENT) == EV_NONE) elf_error();
	elf_fill(0xFC);	/* padding, just for fun */
	errors = warnings = 0;
	if (argc > 1) {
		for (i = 1; i < argc; i++) {
			if (!(fp = fopen(argv[i], "r"))) {
				fatal("%s: %s\n", argv[i], strerror(errno));
			}
			set_input_file(argv[i], fp);
			yyparse();
			fclose(fp);
		}
	}
	else {
		set_input_file("", stdin);
		yyparse();
	}
	/* leave fingerprint */
	add_comment("Generic ELF Assembler v0.0");
	/* prepare output */
	fix_symbols();
	fix_relocs();
	if (errors) {
		fprintf(stderr, "%d errors, %d warnings\n", errors, warnings);
		exit(1);
	}
	if (warnings) {
		fprintf(stderr, "%d warnings\n", warnings);
	}
	/* write output file */
	write_elf_file("a.out");
	exit(0);
}
