/* Definitions of target machine for GNU compiler.  F-CPU version.
   devik@cdi.cz
 */

#include "config.h"
#include "system.h"
#include "rtl.h"
#include "function.h"
#include "output.h"
#include "tree.h"
#include "expr.h"
#include "regs.h"
#include "flags.h"
#include "hard-reg-set.h"
#include "tm_p.h"
#include "target.h"
#include "target-def.h"
#include "insn-config.h"
#include "recog.h"
#include "toplev.h"

extern const char *reg_names[];

static bool fcpu_cannot_modify_jumps_p PARAMS((void));

/* Don't output integers as a sequence of bytes -MR */
#undef  TARGET_ASM_ALIGNED_HI_OP
#define TARGET_ASM_ALIGNED_HI_OP	"\t.short "
#undef  TARGET_ASM_ALIGNED_SI_OP
#define TARGET_ASM_ALIGNED_SI_OP	"\t.int "
#undef  TARGET_ASM_ALIGNED_DI_OP
#define TARGET_ASM_ALIGNED_DI_OP	"\t.long "
#undef  TARGET_ASM_ALIGNED_TI_OP
#define TARGET_ASM_ALIGNED_TI_OP	"\t.llong "
#undef  TARGET_ASM_UNALIGNED_HI_OP
#define TARGET_ASM_UNALIGNED_HI_OP	TARGET_ASM_ALIGNED_HI_OP
#undef  TARGET_ASM_UNALIGNED_SI_OP
#define TARGET_ASM_UNALIGNED_SI_OP	TARGET_ASM_ALIGNED_SI_OP
#undef  TARGET_ASM_UNALIGNED_DI_OP
#define TARGET_ASM_UNALIGNED_DI_OP	TARGET_ASM_ALIGNED_DI_OP
#undef  TARGET_ASM_UNALIGNED_TI_OP
#define TARGET_ASM_UNALIGNED_TI_OP	TARGET_ASM_ALIGNED_TI_OP

#undef TARGET_CANNOT_MODIFY_JUMPS_P
#define TARGET_CANNOT_MODIFY_JUMPS_P fcpu_cannot_modify_jumps_p

/* we need scratch register for jumps so no jumps after reload are possible */
static bool
fcpu_cannot_modify_jumps_p ()
{
	  return (reload_in_progress || reload_completed);
}

struct gcc_target targetm = TARGET_INITIALIZER;


/* returns valid postmodify of addr by offset. Because
   called afted reload it handles case of big offset */
static rtx reload_modifyptr(rtx addr,int offset)
{
    rtx off = GEN_INT(offset);
    
    if (offset < -256 || offset > 255) {
	rtx tmp = gen_rtx_REG(Pmode,31); /* use R31 as intermediate */
	emit_move_insn(tmp,off);
	off = tmp;
    }
    return gen_rtx_POST_MODIFY(Pmode,addr,gen_rtx_PLUS(Pmode,addr,off));
}

/* returns 1 if zextop is ZERO_EXTEND with mode equal to op[0] and larger 
   than op[1]. Optionaly check whether op[op2] has the same mode as op[1],
   op2 can be constant too */
int 
zero_extend_insn_ok(op,zextop,op2)
	rtx op[];
	int zextop,op2;
{
	return (!zextop || (
		GET_CODE(op[zextop]) == ZERO_EXTEND
		&& GET_MODE(op[zextop]) == GET_MODE(op[0])
		&& (!op2 || GET_MODE(op[1]) == GET_MODE(op[op2]) 
			|| GET_CODE(op[op2]) == CONST_INT)
		&& GET_MODE_CLASS(GET_MODE(op[0])) == MODE_INT
		&& GET_MODE_CLASS(GET_MODE(op[1])) == MODE_INT
		&& GET_MODE_SIZE(GET_MODE(op[0])) <= 8
		&& GET_MODE_SIZE(GET_MODE(op[0])) > GET_MODE_SIZE(GET_MODE(op[1]))));
}

int 
mode_check(op,op1,op2,op3,op4)
	rtx op[];
	int op1,op2,op3,op4;
{
	return (GET_MODE(op[op1]) == GET_MODE(op[op2]) &&
		GET_MODE(op[op1]) == GET_MODE(op[op3]) &&
		GET_MODE(op[op1]) == GET_MODE(op[op4]));
}
		
void fcpu_expand_epilogue()
{
    int regno,mknote = 0;
    rtx lastsrc = NULL, sp = gen_rtx_REG(Pmode,STACK_POINTER_REGNUM);
    int sz = ((get_frame_size() + 3) & ~3);
    
    for (regno = 1; regno < FIRST_PSEUDO_REGISTER; regno++) {
	int rn = regno;
	if (rn == FRAME_POINTER_REGNUM || rn == 63)
	    rn ^= 63 ^ FRAME_POINTER_REGNUM;
	if (regs_ever_live[rn] && !call_used_regs[rn]) {
	    if (lastsrc) 
		emit_move_insn(lastsrc,gen_rtx_MEM(DImode,
			    gen_rtx_POST_MODIFY(Pmode,sp,
		    	gen_rtx_PLUS(Pmode,sp,GEN_INT(8)))));
	    else 
		emit_move_insn(sp,gen_rtx_PLUS(Pmode,sp,GEN_INT(8)));
	    
	    lastsrc = gen_rtx_REG(DImode,rn);
	}
    }
    if (!lastsrc) {
	lastsrc = gen_rtx_REG(DImode,0); mknote = 1;
    }
    /* deallocate locals here */
    if (sz || !mknote)
	lastsrc = emit_move_insn(lastsrc,gen_rtx_MEM(DImode,
		    reload_modifyptr(sp,sz)));

    if (mknote && sz)
	/* ensure that it will not be optimized away */
	REG_NOTES (lastsrc) = gen_rtx_EXPR_LIST (REG_INC, sp, 0);
    
    emit_jump_insn (gen_return_internal ());
}
        
void fcpu_expand_prologue()
{
    int regno;
    rtx sp = gen_rtx_REG(Pmode,STACK_POINTER_REGNUM);
    int sz = ((get_frame_size() + 3) & ~3);
    
    if (frame_pointer_needed) {
	regs_ever_live[FRAME_POINTER_REGNUM] = 1;
	call_used_regs[FRAME_POINTER_REGNUM] = 0;
    }

    /* local variables space */
    if (sz) {
	rtx insn = emit_move_insn(gen_rtx_REG(DImode,0),
		gen_rtx_MEM(DImode,reload_modifyptr(sp,-sz)));
	/* ensure that it will not be optimized away */
	REG_NOTES (insn) = gen_rtx_EXPR_LIST (REG_INC, sp, 0);
    }
    
    /* save all used regs; start with FP */
    call_used_regs[63] = 0;
    for (regno = FIRST_PSEUDO_REGISTER-1; regno > 0; regno--) {
	int rn = regno;
	if (rn == FRAME_POINTER_REGNUM || rn == 63)
	    rn ^= 63 ^ FRAME_POINTER_REGNUM;
	if (regs_ever_live[rn] && !call_used_regs[rn]) {
	    rtx r = gen_rtx_POST_MODIFY(Pmode,sp,
		    	gen_rtx_PLUS(Pmode,sp,GEN_INT(-8)));
	    
	    emit_move_insn(gen_rtx_MEM(DImode,r),gen_rtx_REG(DImode,rn));
	}
    }
}
	
int
fcpu_need_fp_p ()
{
	return (current_function_calls_alloca
			|| current_function_varargs
			|| fcpu_init_fp_off () > 127);
}

int
fcpu_init_fp_off()
{ 
	int regno;
	int offset = 0;	
	for (regno=0; regno < FIRST_PSEUDO_REGISTER; regno++)
		if (regs_ever_live[regno] && !call_used_regs[regno])
			offset += 8;						
	return offset + ((get_frame_size() + 3) & ~3);
}

const char *fcpu_branch_zero_code(enum rtx_code);
static const char *
fcpu_branch_zero_code (x)
     enum rtx_code x;
{
    switch (x) {
	case EQ: return "z";
	case NE: return "nz";
	default: abort();
    }
}

const char *fcpu_branch_lsb_code(enum rtx_code);
static const char *
fcpu_branch_lsb_code (x)
     enum rtx_code x;
{
    switch (x) {
	case EQ: return "nl";
	case NE: return "l";
	default: abort();
    }
}

void
print_operand (file, x, code)
     FILE *file;
     rtx x;
     const char code;
{
    /* print just address register from memory reference */
    if (code == 'r' && GET_CODE (x) == MEM && GET_CODE (XEXP (x, 0)) == REG)	
	fprintf (file, "%s", reg_names[REGNO (XEXP (x, 0))]);
    
    /* label number */
    else if (code == 'r' && GET_CODE (x) == CODE_LABEL)
	fprintf (file, "%d", XINT(x,5));

    /* register size in bits */
    else if (code == 's' && GET_CODE (x) == REG)
	fprintf (file, "%d", GET_MODE_SIZE(GET_MODE(x)) * BITS_PER_UNIT);
    
    /* prints "i" if x is constant, else nothing (with j, don't do it for 0) */
    else if (code == 'i' || code == 'j') {
	if (GET_CODE (x) == CONST_INT && (x != const0_rtx || code == 'i'))
	    fprintf (file,"i");
    }

    /* prints "r0" if x is const 0 */
    else if (code == 'o' && x == const0_rtx)
	fprintf (file, "r0");

    /* comparison (eq/ne -> z/nz) code of an operator x */
    else if (code == 'c')
	fprintf (file, "%s", fcpu_branch_zero_code(GET_CODE(x)));
    /* comparison (eq/ne -> nz/z) code of an operator x */
    else if (code == 'C')
	fprintf (file, "%s", fcpu_branch_zero_code(reverse_condition(GET_CODE(x))));
    /* comparison (eq/ne -> nl/l) code of an operator x */
    else if (code == 'e')
	fprintf (file, "%s", fcpu_branch_lsb_code(GET_CODE(x)));
    /* comparison (eq/ne -> nl/l) code of an operator x */
    else if (code == 'E')
	fprintf (file, "%s", fcpu_branch_lsb_code(reverse_condition(GET_CODE(x))));

    /* register name output */
    else if (GET_CODE (x) == REG)	
	fprintf (file, "%s", reg_names[REGNO (x)]);	

    else if (GET_CODE (x) == MEM) {
	rtx addr = XEXP (x, 0);
	/* output post dec/inc */
	if (GET_CODE(addr) == POST_MODIFY) {
	    int size;
	    rtx plus = XEXP (addr, 1);
	    
	    if (GET_CODE (plus) != PLUS) 
		abort();
	    if (code == 'k') {
		/* might also be a (temporary) register -MR */
		if (GET_CODE (XEXP (plus,1)) == CONST_INT)
		    fprintf (file, "i");
		return;
	    }
	    addr = XEXP (addr, 0); /* to print it later */
	    if (GET_CODE (XEXP (plus,1)) == CONST_INT) {
		/* const modify */
		size = INTVAL (XEXP (plus,1));
		if (size > 255 || size < -256) 
		    abort();
	    
		fprintf (file, "$%d,", size);
	    } else {
		/* register modify */
		if (!REG_P(XEXP (plus,1))) 
		    abort();
		fprintf (file, "%s,", reg_names[REGNO (XEXP (plus,1))]);
	    }	
	}
	/* see below */
	if (code != 'k')
	    output_address (addr);
    }

    /* output number of first nonzero LSB */
    else if (GET_CODE (x) == CONST_INT && code == 'b') {
	int b = ffz_big(~INTVAL(x)); 
	if (b < 0) 
	    abort();
	fprintf (file, "$%d", b); 
    }	
    /* output constant divided by 16; needed by loadcons */
    else if (GET_CODE (x) == CONST_INT && code == 'd') {
	fprintf (file, "%d", (int)INTVAL(x)/16);
    }
    else {
	/*if (!CONSTANT_P(x)) abort();*/
	putc ('$', file); 
	output_addr_const (file, x); 
    }
}

int 
fcpu_is_legitimate_address (mode,x)
	enum machine_mode mode;
	rtx x;
{
    /* accept post modify */
    if (GET_CODE (x) == POST_MODIFY && 
	    REG_P (XEXP (x,0)) && 
	    GET_CODE (XEXP (x,1)) == PLUS && 
	    ((GET_CODE (XEXP (XEXP (x,1),1)) == CONST_INT && 
	      INTVAL (XEXP (XEXP (x,1),1)) >= -256 && 
	      INTVAL (XEXP (XEXP (x,1),1)) <= 255) || 
	     REG_P (XEXP (XEXP (x,1),1)))) 
	return 1;
    
    /* single register address */
    if (GET_CODE (x) == REG) return 1;

    return 0;
    /* indexing (devik's test, not in FCPU ISA yet) */
    if (GET_CODE (x) == PLUS) {
	int off = 32 * GET_MODE_SIZE(mode);
	rtx idx = XEXP (x, 0), cons = XEXP (x, 1);
off=32;
	if (GET_CODE (idx) == CONST_INT) { /* handle swapped version */
	    rtx tmp = idx; idx = cons; cons = tmp;
	}
	/* handle reg + 6bit const */
	if (GET_CODE (idx) == REG && REG_OK_FOR_BASE_P (idx)
		&& GET_CODE (cons) == CONST_INT
	        && INTVAL (cons) >= -off
		&& INTVAL (cons) <= off) return 1;
    }

    return 0;
}

void
print_operand_address (file, addr)
     FILE *file;
     register rtx addr;
{
    if (GET_CODE (addr) == PLUS) {
	rtx idx = XEXP (addr, 0), cons = XEXP (addr, 1);
	if (GET_CODE (idx) == CONST_INT) { /* handle swapped version */
	    rtx tmp = idx; idx = cons; cons = tmp;
	}
	if (GET_CODE (idx) != REG || GET_CODE (cons) != CONST_INT)
	    abort();
	
	fprintf (file, "[%d]%s", (int)INTVAL (cons),reg_names[REGNO (idx)]);
	return;
    }
    
    /* only one indirection level supported */
    if (GET_CODE (addr) != REG)
	abort();

    /* [%s] would be probably nicer */
    fprintf (file, "%s", reg_names[REGNO (addr)]); 
}

#define GEN_IMMXOP(suff,lo,hi) \
int imm ## suff ## _operand (op, mode) \
	rtx op; enum machine_mode mode ATTRIBUTE_UNUSED; \
 { return (GET_CODE(op) == CONST_INT && INTVAL(op) >= lo && INTVAL(op) <= hi); } \
int regimm ## suff ## _operand (op, mode) \
	rtx op; enum machine_mode mode; \
 { return imm ## suff ## _operand (op,mode) || register_operand (op,mode); } \
int regmemimm ## suff ## _operand (op, mode) \
	rtx op; enum machine_mode mode; \
 { return imm ## suff ## _operand (op,mode) || nonimmediate_operand (op,mode); }

GEN_IMMXOP (8,-128,255)
GEN_IMMXOP (8s,-128,127)
GEN_IMMXOP (8u,0,255)
GEN_IMMXOP (9s,-256,255)
GEN_IMMXOP (9x,-255,255)
GEN_IMMXOP (16,-32768,65535)
GEN_IMMXOP (16s,-32768,32767)
GEN_IMMXOP (16u,0,65535)
GEN_IMMXOP (0,0,0)

#undef GEN_IMMXOP

int
ternary_operator (op, mode)
    rtx op;
    enum machine_mode mode;
{
    return ((mode == VOIDmode || GET_MODE (op) == mode)
	    && (GET_RTX_CLASS (GET_CODE (op)) == 'b' 
		    || GET_RTX_CLASS (GET_CODE (op)) == '3') );
}
int
binary_operator (op, mode)
    rtx op;
    enum machine_mode mode;
{
    return ((mode == VOIDmode || GET_MODE (op) == mode)
	    && GET_RTX_CLASS (GET_CODE (op)) == 'c');
}
int
unary_operator (op, mode)
    rtx op;
    enum machine_mode mode;
{
    return ((mode == VOIDmode || GET_MODE (op) == mode)
	    && GET_RTX_CLASS (GET_CODE (op)) == '1');
}


/* loadcons immediate not handled be fcpu_expand_move */
int bigimm_operand (op, mode)
	rtx op;
	enum machine_mode mode ATTRIBUTE_UNUSED;
{
	return (CONSTANT_P(op) && (INTVAL(op)<0 || INTVAL(op)>0xffff));
}

int fcpu_is_label(rtx op);
int fcpu_is_label(op)
	rtx op;
{
	switch (GET_CODE (op)) {
		case LABEL_REF:
		case SYMBOL_REF:
			return 1;
		default: break;
	}
	return 0;
}

/* register or label_refs and symbol_refs before reload */
int target_operand (op, mode)
	rtx op;
	enum machine_mode mode;
{
	if (mode != DImode)
		return 0;

	if ((GET_MODE (op) == DImode || GET_MODE (op) == VOIDmode)
			&& fcpu_is_label (op))
	      return ! reload_completed;

        return register_operand (op, mode);
}
/* constant which can be loaded by bitop (has one 1 or one 0 bit inside) 
  TODO: we disabled zero bit generator until MR fixes it */
int bitable_operand (op, mode)
	rtx op;
	enum machine_mode mode;
{
    	int log;
	if (GET_CODE(op) != CONST_INT) 
	    return 0;
    	log = exact_log2(INTVAL(op) & GET_MODE_MASK(mode));
//	if (log < 0) 
//	    log = exact_log2((~INTVAL(op)) & GET_MODE_MASK(mode));
	return log >= 0;
}

#define SW_R_OP(C,MIN,MAX) case C: return v >= MIN && v <= MAX
int 
fcpu_const_ok_for_letter_p(v,c)
	HOST_WIDE_INT v;
	char c;
{
	switch (c) {
		SW_R_OP('I',0,0xffff);
		SW_R_OP('J',-32768,0x7fff);
		SW_R_OP('K',0,63);
		SW_R_OP('L',0,0xff);
		SW_R_OP('N',1,1);
		SW_R_OP('O',0,0);
		case 'M': return exact_log2(v)>=0;
	}
	return 0;
}
#undef SW_R_OP

int
ffz_big (x)
	unsigned HOST_WIDE_INT x;
{
    	int log = 0;
    	while (x & 1) {
	    	x >>= 1;
		log++;
	}
	return (unsigned)log >= sizeof(x)*8 ? -1 : log;
}

void                   
fcpu_expand_move (mode, ops)
	enum machine_mode mode;
	rtx ops[];           
{
	int i;
#if 1
	/* can't copy directly to MEM pointed by invalid  */
	if (GET_CODE (ops[0]) == MEM && !memory_operand(ops[0],mode))
		ops[0] = gen_rtx_MEM(mode,force_reg(Pmode,XEXP(ops[0],0)));
	
	/* also can't copy from MEM pointed by NONREG */
	if (GET_CODE (ops[1]) == MEM && !memory_operand(ops[1],mode))
		ops[1] = gen_rtx_MEM(mode,force_reg(Pmode,XEXP(ops[1],0)));
#endif
	/* generate big nonzero nonbitable constants */
	if (GET_CODE (ops[1]) == CONST_INT && GET_CODE (ops[0]) == REG && 
		GET_MODE_SIZE(mode) > 2 && !bitable_operand(ops[1],mode) && 
		(INTVAL(ops[1]) > 32767 || INTVAL(ops[1]) < -32768)) {
		HOST_WIDE_INT x = INTVAL(ops[1]), y;

		/* prefer loadconsx as the first load because it prevents
		   gcc to emit spurious zero load */
		int v = x & 0xffff;
		y = v & 0x8000 ? -1 : 0;
		
		/* convert to range used by *mov templates */
		if (v > 0x7fff) v -= 65536;

		emit_insn(gen_rtx_SET(mode,ops[0], GEN_INT(v)));

		/* 2 or 4 cycles; y holds already constructed value */
		for (i = 1; i < GET_MODE_SIZE(mode)/2; i++) {
			int j,f,esign;

			y >>= 16, x >>= 16;
			esign = x & 0x8000 ? 0xffff : 0;

			/* is current block ok ? */
			if (!((x ^ y) & 0xffff)) continue;

			
			/* decide whether to use loadconsx; it is true if
			   current sign is the same as some 16bit block */
			f = 0;
			for (j = 1; j < GET_MODE_SIZE(mode)/2-i; j++)
				if (((x >> (j*16)) & 0xffff) == esign) {
					f = 1; break;
				}
			if (f) {
				/* use loadconsx, update y */
				y = esign ? -1 : 0;
				emit_insn(gen_rtx_SET(mode,
					gen_rtx_ZERO_EXTRACT(mode,ops[0],
						GEN_INT(GET_MODE_SIZE(mode)*8-16*i),
						GEN_INT(i*16)), GEN_INT(x & 0xffff)));
			} else {
				/* loadcons */
				emit_insn(gen_rtx_SET(mode,
					gen_rtx_ZERO_EXTRACT(mode,ops[0],GEN_INT(16),
						GEN_INT(i*16)), GEN_INT(x & 0xffff)));

			}
		}
		return;
	}
	
	/* if the first were MEM and second NONREG then use other register */
	if (GET_CODE (ops[0]) == MEM && GET_CODE (ops[1]) != REG)
		ops[1] = force_reg(mode,ops[1]);

	/* ok can generate direct move */
	emit_insn(gen_rtx_SET(VOIDmode,ops[0],ops[1]));
}

struct fcpu_compare fcpu_compare;

/* generates and returns code to compare fcpu_compare operands
   in relation "code"; comparison generates value ot the same
   mode as operands are which is nonzero resp. zero if the
   condition is met (reversed if set to 0 resp 1). 
   Note that we detect special cases (cmp with 0) here. */
static rtx
fcpu_gen_compare (code,reversed)
	enum rtx_code code;
	int *reversed;
{
	enum machine_mode opmode;
	rtx op0 = fcpu_compare.op0, op1 = fcpu_compare.op1, tem;
	HOST_WIDE_INT wi;
	
	opmode = GET_MODE(op0);
	if (GET_MODE_CLASS(opmode) != MODE_INT ||
		(opmode != GET_MODE(op1) && GET_MODE(op1) != VOIDmode)) 
	    abort(); /* don't support FP yet */
	
	/* if second is constant, offset it */
	wi = -(GET_MODE_MASK(opmode)>>1);
	if (GET_CODE(op1) == CONST_INT
		&& (INTVAL(op1) > 0 || (code != GEU && code != LTU))
	 	&& (INTVAL(op1) >= wi || (code != GE && code != LT)) ) {
	    int off = 1;
	    switch (code) {
		case GE:  code = GT; break;
		case LT:  code = LE; break;
		case GEU: code = GTU; break;
		case LTU: code = LEU; break;
		default:  off = 0; break;
	    }
	    if (off) op1 = GEN_INT(INTVAL(op1)-1);
	}
	/* we have no LT and GE in F-CPU, swap the operands if in regs */
	*reversed = 0;
	switch (code) {
	    case GE: case LT: case GEU: case LTU: 
		tem = op1; op1 = op0; op0 = tem;
		code = swap_condition(code);
		break;
	    case EQ:
		code = NE, *reversed = 1;
	    case LE:  case GT:  case LEU:  case GTU: case NE:
		break; /* ok */
	    default:  
		abort();
	}

	/* we need ops in registers; combiner will make immediates again */
	if (!regimm0_operand (op0, opmode))
		op0 = force_reg (opmode, op0);
	if (!regimm0_operand (op1, opmode))
		op1 = force_reg (opmode, op1);
	
	/* compare and branch against 0 directly. TODO: use 63th bit too */
	if (op1 == const0_rtx && code == NE)
	    	return op0;

	/* treat NE as XOR since now */
	return gen_rtx_fmt_ee (code == NE ? XOR : code, 
		    opmode, op0, op1);
}

/* like above but the value must comply with STORE_FLAG_VALUE;
   resmode is mode wanted for the result; we also want result in QImode */
void
fcpu_emit_scc (code,ops)
	enum rtx_code code;
	rtx ops[];
{
	enum machine_mode opmode;
	int reversed;
	rtx tem,tem2;

	/* generate the value in "native" mode */
	tem = fcpu_gen_compare(code,&reversed);
	opmode = GET_MODE(tem);

	/* not needed anymore */
	memset (&fcpu_compare, 0, sizeof (fcpu_compare));

	/* let's normalize the value - it is ok unless
	   XOR was used - then compare it with 0 */
	if (code == EQ || code == NE) {
	    	/* we need tem in register */
	    	if (!regimm0_operand (tem, opmode))
			tem = force_reg (opmode, tem);
	    	tem = gen_rtx_fmt_ee (reversed ? LEU : GTU,
			opmode, tem, CONST0_RTX (opmode));
	}
	if (GET_MODE(ops[0]) != QImode) abort();

	if (opmode != GET_MODE(ops[0])) {
	    emit_move_insn(tem2 = gen_reg_rtx(opmode),tem);
//	    tem = gen_rtx_TRUNCATE(QImode,tem2);
	    tem = gen_rtx_SUBREG(QImode,tem2,0);
	}
		
	emit_move_insn(ops[0],tem);
}

rtx 
fcpu_expand_cmove (ops, mode)
	rtx ops[];
	enum machine_mode mode ATTRIBUTE_UNUSED;
{
    	rtx scc;
	enum machine_mode opmode;
	int reversed;
	enum rtx_code cmp = GET_CODE(ops[1]);

	/* test for case with one branch only */
	if (REG_P(ops[2]) && REGNO(ops[0]) == REGNO(ops[2]))
	    	;
	else if (!REG_P(ops[3]) || REGNO(ops[0]) != REGNO(ops[3]))
	    	return NULL; /* can't handle full if-then-else */
	
	opmode = GET_MODE(fcpu_compare.op0);
	if (GET_MODE_CLASS(opmode) != MODE_INT ||
		(opmode != GET_MODE(fcpu_compare.op1) && 
		GET_MODE(fcpu_compare.op1) != VOIDmode)) 
	    abort(); /* don't support FP yet */

    	/* simple case: NE/EQ register comparison with 0 in any mode;
	   this doesn't need new pseudos */
    	if ((cmp == NE || cmp == EQ)
		&& fcpu_compare.op1 == const0_rtx
		&& GET_CODE(fcpu_compare.op0) == REG) {
	    	return gen_rtx(cmp,VOIDmode,
			fcpu_compare.op0,fcpu_compare.op1);
	}
	/* above was the only case if we can't create new imtermediate (it
	   would be needed to hold result of compare */
	if (no_new_pseudos) return NULL;

	/* store the raw condition value */
	scc = fcpu_gen_compare(cmp,&reversed);

	/* not needed anymore */
	memset (&fcpu_compare, 0, sizeof (fcpu_compare));

	/* we need scc in register */
	if (!regimm0_operand (scc, opmode))
		scc = force_reg (opmode, scc);

	/* other than EQ/NE can use LSB as test */
	if (cmp != EQ && cmp != NE)
		scc = gen_rtx_ZERO_EXTRACT (opmode, scc, const1_rtx, const0_rtx);
	else 
		/* we need tem in DI mode */
		scc = convert_to_mode (DImode, scc, 1);

	/* can directly generate cmove with it (no normalization needed) */
	return gen_rtx_fmt_ee (reversed ? EQ : NE,
		opmode, scc, CONST0_RTX (opmode));
}

rtx
fcpu_emit_conditional_branch (code)
	enum rtx_code code;
{
	enum machine_mode opmode;
	int reversed;
	rtx tem;

	/* generate the compare value */
	tem = fcpu_gen_compare(code,&reversed);
	opmode = GET_MODE(tem);

	/* not needed anymore */
	memset (&fcpu_compare, 0, sizeof (fcpu_compare));

	/* we need tem in register */
	if (!regimm0_operand (tem, opmode))
		tem = force_reg (opmode, tem);

	/* other than EQ/NE can use LSB as test */
	if (code != EQ && code != NE)
		tem = gen_rtx_ZERO_EXTRACT (opmode, tem, const1_rtx, const0_rtx);
	else 
		/* we need tem in DI mode */
		tem = convert_to_mode (DImode, tem, 1);

	/* Return the branch comparison.  */
	return gen_rtx_fmt_ee (reversed ? EQ : NE, 
		opmode, tem, CONST0_RTX (opmode));
}
