/*
 * strings.c - SVR4 style strings(1).
 * Copyright (C) 2001, 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <ctype.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: strings.c,v 1.3 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static int opt_a = 0;
static int opt_n = 4;
static int opt_t = '\0';

static const char *fmt = NULL;

static void
process_buffer(const unsigned char *buf, size_t len, size_t soff) {
	size_t offset;
	size_t i;

	i = 0;
	for (offset = 0; offset < len; offset++) {
		if (isprint(buf[offset])) {
			/* accumulate characters */
			i++;
		}
		else {
			/* print string if it is long enough */
			if (i >= opt_n) {
				if (fmt) {
					printf(fmt, (unsigned long)(offset - i + soff));
				}
				fwrite(buf + (offset - i), 1, i, stdout);
				putchar('\n');
			}
			i = 0;
		}
	}
	if (i >= opt_n) {
		if (fmt) {
			printf(fmt, (unsigned long)(offset - i + soff));
		}
		fwrite(buf + (offset - i), 1, i, stdout);
		putchar('\n');
	}
}

static int
process_elf(const char *fname, Elf *elf) {
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	int err;
	size_t len;
	void *img;

	if (opt_a || !gelf_getehdr(elf, &eh) || eh.e_shoff == 0) {
		elf_errno();
		if ((img = elf_rawfile(elf, &len))) {
			process_buffer(img, len, 0);
		}
		else if ((err = elf_errno())) {
			file_error(fname, "elf_rawfile: %s", elf_errmsg(err));
			return 1;
		}
		return 0;
	}
	elf_errno();
	scn = NULL;
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			file_error(fname, "gelf_getshdr: %s", elf_errmsg(-1));
			return 1;
		}
		if (sh.sh_type != SHT_PROGBITS) {
			continue;
		}
		if (!(sh.sh_flags & SHF_ALLOC)) {
			continue;
		}
		if (sh.sh_flags & SHF_EXECINSTR) {
			continue;
		}
		if (!(data = elf_rawdata(scn, NULL))) {
			file_error(fname, "elf_rawdata: %s", elf_errmsg(-1));
			return 1;
		}
		assert(!elf_rawdata(scn, data));
		process_buffer(data->d_buf, data->d_size, (size_t)sh.sh_offset);
	}
	if ((err = elf_errno())) {
		file_error(fname, "elf_nextscn: %s", elf_errmsg(err));
		return 1;
	}
	return 0;
}

static int
process_file(const char *fname) {
	int err = 0;
	Elf *elf;
	int fd;

	if ((fd = open(fname, O_RDONLY)) != -1) {
		if ((elf = elf_begin(fd, ELF_C_READ, NULL))) {
			err = process_elf(fname, elf);
			elf_end(elf);
		}
		else {
			file_error(fname, "elf_begin: %s", elf_errmsg(-1));
			err = 1;
		}
		close(fd);
	}
	else {
		file_error(fname, "open: %s", strerror(errno));
		err = 1;
	}
	return err;
}

static int
process_stream(FILE *fp) {
	unsigned char *buf;
	size_t offset;
	size_t i;
	int c;

	i = 0;
	offset = 0;
	buf = xmalloc(opt_n + 1);
	while ((c = getc(fp)) != EOF) {
		if (!isprint(c)) {
			if (i >= opt_n) {
				/* terminate string */
				putchar('\n');
			}
			i = 0;
		}
		else if (i < opt_n) {
			/* accumulate characters until the string is long enough */
			buf[i++] = c;
			if (i == opt_n) {
				/* string is long enough to print */
				if (fmt) {
					printf(fmt, (unsigned long)(offset - i));
				}
				fwrite(buf, 1, i, stdout);
			}
		}
		else {
			putchar((unsigned char)c);
			i++;
		}
		offset++;
	}
	if (i >= opt_n) {
		/* terminate last string */
		putchar('\n');
	}
	xfree(buf);
	return ferror(fp) != 0;
}

static const char usage[] =
	"usage: %s [-aoV] [-n number] [-t o|d|x] [file...]\n"
	"  -a        scan the whole file\n"
	"  -o        print string offsets (same as `-t d')\n"
	"  -n number ignore strings shorter than <number> characters\n"
	"  -t o|d|x  print string offsets in octal/decimal/hex\n"
	"  -V        display program version\n"
	;

int
main(int argc, char **argv) {
	char *progname = NULL;
	int opt_V = 0;
	int err;
	int c;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		*argv = ++progname;
	}
	else if (!(progname = *argv)) {
		progname = "strip";
	}
	setprogname(progname);

	while ((c = getopt(argc, argv, "an:ot:V")) != EOF) {
		switch (c) {
			case 'a':
				opt_a = 1;
				break;
			case 'n':
				opt_n = atoi(optarg);
				if (opt_n <= 0) {
					error("invalid number \"%s\"", optarg);
					exit(1);
				}
				break;
			case 'o':
				opt_t = 'd';
				break;
			case 't':
				c = *optarg;
				if ((c != 'o' && c != 'd' && c != 'x') || optarg[1]) {
					error("invalid argument \"%s\"", optarg);
					exit(1);
				}
				opt_t = c;
				break;
			case 'V':
				opt_V = 1;
				break;
			case '?':
				fprintf(stderr, usage, progname);
				exit(1);
		}
	}
	switch (opt_t) {
		case 'o': fmt = "%7lo "; break;
		case 'd': fmt = "%7lu "; break;
		case 'x': fmt = "%7lx "; break;
	}
	if (opt_V) {
		show_version("strings");
		if (optind == argc) {
			exit(0);
		}
	}
	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}
	err = 0;
	if (optind == argc) {
		err = process_stream(stdin);
	}
	else {
		while (optind < argc) {
			err |= process_file(argv[optind++]);
		}
	}
	exit(err);
}
