/*
 * size.c - SVR4 style size(1).
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: size.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static int fmt = 0;
static int opt_f = 0;
static int opt_F = 0;
static int opt_n = 0;

static int print_name = 0;

#if SIZEOF_LONG_LONG == 8 && SIZEOF_LONG != 8
# define I64	long long
# define U64	unsigned long long
# define IF64	"lld"
# define UF64	"llu"
# define OF64	"llo"
# define XF64	"llx"
#else
# define I64	long
# define U64	unsigned long
# define IF64	"ld"
# define UF64	"lu"
# define OF64	"lo"
# define XF64	"lx"
#endif

static const char *formats[3][3] = {
	{	/* size */
		"%" UF64 " + %" UF64 " + %" UF64 " = %" UF64 "\n",
		"%" OF64 " + %" OF64 " + %" OF64 " = %" OF64 "\n",
		"%" XF64 " + %" XF64 " + %" XF64 " = %" XF64 "\n",
	},
	{	/* size -[fF] */
		"%s%" UF64 "(%s)",
		"%s%" OF64 "(%s)",
		"%s%" XF64 "(%s)",
	},
	{	/* size -[fF], last part */
		" = %" UF64 "\n",
		" = %" OF64 "\n",
		" = %" XF64 "\n",
	},
};

static const char*
segment_flags(GElf_Word flags) {
	static char buf[4];

	buf[0] = flags & PF_R ? 'r' : '-';
	buf[1] = flags & PF_W ? 'w' : '-';
	buf[2] = flags & PF_X ? 'x' : '-';
	buf[3] = '\0';
	return buf;
}

static int
process_elf(const char *name, Elf *elf, int ofd) {
	GElf_Xword rfsize, wfsize, wmsize;
	GElf_Ehdr eh;
	GElf_Phdr ph;
	GElf_Shdr sh;
	GElf_Addr shnum;
	Elf_Scn *scn;
	char *sname, *fill = "";
	unsigned i;
	int err;

	assert(ofd == -1);
	if (!gelf_getehdr(elf, &eh)) {
		file_error(name, "file format not recognized");
		return 1;
	}
	if (!(shnum = et_get_shnum(elf)) && eh.e_shoff) {
		file_error(name, "no section header table at offset 0x%lx",
			(unsigned long)eh.e_shoff);
		return 1;
	}
	if (opt_F || (!opt_f && opt_n && eh.e_phnum)) {
		if (!eh.e_phnum) {
			file_error(name, "no segments");
			return 1;
		}
		if (print_name || elf_getarhdr(elf)) {
			printf("%s: ", name);
		}
		rfsize = 0;
		for (i = 0; i < eh.e_phnum; i++) {
			GElf_Xword sz;

			if (!gelf_getphdr(elf, i, &ph)) {
				file_error(name, "gelf_getphdr: %s", elf_errmsg(-1));
				return 1;
			}
			if (ph.p_type == PT_LOAD) {
				sz = ph.p_memsz;
			}
			else if (opt_n) {
				sz = ph.p_filesz;
			}
			else {
				continue;
			}
			printf(formats[1][fmt], fill, (U64)sz, segment_flags(ph.p_flags));
			rfsize += sz;
			fill = " + ";
		}
		printf(formats[2][fmt], (U64)rfsize);
		return 0;
	}
	if (opt_f || (opt_n && shnum)) {
		GElf_Addr shstrndx;

		if (!shnum) {
			file_error(name, "no sections");
			return 1;
		}
		shstrndx = et_get_shstrndx(elf);
		if (print_name || elf_getarhdr(elf)) {
			printf("%s: ", name);
		}
		rfsize = 0;
		scn = NULL;
		elf_errno();
		while ((scn = elf_nextscn(elf, scn))) {
			if (!gelf_getshdr(scn, &sh)) {
				file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
				return 1;
			}
			if ((sh.sh_flags & SHF_ALLOC) || opt_n) {
				if (!(sname = elf_strptr(elf, shstrndx, sh.sh_name))) {
					file_error(name, "section name: %s", elf_errmsg(-1));
					return 1;
				}
				printf(formats[1][fmt], fill, (U64)sh.sh_size, sname);
				rfsize += sh.sh_size;
				fill = " + ";
			}
		}
		if ((err = elf_errno())) {
			file_error(name, "elf_nextscn: %s", elf_errmsg(err));
			return 1;
		}
		printf(formats[2][fmt], (U64)rfsize);
		return 0;
	}
	if (eh.e_phnum) {
		if (print_name || elf_getarhdr(elf)) {
			printf("%s: ", name);
		}
		rfsize = wfsize = wmsize = 0;
		for (i = 0; i < eh.e_phnum; i++) {
			if (!gelf_getphdr(elf, i, &ph)) {
				file_error(name, "gelf_getphdr: %s", elf_errmsg(-1));
				return 1;
			}
			if (ph.p_type == PT_LOAD) {
				if (ph.p_flags & PF_W) {
					wmsize += ph.p_memsz;
					wfsize += ph.p_filesz;
				}
				else {
					rfsize += ph.p_filesz;
				}
			}
		}
		printf(formats[0][fmt], (U64)rfsize, (U64)wfsize,
			(U64)(wmsize - wfsize), (U64)(rfsize + wmsize));
		return 0;
	}
	if (shnum) {
		if (print_name || elf_getarhdr(elf)) {
			printf("%s: ", name);
		}
		rfsize = wfsize = wmsize = 0;
		scn = NULL;
		elf_errno();
		while ((scn = elf_nextscn(elf, scn))) {
			if (!gelf_getshdr(scn, &sh)) {
				file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
				return 1;
			}
			if (sh.sh_flags & SHF_ALLOC) {
				if (sh.sh_type == SHT_NOBITS) {
					wmsize += sh.sh_size;
				}
				else if (sh.sh_flags & SHF_WRITE) {
					wfsize += sh.sh_size;
				}
				else {
					rfsize += sh.sh_size;
				}
			}
		}
		if ((err = elf_errno())) {
			file_error(name, "elf_nextscn: %s", elf_errmsg(err));
			return 1;
		}
		printf(formats[0][fmt], (U64)rfsize, (U64)wfsize,
			(U64)wmsize, (U64)(rfsize + wfsize + wmsize));
		return 0;
	}
	file_error(name, "no size information");
	return 1;
}

static const char usage[] =
	"usage: %s [-fFnoVx] [file...]\n"
	"  -f    show allocatable sections\n"
	"  -F    show loadable segments\n"
	"  -n    show non-loadable segments\n"
	"  -o    print numbers in octal\n"
	"  -V    display program version\n"
	"  -x    print numbers in hexadecimal\n"
	;

int
main(int argc, char **argv) {
	char *progname = NULL;
	int opt_V = 0;
	int err;
	int c;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		*argv = ++progname;
	}
	else if (!(progname = *argv)) {
		progname = "strip";
	}
	setprogname(progname);

	while ((c = getopt(argc, argv, "fFnoVx")) != EOF) {
		switch (c) {
			case 'f': opt_f = 1; break;
			case 'F': opt_F = 1; break;
			case 'n': opt_n = 1; break;
			case 'V': opt_V = 1; break;
			case 'o': if (!fmt) fmt = 1; break;
			case 'x': if (!fmt) fmt = 2; break;
			case '?': fprintf(stderr, usage, progname); exit(1);
			default: break;
		}
	}
	if (opt_V) {
		show_version("size");
		if (optind == argc) {
			exit(0);
		}
	}
	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}
	if (optind == argc) {
		err = et_read_file("a.out", process_elf);
	}
	else {
		if (argc - optind > 1) {
			print_name = 1;
		}
		for (err = 0; optind < argc; optind++) {
			err |= et_read_file(argv[optind], process_elf);
		}
	}
	exit(err);
}
