/*
 * proc4.c - process ELF and AR files.
 * Copyright (C) 1995 - 2001 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <elftools.h>
#include <ar.h>
#include <mkstemp.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: proc4.c,v 1.1 2002/12/27 18:17:36 michael Exp $";
#endif /* lint */

static char*
member_name(const char *name, const char *ar_name) {
	char *n2;

	n2 = xmalloc(strlen(name) + strlen(ar_name) + 3);
	strcpy(n2, name);
	strcat(n2, "[");
	strcat(n2, ar_name);
	strcat(n2, "]");
	return n2;
}

static int
modify_member(ardesc_t *ap, Elf_Arhdr *ah, const char *name, Elf *elf, pfunc_t func) {
	char tmpl[] = "/tmp/etXXXXXX";
	char *n2;
	mode_t umsk;
	int err;
	int fd;

	assert(ap);
	assert(ah);
	assert(name);
	assert(elf);
	assert(func);
	umsk = umask(077);
	if ((fd = mkstemp(tmpl)) == -1) {
		file_error(tmpl, "mkstemp: %s", strerror(errno));
		err = 1;
	}
	else {
		n2 = member_name(name, ah->ar_name);
		err = func(n2, elf, fd);
		xfree(n2);
		if (!err) {
			if (!(elf = elf_begin(fd, ELF_C_READ, NULL))) {
				file_error(tmpl, "elf_begin: %s", elf_errmsg(-1));
				err = 1;
			}
			else if (ar_add_elf(ap, elf, *ah)) {
				err = 1;
			}
		}
		unlink(tmpl);
		close(fd);
	}
	umask(umsk);
	return err != 0;
}

static int
modify_single(const char *name, int fd, Elf *elf, pfunc_t func, int bak) {
	char *tmpl;
	char *s;
	int err;
	int ofd;
	mode_t umsk;
	struct stat st;

	assert(name);
	assert(elf);
	assert(fd != -1);
	assert(func);
	if (fstat(fd, &st)) {
		fatal("fstat: %s", strerror(errno));
	}
	tmpl = xmalloc(strlen(name) + 9);
	if (bak) {
		/*
		 * Backup the original file
		 */
		strcpy(tmpl, name);
		strcat(tmpl, ".bak");
		unlink(tmpl);
		if (link(name, tmpl)) {
			file_warn(name, "can not create backup: %s", strerror(errno));
		}
	}
	strcpy(tmpl, name);
	s = (s = strrchr(tmpl, '/')) ? s + 1 : tmpl;
	strcpy(s, "etXXXXXX");
	umsk = umask(077);	/* force mode 0600 for mkstemp */
	err = 0;
	if ((ofd = mkstemp(tmpl)) == -1) {
		file_error(tmpl, "mkstemp: %s", strerror(errno));
		err = 1;
	}
	else if (func(name, elf, ofd)) {
		unlink(tmpl);
		err = 1;
	}
	else if (rename(tmpl, name)) {
		file_error(tmpl, "rename: %s", strerror(errno));
		unlink(tmpl);
		err = 1;
	}
	else {
		if (fchown(ofd, st.st_uid, st.st_gid) == -1) {
			/* drop setuid/setgid/sticky attributes */
			st.st_mode &= 0777;
		}
		fchmod(ofd, st.st_mode & 07777);
	}
	umask(umsk);
	xfree(tmpl);
	close(ofd);
	return err;
}

int
et_modify_file(const char *name, pfunc_t func, int bak) {
	Elf *arf;
	Elf *elf;
	Elf_Cmd cmd;
	int err;
	int fd;

	if ((fd = open(name, O_RDWR)) == -1) {
		file_error(name, "open: %s", strerror(errno));
		return 1;
	}
	if (!(arf = elf_begin(fd, ELF_C_READ, NULL))) {
		file_error(name, "elf_begin: %s", elf_errmsg(-1));
		close(fd);
		return 1;
	}
	err = 0;
	if (elf_kind(arf) == ELF_K_AR) {
		ardesc_t *ap = ar_create();
		Elf_Arhdr *ah;

		elf_errno();
		cmd = ELF_C_READ;
		while ((elf = elf_begin(fd, cmd, arf))) {
			if (!(ah = elf_getarhdr(elf))) {
				file_error(name, "bad archive member");
				err = 1;
			}
			else if (ah->ar_name[0] == '/') {
				/* skip */
			}
			else {
				err |= modify_member(ap, ah, name, elf, func);
			}
			cmd = elf_next(elf);
			elf_end(elf);
		}
		if (!err) {
			if ((err = elf_errno())) {
				file_error(name, "elf_begin: %s", elf_errmsg(err));
			}
			else {
				err = ar_update_archive(ap, name, bak);
			}
		}
		ar_destroy(ap);
	}
	else {
		err = modify_single(name, fd, arf, func, bak);
	}
	elf_end(arf);
	close(fd);
	return err != 0;
}

int
et_read_file(const char *name, pfunc_t func) {
	Elf *arf;
	Elf *elf;
	Elf_Cmd cmd;
	int err;
	int fd;

	if ((fd = open(name, O_RDONLY)) == -1) {
		file_error(name, "open: %s", strerror(errno));
		return 1;
	}
	if (!(arf = elf_begin(fd, ELF_C_READ, NULL))) {
		file_error(name, "elf_begin: %s", elf_errmsg(-1));
		close(fd);
		return 1;
	}
	err = 0;
	if (elf_kind(arf) == ELF_K_AR) {
		Elf_Arhdr *ah;
		char *n2;

		elf_errno();
		cmd = ELF_C_READ;
		while ((elf = elf_begin(fd, cmd, arf))) {
			if (!(ah = elf_getarhdr(elf))) {
				file_error(name, "bad archive member");
				err = 1;
			}
			else if (ah->ar_name[0] == '/') {
				/* skip */
			}
			else {
				n2 = member_name(name, ah->ar_name);
				err |= func(n2, elf, -1);
				xfree(n2);
			}
			cmd = elf_next(elf);
			elf_end(elf);
		}
		if (!err) {
			if ((err = elf_errno())) {
				file_error(name, "elf_begin: %s", elf_errmsg(err));
			}
		}
	}
	else {
		err = func(name, arf, -1);
	}
	elf_end(arf);
	close(fd);
	return err != 0;
}
