/*
 * mcs.c - SVR4 style mcs(1), manipulates comment section of ELF files.
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <errno.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: mcs.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static char *scnname = ".comment";
static char *append_string = NULL;
static size_t append_length = 0;
static int do_compress = 0;
static int do_delete = 0;
static int do_print = 0;
static int do_backup = 0;

static const unsigned blocksize = 1024;

static void
add_string(const char *s) {
	size_t n, len = strlen(s) + 1;

	n = append_length + blocksize - 1, n -= n % blocksize;
	if (append_length + len > n) {
		n = append_length + len + blocksize - 1, n -= n % blocksize;
		append_string = xrealloc(append_string, n);
	}
	strcpy(append_string + append_length, s);
	append_length += len;
}

static void
print_comment(const char *name, const char *str, size_t len) {
	char c = '\0';

	printf("%s:\n", name);
	while (len--) {
		c = *str++;
		putchar(c != '\0' ? c : '\n');
	}
	if (c != '\0' && c != '\n') {
		putchar('\n');
	}
}

static size_t
compress_section(char *str, size_t n) {
	size_t copied;
	size_t len;
	size_t i, j;

	copied = i = j = 0;
	while (i < n) {
		if (j < copied) {
			/*
			 * Compare with copied string
			 */
			len = strlen(str + j) + 1;
			assert(j + len <= copied);
			if (i + len > n || memcmp(str + j, str + i, len) != 0) {
				/*
				 * Compare with next string (if any)
				 */
				j += len;
			}
			else {
				/*
				 * Skip duplicate
				 */
				i += len;
				/*
				 * Start over
				 */
				j = 0;
			}
		}
		else {
			assert(j == copied);
			if (copied < i) {
				/*
				 * Copy new string
				 */
				while (i < n && (str[copied++] = str[i++])) {
					/* do nothing */
				}
			}
			else {
				/*
				 * Source and destination are equal
				 */
				assert(copied == i);
				while (i < n && str[i++]) {
					/* do nothing */
				}
				copied = i;
			}
			/*
			 * Start over
			 */
			j = 0;
		}
	}
	/*
	 * Return new size
	 */
	return copied;
}

static int
mcs_modify(const char *name, Elf *elf, Elf *nelf, Elf_Scn *scn, int *found) {
	Elf_Data *data;
	Elf_Data *ndata;
	Elf_Scn *nscn;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	GElf_Addr shnum;
	GElf_Addr shstrndx;
	char *buf;
	const char *sname;
	int match;
	size_t idx;

	/*
	 * Get headers
	 */
	if (!gelf_getehdr(elf, &eh)) {
		file_error(name, "gelf_getehdr: %s", elf_errmsg(-1));
		return -1;
	}
	if (!(shnum = et_get_shnum(elf)) && eh.e_shoff) {
		file_error(name, "no section header table at offset 0x%lx",
			(unsigned long)eh.e_shoff);
		return -1;
	}
	if (!gelf_getshdr(scn, &sh)) {
		file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
		return -1;
	}
	shstrndx = et_get_shstrndx(elf);
	if (!(sname = elf_strptr(elf, shstrndx, sh.sh_name))) {
		file_error(name, "elf_strptr: %s", elf_errmsg(-1));
		return -1;
	}
	idx = elf_ndxscn(scn);
	if (idx <= 0 || idx >= shnum) {
		file_error(name, "elf_ndxscn: %s", elf_errmsg(-1));
		return -1;
	}
	/*
	 * Compare section names
	 */
	if (strcmp(sname, scnname) == 0) {
		if (found) {
			*found += 1;
		}
		match = 1;
		/*
		 * Check process header table
		 */
		if (eh.e_phnum) {
			GElf_Off fixed;

			if (!(fixed = et_layout_fixed(elf))) {
				file_error(name, "can't read process header table: %s", elf_errmsg(-1));
				return -1;
			}
			if (sh.sh_offset <= fixed) {
				if (sh.sh_type == SHT_NOBITS || sh.sh_offset < fixed) {
					file_error(name, "section %s inside segment", sname);
					return -1;
				}
			}
		}
		/*
		 * Check section attributes
		 */
		if (idx == shstrndx) {
			file_error(name, "modifying %s is futile", sname);
			return -1;
		}
		if (sh.sh_type == SHT_NOBITS) {
			file_error(name, "can't modify a NOBITS section");
			return -1;
		}
		if (sh.sh_flags & SHF_ALLOC) {
			file_error(name, "section %s is allocatable", sname);
			return -1;
		}
		/*
		 * Skip section if we're in delete mode
		 */
		if (do_delete && !append_string) {
			return SHN_UNDEF;
		}
	}
	else {
		match = 0;
	}
	/*
	 * Copy section
	 */
	if (!(nscn = elf_newscn(nelf))) {
		file_error(name, "elf_newscn: %s", elf_errmsg(-1));
		return -1;
	}
	if (!(data = elf_getdata(scn, NULL))) {
		file_error(name, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	assert(!elf_getdata(scn, data));
	if (!(ndata = elf_newdata(nscn))) {
		file_error(name, "elf_newdata: %s", elf_errmsg(-1));
		return -1;
	}
	*ndata = *data;
	if (match) {
		/*
		 * Process section data
		 */
		if (do_delete) {
			assert(append_string);
			ndata->d_buf = NULL;
			ndata->d_size = 0;
		}
		if (append_string) {
			buf = xmalloc(ndata->d_size + append_length);
			if (ndata->d_size) {
				memcpy(buf, ndata->d_buf, ndata->d_size);
			}
			memcpy(buf + ndata->d_size, append_string, append_length);
			ndata->d_buf = buf;
			ndata->d_size += append_length;
		}
		if (do_compress && ndata->d_size) {
			ndata->d_size = compress_section(ndata->d_buf, ndata->d_size);
		}
		if (do_print) {
			print_comment(name, ndata->d_buf, ndata->d_size);
		}
		sh.sh_size = ndata->d_size;
	}
	/*
	 * Copy the section header
	 */
	if (!gelf_update_shdr(nscn, &sh)) {
		file_error(name, "gelf_update_shdr: %s", elf_errmsg(-1));
		return -1;
	}
	return elf_ndxscn(nscn);
}

static int
mcs_print(const char *name, Elf *elf, Elf_Scn *scn, int *found) {
	Elf_Data *data;
	GElf_Shdr sh;
	GElf_Addr shstrndx;
	const char *sname;

	/*
	 * Get headers
	 */
	if (!gelf_getshdr(scn, &sh)) {
		file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
		return -1;
	}
	shstrndx = et_get_shstrndx(elf);
	if (!(sname = elf_strptr(elf, shstrndx, sh.sh_name))) {
		file_error(name, "elf_strptr: %s", elf_errmsg(-1));
		return -1;
	}
	/*
	 * Compare section names
	 */
	if (strcmp(sname, scnname) != 0) {
		return 0;
	}
	if (found) {
		*found += 1;
	}
	if (sh.sh_type == SHT_NOBITS) {
		return 0;
	}
	/*
	 * Process data
	 */
	if (!(data = elf_getdata(scn, NULL))) {
		file_error(name, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	assert(!elf_getdata(scn, data));
	print_comment(name, data->d_buf, data->d_size);
	return 0;
}

static int
_process_elf(const char *name, Elf *elf, Elf *nelf) {
	static int *xltab = NULL;
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Ehdr eh;
	GElf_Phdr ph;
	GElf_Shdr sh;
	GElf_Addr shnum;
	GElf_Addr shstrndx;
	int found;
	int x;
	size_t i;

	if (!gelf_getehdr(elf, &eh)) {
		file_error(name, "gelf_getehdr: %s", elf_errmsg(-1));
		return -1;
	}
	shstrndx = et_get_shstrndx(elf);
	if (!(shnum = et_get_shnum(elf)) && eh.e_shoff) {
		file_error(name, "no section header table at offset 0x%lx",
			(unsigned long)eh.e_shoff);
		return -1;
	}
	if (nelf) {
		/*
		 * Copy headers
		 */
		if (!gelf_newehdr(nelf, gelf_getclass(elf))) {
			file_error(name, "gelf_newehdr: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_update_ehdr(nelf, &eh)) {
			file_error(name, "gelf_update_ehdr: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_newphdr(nelf, eh.e_phnum) && eh.e_phnum) {
			file_error(name, "gelf_newphdr: %s", elf_errmsg(-1));
			return -1;
		}
		for (i = 0; i < eh.e_phnum; i++) {
			if (!gelf_getphdr(elf, i, &ph)) {
				file_error(name, "gelf_getphdr: %s", elf_errmsg(-1));
				return -1;
			}
			if (!gelf_update_phdr(nelf, i, &ph)) {
				file_error(name, "gelf_update_phdr: %s", elf_errmsg(-1));
				return -1;
			}
		}
		/*
		 * Create translation table
		 */
		xltab = xrealloc(xltab, shnum * sizeof(*xltab));
		xltab[0] = 0;
	}
	/*
	 * Process sections
	 */
	found = 0;
	scn = NULL;
	for (i = 1; i < shnum; i++) {
		if (!(scn = elf_getscn(elf, i))) {
			file_error(name, "elf_getscn: %s", elf_errmsg(-1));
			return -1;
		}
		if (nelf) {
			if ((x = mcs_modify(name, elf, nelf, scn, &found)) == -1) {
				return -1;
			}
			xltab[i] = x;
		}
		else {
			if ((x = mcs_print(name, elf, scn, &found)) == -1) {
				return -1;
			}
		}
	}
	if (found == 0) {
		if (append_string) {
			assert(nelf);
			if (!(scn = et_append_section(nelf, scnname, SHT_PROGBITS, 0))) {
				file_error(name, "can't append section");
				return -1;
			}
			if (!gelf_getshdr(scn, &sh)) {
				file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
				return -1;
			}
			assert(sh.sh_offset);
			if (!(data = elf_newdata(scn))) {
				file_error(name, "elf_newdata: %s", elf_errmsg(-1));
				return -1;
			}
			data->d_buf = xmalloc(append_length);
			memcpy(data->d_buf, append_string, append_length);
			data->d_size = append_length;
			data->d_off = 0;
			data->d_align = 1;
			if (do_compress) {
				data->d_size = compress_section(data->d_buf, data->d_size);
			}
			sh.sh_size = data->d_size;
			if (!gelf_update_shdr(scn, &sh)) {
				file_error(name, "gelf_update_shdr: %s", elf_errmsg(-1));
				return -1;
			}
		}
		else if (!elf_getarhdr(elf)) {
			file_error(name, "section not found");
			return -1;
		}
	}
	if (nelf) {
		if (et_scnxlate(nelf, xltab, shnum)) {
			file_error(name, "can't change section references: %s", elf_errmsg(-1));
			return -1;
		}
		shstrndx = xltab[shstrndx];
		assert(shstrndx);
		if (et_set_shstrndx(nelf, shstrndx)) {
			file_error(name, "et_set_shstrndx: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_getehdr(nelf, &eh)) {
			file_error(name, "gelf_getehdr: %s", elf_errmsg(-1));
			return -1;
		}
		if (eh.e_phnum) {
			/* loadable files need special treatment */
			if (et_layout(nelf)) {
				file_error(name, "can't change file layout");
				return -1;
			}
			elf_flagelf(nelf, ELF_C_SET, ELF_F_LAYOUT);
		}
		if (elf_update(nelf, ELF_C_WRITE) == (off_t)-1) {
			file_error(name, "write failed: %s", elf_errmsg(-1));
			return -1;
		}
	}
	return 0;
}

static int
process_elf(const char *name, Elf *elf, int ofd) {
	Elf *nelf = NULL;
	char *img;
	int err;
	size_t len;

	if (elf_kind(elf) == ELF_K_ELF) {
		/*
		 * Process ELF file
		 */
		if (ofd == -1) {
			err = _process_elf(name, elf, NULL);
		}
		else if (!(nelf = elf_begin(ofd, ELF_C_WRITE, NULL))) {
			file_error(name, "elf_begin: %s", elf_errmsg(-1));
			err = 1;
		}
		else {
			err = _process_elf(name, elf, nelf);
			elf_end(nelf);
		}
		return err;
	}
	if (!elf_getarhdr(elf)) {
		file_error(name, "not an ELF file");
		return 1;
	}
	if (ofd != -1) {
		/*
		 * Copy file verbatim
		 */
		elf_errno();
		if (!(img = elf_rawfile(elf, &len))) {
			if ((err = elf_errno())) {
				file_error(name, "elf_rawfile: %s", elf_errmsg(err));
				return 1;
			}
		}
		if (len && xwrite(ofd, img, len)) {
			return 1;
		}
	}
	return 0;
}

static int
do_mcs(const char *name) {
	if (do_delete && !append_string) {
		/* delete section */
		return et_modify_file(name, process_elf, do_backup);
	}
	if (append_string || do_compress) {
		/* modify section */
		return et_modify_file(name, process_elf, do_backup);
	}
	/* file remains unmodified */
	return et_read_file(name, process_elf);
}

static const char usage[] =
	"usage: %s [-BcdpV] [-a string] [-n name] file...\n"
	"  -a string     append string to section\n"
	"  -B            backup files before modifying them\n"
	"  -c            compress section\n"
	"  -d            delete section\n"
	"  -n name       use section name instead of `.comment'\n"
	"  -p            print section (default)\n"
	"  -V            display program version\n"
	;

int
main(int argc, char **argv) {
	char *progname;
	int do_version = 0;
	int c;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		*argv = ++progname;
	}
	else if (!(progname = *argv)) {
		progname = "mcs";
	}
	setprogname(progname);

	while ((c = getopt(argc, argv, "a:Bcdn:pV")) != EOF) {
		switch (c) {
			case 'a': add_string(optarg); break;
			case 'B': do_backup = 1; break;
			case 'c': do_compress = 1; break;
			case 'd': do_delete = 1; break;
			case 'n': scnname = optarg; break;
			case 'p': do_print = 1; break;
			case 'V': do_version = 1; break;
			case '?': fprintf(stderr, usage, progname); exit(1);
			default: break;
		}
	}

	if (do_version) {
		show_version("mcs");
		if (optind == argc) {
			exit(0);
		}
	}
	if (do_compress + do_delete + do_print == 0 && !append_string) {
		do_print++;
	}

	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}

	c = 0;
	while (optind < argc) {
		c |= do_mcs(argv[optind++]);
	}

	exit(c ? 1 : 0);
}
