/*
 * layout.c - calculate ELF file layout.
 * Copyright (C) 1995 - 2001 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: layout.c,v 1.1 2002/12/27 18:17:36 michael Exp $";
#endif /* lint */

GElf_Off
et_layout_fixed(Elf *elf) {
	GElf_Ehdr eh;
	GElf_Off fixed;
	GElf_Off off;
	GElf_Phdr ph;
	GElf_Word i;

	/*
	 * Calculate length of `fixed' part.
	 * This includes the Ehdr, Phdrs, and all segments.
	 */
	if (!gelf_getehdr(elf, &eh)) {
		return 0;
	}
	fixed = gelf_fsize(elf, ELF_T_EHDR, 1, eh.e_version);
	assert(fixed);
	if (eh.e_phnum) {
		off = eh.e_phoff + eh.e_phnum * eh.e_phentsize;
		if (fixed < off) {
			fixed = off;
		}
	}
	for (i = 0; i < eh.e_phnum; i++) {
		if (!gelf_getphdr(elf, i, &ph)) {
			return 0;
		}
		off = ph.p_offset + ph.p_filesz;
		if (fixed < off) {
			fixed = off;
		}
	}
	return fixed;
}

int
et_layout(Elf *elf) {
	Elf_Scn *scn;
	GElf_Ehdr eh;
	GElf_Off align;
	GElf_Off fixed;
	GElf_Off off;
	GElf_Shdr sh;

	/*
	 * Calculate length of `fixed' part.
	 * This includes the Ehdr, Phdrs, and all segments.
	 */
	if (!gelf_getehdr(elf, &eh)) {
		return -1;
	}
	if (!(fixed = et_layout_fixed(elf))) {
		return -1;
	}
	/*
	 * Grow fixed part if any sections cross its end.
	 * This should not happen, but...
	 */
	do {
		off = fixed;
		scn = NULL;
		elf_errno();
		while ((scn = elf_nextscn(elf, scn))) {
			if (!gelf_getshdr(scn, &sh)) {
				return -1;
			}
			if (sh.sh_type != SHT_NOBITS) {
				if (sh.sh_offset && sh.sh_offset < fixed) {
					if (fixed < sh.sh_offset + sh.sh_size) {
						fixed = sh.sh_offset + sh.sh_size;
					}
				}
			}
		}
		if (elf_errmsg(0)) {
			return -1;
		}
	}
	while (off != fixed);
	/*
	 * Reassign offsets for movable sections.
	 */
	scn = NULL;
	elf_errno();
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			return -1;
		}
		if (sh.sh_offset && sh.sh_offset < fixed) {
			continue;
		}
		if (sh.sh_addralign > 1) {
			off += sh.sh_addralign - 1;
			off -= off % sh.sh_addralign;
		}
		sh.sh_offset = off;
		if (sh.sh_type != SHT_NOBITS) {
			off += sh.sh_size;
		}
		if (!gelf_update_shdr(scn, &sh)) {
			return -1;
		}
	}
	if (elf_errmsg(0)) {
		return -1;
	}
	/*
	 * put section header table at the end
	 */
	align = gelf_fsize(elf, ELF_T_ADDR, 1, eh.e_version);
	assert(align);
	off += align - 1;
	off -= off % align;
	eh.e_shoff = off;
	/*
	 * update modified elf header
	 */
	if (!gelf_update_ehdr(elf, &eh)) {
		return -1;
	}
	return 0;
}
