/*
 * elfdump.c - display (some of the) contents of an ELF file.
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <common.h>
#include <elftools.h>
#include <ar.h>
#include <stdio.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: elfdump.c,v 1.6 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

#if SIZEOF_LONG_LONG == 8 && SIZEOF_LONG != 8
# define I64	long long
# define U64	unsigned long long
# define IF64	"lld"
# define UF64	"llu"
# define OF64	"llo"
# define XF64	"llx"
#else
# define I64	long
# define U64	unsigned long
# define IF64	"ld"
# define UF64	"lu"
# define OF64	"lo"
# define XF64	"lx"
#endif

#if HAVE_SYMBOL_VERSIONING

#if !defined(SHT_SUNW_versym) && defined(SHT_GNU_versym)
#define SHT_SUNW_versym SHT_GNU_versym
#endif
#if !defined(SHT_SUNW_verdef) && defined(SHT_GNU_verdef)
#define SHT_SUNW_verdef SHT_GNU_verdef
#endif
#if !defined(SHT_SUNW_verneed) && defined(SHT_GNU_verneed)
#define SHT_SUNW_verneed SHT_GNU_verneed
#endif

#define GElf_Verdef		Elf64_Verdef
#define GElf_Verdaux	Elf64_Verdaux
#define GElf_Verneed	Elf64_Verneed
#define GElf_Vernaux	Elf64_Vernaux

#define GElf_Versym		GElf_Half

#endif /* HAVE_SYMBOL_VERSIONING */

static const char *progname = "elfdump";

static int opt_a = 0;	/* dump everything */
static int opt_c = 0;
static int opt_d = 0;
static int opt_e = 0;
static int opt_i = 0;
static int opt_h = 0;
static int opt_k = 0;
static int opt_m = 0;
static int opt_n = 0;
static int opt_p = 0;
static int opt_r = 0;
static int opt_s = 0;
static int opt_S = 0;
static int opt_v = 0;
static int opt_y = 0;
static int opt_G = 0;
static int opt_V = 0;
static const char *opt_w = NULL;
static const char *opt_N = NULL;

#define const2str(x)		case x: return #x
#define pconst2str(pfx, x)	case pfx##x: return #x

#define const2arr(x)		[x] = #x
#define pconst2arr(pfx, x)	[pfx##x] = #x

#define number(A)			(sizeof(A)/sizeof(*(A)))

static const char*
dump_classname(unsigned x) {
	static const char *arr[] = {
		const2arr(ELFCLASSNONE),
		const2arr(ELFCLASS32),
		const2arr(ELFCLASS64),
	};
	static char buf[32];

	if (x < number(arr) && arr[x]) return arr[x];
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_dataname(unsigned x) {
	static const char *arr[] = {
		const2arr(ELFDATANONE),
		const2arr(ELFDATA2LSB),
		const2arr(ELFDATA2MSB),
	};
	static char buf[32];

	if (x < number(arr) && arr[x]) return arr[x];
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_dyntag(GElf_Sxword x) {
	static char buf[32];

	switch (x) {
		pconst2str(DT_, NULL);
		pconst2str(DT_, NEEDED);
		pconst2str(DT_, PLTRELSZ);
		pconst2str(DT_, PLTGOT);
		pconst2str(DT_, HASH);
		pconst2str(DT_, STRTAB);
		pconst2str(DT_, SYMTAB);
		pconst2str(DT_, RELA);
		pconst2str(DT_, RELASZ);
		pconst2str(DT_, RELAENT);
		pconst2str(DT_, STRSZ);
		pconst2str(DT_, SYMENT);
		pconst2str(DT_, INIT);
		pconst2str(DT_, FINI);
		pconst2str(DT_, SONAME);
		pconst2str(DT_, RPATH);
		pconst2str(DT_, SYMBOLIC);
		pconst2str(DT_, REL);
		pconst2str(DT_, RELSZ);
		pconst2str(DT_, RELENT);
		pconst2str(DT_, PLTREL);
		pconst2str(DT_, DEBUG);
		pconst2str(DT_, TEXTREL);
		pconst2str(DT_, JMPREL);
#ifdef DT_BIND_NOW
		pconst2str(DT_, BIND_NOW);
#endif /* DT_BIND_NOW */
#if defined(DT_INIT_ARRAY) && defined(DT_INIT_ARRAYSZ) && \
	defined(DT_FINI_ARRAY) && defined(DT_FINI_ARRAYSZ)
		pconst2str(DT_, INIT_ARRAY);
		pconst2str(DT_, FINI_ARRAY);
		pconst2str(DT_, INIT_ARRAYSZ);
		pconst2str(DT_, FINI_ARRAYSZ);
#endif
#ifdef DT_MOVEENT
		pconst2str(DT_, MOVEENT);
#endif /* DT_MOVEENT */
#ifdef DT_MOVESZ
		pconst2str(DT_, MOVESZ);
#endif /* DT_MOVESZ */
#ifdef DT_FEATURE_1
		pconst2str(DT_, FEATURE_1);
#endif /* DT_FEATURE_1 */
#ifdef DT_POSFLAG_1
		pconst2str(DT_, POSFLAG_1);
#endif /* DT_POSFLAG_1 */
#ifdef DT_SYMINSZ
		pconst2str(DT_, SYMINSZ);
#endif /* DT_SYMINSZ */
#ifdef DT_SYMINENT
		pconst2str(DT_, SYMINENT);
#endif /* DT_SYMINENT */
#ifdef DT_MOVETAB
		pconst2str(DT_, MOVETAB);
#endif /* DT_MOVETAB */
#ifdef DT_SYMINFO
		pconst2str(DT_, SYMINFO);
#endif /* DT_SYMINFO */
#ifdef DT_RELACOUNT
		pconst2str(DT_, RELACOUNT);
#endif /* DT_RELACOUNT */
#ifdef DT_RELCOUNT
		pconst2str(DT_, RELCOUNT);
#endif /* DT_RELCOUNT */
#ifdef DT_FLAGS_1
		pconst2str(DT_, FLAGS_1);
#endif /* DT_FLAGS_1 */
#ifdef DT_VERDEF
		pconst2str(DT_, VERDEF);
#endif /* DT_VERDEF */
#ifdef DT_VERDEFNUM
		pconst2str(DT_, VERDEFNUM);
#endif /* DT_VERDEFNUM */
#ifdef DT_VERNEED
		pconst2str(DT_, VERNEED);
#endif /* DT_VERNEED */
#ifdef DT_VERNEEDNUM
		pconst2str(DT_, VERNEEDNUM);
#endif /* DT_VERNEEDNUM */
#ifdef DT_SPARC_REGISTER
		pconst2str(DT_, SPARC_REGISTER);
#endif /* DT_SPARC_REGISTER */
#ifdef DT_AUXILIARY
		pconst2str(DT_, AUXILIARY);
#endif /* DT_AUXILIARY */
#ifdef DT_USED
		pconst2str(DT_, USED);
#endif /* DT_USED */
#ifdef DT_FILTER
		pconst2str(DT_, FILTER);
#endif /* DT_FILTER */
#ifdef DT_VERSYM
		pconst2str(DT_, VERSYM);
#endif /* DT_VERSYM */
	}
	sprintf(buf, "%#8" XF64, (U64)x);
	return buf;
}

static const char*
dump_machname(unsigned x) {
	static char buf[32];

	switch (x) {
#ifdef EM_M32
		const2str(EM_M32);
#endif /* EM_M32 */
#ifdef EM_SPARC
		const2str(EM_SPARC);
#endif /* EM_SPARC */
#ifdef EM_386
		const2str(EM_386);
#endif /* EM_386 */
#ifdef EM_68K
		const2str(EM_68K);
#endif /* EM_68K */
#ifdef EM_88K
		const2str(EM_88K);
#endif /* EM_88K */
#ifdef EM_486
		const2str(EM_486);
#endif /* EM_486 */
#ifdef EM_860
		const2str(EM_860);
#endif /* EM_860 */
#ifdef EM_MIPS
		const2str(EM_MIPS);
#endif /* EM_MIPS */
#ifdef EM_MIPS_RS3_LE
		const2str(EM_MIPS_RS3_LE);
#endif /* EM_MIPS_RS3_LE */
#ifdef EM_PARISC
		const2str(EM_PARISC);
#endif /* EM_PARISC */
#ifdef EM_VPP500
		const2str(EM_VPP500);
#endif /* EM_VPP500 */
#ifdef EM_SPARC32PLUS
		const2str(EM_SPARC32PLUS);
#endif /* EM_SPARC32PLUS */
#ifdef EM_960
		const2str(EM_960);
#endif /* EM_960 */
#ifdef EM_PPC
		const2str(EM_PPC);
#endif /* EM_PPC */
#ifdef EM_V800
		const2str(EM_V800);
#endif /* EM_V800 */
#ifdef EM_FR20
		const2str(EM_FR20);
#endif /* EM_FR20 */
#ifdef EM_RH32
		const2str(EM_RH32);
#endif /* EM_RH32 */
#ifdef EM_RCE
		const2str(EM_RCE);
#endif /* EM_RCE */
#ifdef EM_ARM
		const2str(EM_ARM);
#endif /* EM_ARM */
#ifdef EM_ALPHA
		const2str(EM_ALPHA);
#endif /* EM_ALPHA */
#ifdef EM_SH
		const2str(EM_SH);
#endif /* EM_SH */
#ifdef EM_SPARCV9
		const2str(EM_SPARCV9);
#endif /* EM_SPARCV9 */
#ifdef EM_TRICORE
		const2str(EM_TRICORE);
#endif /* EM_TRICORE */
#ifdef EM_ARC
		const2str(EM_ARC);
#endif /* EM_ARC */
#ifdef EM_H8_300
		const2str(EM_H8_300);
#endif /* EM_H8_300 */
#ifdef EM_H8_300H
		const2str(EM_H8_300H);
#endif /* EM_H8_300H */
#ifdef EM_H8S
		const2str(EM_H8S);
#endif /* EM_H8S */
#ifdef EM_H8_500
		const2str(EM_H8_500);
#endif /* EM_H8_500 */
#ifdef EM_IA_64
		const2str(EM_IA_64);
#endif /* EM_IA_64 */
#ifdef EM_MIPS_X
		const2str(EM_MIPS_X);
#endif /* EM_MIPS_X */
#ifdef EM_COLDFIRE
		const2str(EM_COLDFIRE);
#endif /* EM_COLDFIRE */
#ifdef EM_64HC12
		const2str(EM_64HC12);
#endif /* EM_64HC12 */
	}
	sprintf(buf, "%u", x);
	return buf;
}

static const char*
dump_pflags(unsigned x) {
	static char buf[80];
	char *p = buf;

	*p = '\0';
	if (x & PF_X) {
		p = stpcpy(p, " PF_X ");
	}
	if (x & PF_W) {
		p = stpcpy(p, " PF_W ");
	}
	if (x & PF_R) {
		p = stpcpy(p, " PF_R ");
	}
	if (x &= ~(PF_X | PF_W | PF_R)) {
		sprintf(p, " %#x ", x);
	}
	return buf;
}

static const char*
dump_ptype(unsigned x) {
	static const char *arr[] = {
		const2arr(PT_NULL),
		const2arr(PT_LOAD),
		const2arr(PT_DYNAMIC),
		const2arr(PT_INTERP),
		const2arr(PT_NOTE),
		const2arr(PT_SHLIB),
		const2arr(PT_PHDR),
	};
	static char buf[32];

	if (x < number(arr) && arr[x]) return arr[x];
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_reltype(Elf *elf, unsigned x) {
	static char buf[32];

	/* XXX: add processor-specific stuff */
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_scnname(Elf *elf, size_t scnno) {
	static const char estr[] = "*ERROR*";
	GElf_Shdr sh;
	GElf_Addr shstrndx;
	char *s;

	switch (scnno) {
		pconst2str(SHN_, UNDEF);
		pconst2str(SHN_, ABS);
		pconst2str(SHN_, COMMON);
	}
	if (!(shstrndx = et_get_shstrndx(elf))) {
		return estr;
	}
	if (!gelf_getshdr(elf_getscn(elf, scnno), &sh)) {
		return estr;
	}
	if (!(s = elf_strptr(elf, shstrndx, sh.sh_name))) {
		return estr;
	}
	return s;
}

static const char*
dump_shflags(GElf_Xword x) {
	static char buf[80];
	char *p;

	if (!x || (x & ~(SHF_WRITE | SHF_ALLOC | SHF_EXECINSTR))) {
		sprintf(buf, "%#" XF64, (U64)x);
	}
	else {
		p = stpcpy(buf, "[");
		if (x & SHF_WRITE) {
			p = stpcpy(p, " SHF_WRITE ");
		}
		if (x & SHF_ALLOC) {
			p = stpcpy(p, " SHF_ALLOC ");
		}
		if (x & SHF_EXECINSTR) {
			p = stpcpy(p, " SHF_EXECINSTR ");
		}
		strcpy(p, "]");
	}
	return buf;
}

static const char*
dump_shtype(unsigned x) {
	static char buf[32];

	switch (x) {
		const2str(SHT_NULL);
		const2str(SHT_PROGBITS);
		const2str(SHT_SYMTAB);
		const2str(SHT_STRTAB);
		const2str(SHT_RELA);
		const2str(SHT_HASH);
		const2str(SHT_DYNAMIC);
		const2str(SHT_NOTE);
		const2str(SHT_NOBITS);
		const2str(SHT_REL);
		const2str(SHT_SHLIB);
		const2str(SHT_DYNSYM);
#ifdef SHT_SUNW_move
		const2str(SHT_SUNW_move);
#endif /* SHT_SUNW_move */
#ifdef SHT_SUNW_COMDAT
		const2str(SHT_SUNW_COMDAT);
#endif /* SHT_SUNW_COMDAT */
#ifdef SHT_SUNW_syminfo
		const2str(SHT_SUNW_syminfo);
#endif /* SHT_SUNW_syminfo */
#ifdef SHT_SUNW_verdef
		const2str(SHT_SUNW_verdef);
#endif /* SHT_SUNW_verdef */
#ifdef SHT_SUNW_verneed
		const2str(SHT_SUNW_verneed);
#endif /* SHT_SUNW_verneed */
#ifdef SHT_SUNW_versym
		const2str(SHT_SUNW_versym);
#endif /* SHT_SUNW_versym */
	}
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_symtype(unsigned x) {
	static char buf[32];

	x = GELF_ST_TYPE(x);
	switch (x) {
		case STT_NOTYPE:  return "NOTY";
		case STT_OBJECT:  return "OBJT";
		case STT_FUNC:    return "FUNC";
		case STT_SECTION: return "SECT";
		case STT_FILE:    return "FILE";
	}
	sprintf(buf, "%u", x);
	return buf;
}

static const char*
dump_symbind(unsigned x) {
	static char buf[32];

	x = GELF_ST_BIND(x);
	switch (x) {
		case STB_LOCAL:  return "LOCL";
		case STB_GLOBAL: return "GLOB";
		case STB_WEAK:   return "WEAK";
	}
	sprintf(buf, "%u", x);
	return buf;
}

static const char*
dump_symname(Elf *elf, size_t scnno, GElf_Word symno, int scnsyms) {
	static char *buf = NULL;
	Elf_Scn *scn;
	GElf_Shdr sh;
	GElf_Sym sym;
	const char *s;

	if (!gelf_getshdr(scn = elf_getscn(elf, scnno), &sh)
	 || !gelf_getsym(elf_getdata(scn, NULL), symno, &sym)) {
		return "*ERROR*";
	}
	if (scnsyms && GELF_ST_TYPE(sym.st_info) == STT_SECTION) {
		s = dump_scnname(elf, sym.st_shndx);
		assert(s);
		if (*s != '*') {
			buf = xrealloc(buf, strlen(s) + 11);
			sprintf(buf, "%s (section)", s);
			s = buf;
		}
	}
	else if (!(s = elf_strptr(elf, sh.sh_link, sym.st_name))) {
		return "*ERROR*";
	}
	return s;
}

static const char*
dump_typename(unsigned x) {
	static const char *arr[] = {
		const2arr(ET_NONE),
		const2arr(ET_REL),
		const2arr(ET_EXEC),
		const2arr(ET_DYN),
		const2arr(ET_CORE),
	};
	static char buf[32];

	if (x < number(arr) && arr[x]) return arr[x];
	sprintf(buf, "%#x", x);
	return buf;
}

static const char*
dump_versname(unsigned x) {
	static char buf[32];

	switch (x) {
		const2str(EV_NONE);
		const2str(EV_CURRENT);
	}
	sprintf(buf, "%u", x);
	return buf;
}

static const char*
strptr(Elf_Data *data, size_t index) {
	if (data && data->d_buf && index >= 0 && index < data->d_size) {
		return (char*)data->d_buf + index;
	}
	return "*ERROR*";
}

void
dump_dyn(FILE *fp, GElf_Dyn *dyn, size_t index, Elf_Data *str) {
	static const char string_format[]  = "%10s  %-12s  %#-16" XF64 "  %s\n";
	static const char value_format[]   = "%10s  %-12s  %#" XF64 "\n";
	static const char pointer_format[] = "%10s  %-12s  %#" XF64 "\n";
	static const char unknown_format[] = "%10s  %#-12" XF64 "  %#" XF64 "\n";
	char buf[32];

	sprintf(buf, "[%lu]", (unsigned long)index);
	switch (dyn->d_tag) {
		case DT_NEEDED:
		case DT_SONAME:
		case DT_RPATH:
#ifdef DT_AUXILIARY
		case DT_AUXILIARY:
#endif /* DT_AUXILIARY */
#ifdef DT_FILTER
		case DT_FILTER:
#endif /* DT_FILTER */
			fprintf(fp, string_format, buf, dump_dyntag(dyn->d_tag),
				(U64)dyn->d_un.d_val, strptr(str, dyn->d_un.d_val));
			break;
		case DT_SYMBOLIC:
		case DT_TEXTREL:
#ifdef DT_BIND_NOW
		case DT_BIND_NOW:
#endif /* DT_BIND_NOW */
			fprintf(fp, value_format, buf, dump_dyntag(dyn->d_tag),
				(U64)dyn->d_un.d_val);
			break;
		case DT_PLTGOT:
		case DT_HASH:
		case DT_STRTAB:
		case DT_SYMTAB:
		case DT_RELA:
		case DT_INIT:
		case DT_FINI:
		case DT_REL:
		case DT_DEBUG:
		case DT_JMPREL:
#if defined(DT_INIT_ARRAY) && defined(DT_INIT_ARRAYSZ) && \
	defined(DT_FINI_ARRAY) && defined(DT_FINI_ARRAYSZ)
		case DT_INIT_ARRAYSZ:
		case DT_FINI_ARRAYSZ:
#endif
#ifdef DT_MOVETAB
		case DT_MOVETAB:
#endif /* DT_MOVETAB */
#ifdef DT_SYMINFO
		case DT_SYMINFO:
#endif /* DT_SYMINFO */
#ifdef DT_VERDEF
		case DT_VERDEF:
#endif /* DT_VERDEF */
#ifdef DT_VERNEED
		case DT_VERNEED:
#endif /* DT_VERNEED */
#ifdef DT_VERSYM
		case DT_VERSYM:
#endif /* DT_VERSYM */
			fprintf(fp, pointer_format, buf, dump_dyntag(dyn->d_tag),
				(U64)dyn->d_un.d_ptr);
			break;
		case DT_PLTRELSZ:
		case DT_RELASZ:
		case DT_RELAENT:
		case DT_STRSZ:
		case DT_SYMENT:
		case DT_RELSZ:
		case DT_RELENT:
		case DT_PLTREL:
#if defined(DT_INIT_ARRAY) && defined(DT_INIT_ARRAYSZ) && \
	defined(DT_FINI_ARRAY) && defined(DT_FINI_ARRAYSZ)
		case DT_INIT_ARRAY:
		case DT_FINI_ARRAY:
#endif
#ifdef DT_MOVEENT
		case DT_MOVEENT:
#endif /* DT_MOVEENT */
#ifdef DT_MOVESZ
		case DT_MOVESZ:
#endif /* DT_MOVESZ */
#ifdef DT_FEATURE_1
		case DT_FEATURE_1:
#endif /* DT_FEATURE_1 */
#ifdef DT_POSFLAG_1
		case DT_POSFLAG_1:
#endif /* DT_POSFLAG_1 */
#ifdef DT_SYMINSZ
		case DT_SYMINSZ:
#endif /* DT_SYMINSZ */
#ifdef DT_SYMINENT
		case DT_SYMINENT:
#endif /* DT_SYMINENT */
#ifdef DT_RELACOUNT
		case DT_RELACOUNT:
#endif /* DT_RELACOUNT */
#ifdef DT_RELCOUNT
		case DT_RELCOUNT:
#endif /* DT_RELCOUNT */
#ifdef DT_FLAGS_1
		case DT_FLAGS_1:
#endif /* DT_FLAGS_1 */
#ifdef DT_VERDEFNUM
		case DT_VERDEFNUM:
#endif /* DT_VERDEFNUM */
#ifdef DT_VERNEEDNUM
		case DT_VERNEEDNUM:
#endif /* DT_VERNEEDNUM */
#ifdef DT_SPARC_REGISTER
		case DT_SPARC_REGISTER:
#endif /* DT_SPARC_REGISTER */
#ifdef DT_USED
		case DT_USED:
#endif /* DT_USED */
			fprintf(fp, value_format, buf, dump_dyntag(dyn->d_tag),
				(U64)dyn->d_un.d_val);
			break;
		default:
			fprintf(fp, unknown_format, buf, (U64)dyn->d_tag,
				(U64)dyn->d_un.d_val);
			break;
	}
}

int
dump_hash(FILE *fp, Elf *elf, GElf_Shdr *sh, GElf_Word *hash, size_t n) {
	Elf_Data *data;
	Elf_Scn *symscn;
	GElf_Shdr symsh;
	GElf_Word *buckets;
	GElf_Word *chain;
	GElf_Word nbuckets;
	GElf_Word nchain;
	size_t *stats;
	size_t maxcount;
	size_t i;
	size_t j;

	assert(sizeof(Elf32_Word) == sizeof(Elf64_Word));
	nbuckets = hash[0];
	nchain = hash[1];
	assert(2 + nbuckets + nchain <= n);
	buckets = hash + 2;
	chain = buckets + nbuckets;
	/* get .dynsym */
	if (!gelf_getshdr(symscn = elf_getscn(elf, sh->sh_link), &symsh)) {
		return -1;
	}
	/* get data */
	if (!(data = elf_getdata(symscn, NULL))) {
		return -1;
	}
	maxcount = 0;
	stats = NULL;
	for (i = 0; i < nbuckets; i++) {
		size_t count = 0;
		GElf_Sym sym;
		char buf[32];
		char *sname;

		for (j = buckets[i]; j; j = chain[j]) {
			if (!gelf_getsym(data, j, &sym)) {
				return -1;
			}
			if (!(sname = elf_strptr(elf, symsh.sh_link, sym.st_name))) {
				return -1;
			}
			sprintf(buf, "[%lu]", (unsigned long)j);
			if (count++ == 0) {
				fprintf(fp, "%10u  %-10s  %s\n", (unsigned)i, buf, sname);
			}
			else {
				fprintf(fp, "%10s  %-10s  %s\n", "", buf, sname);
			}
		}
		/* collect statistics */
		if (count >= maxcount) {
			j = count + 7u;
			j -= j % 8u;
			stats = xrealloc(stats, j * sizeof(size_t));
			while (maxcount <= count) {
				stats[maxcount++] = 0;
			}
		}
		stats[count]++;
	}
	fprintf(fp, "\n");
	for (i = j = 0; i < maxcount; i++) {
		fprintf(fp, "%10u  buckets contain %8u symbols\n", stats[i], i);
		j += i * stats[i];
	}
	fprintf(fp, "%10u  buckets         %8u symbols (globals)\n",
		(unsigned)nbuckets, j);
	return 0;
}

int
dump_hash64(FILE *fp, Elf *elf, GElf_Shdr *sh, GElf_Addr *hash, size_t n) {
	Elf_Data *data;
	Elf_Scn *symscn;
	GElf_Shdr symsh;
	GElf_Addr *buckets;
	GElf_Addr *chain;
	GElf_Addr nbuckets;
	GElf_Addr nchain;
	size_t *stats;
	size_t maxcount;
	size_t i;
	size_t j;

	nbuckets = hash[0];
	nchain = hash[1];
	assert(2 + nbuckets + nchain <= n);
	buckets = hash + 2;
	chain = buckets + nbuckets;
	/* get .dynsym */
	if (!gelf_getshdr(symscn = elf_getscn(elf, sh->sh_link), &symsh)) {
		return -1;
	}
	/* get data */
	if (!(data = elf_getdata(symscn, NULL))) {
		return -1;
	}
	maxcount = 0;
	stats = NULL;
	for (i = 0; i < nbuckets; i++) {
		size_t count = 0;
		GElf_Sym sym;
		char buf[32];
		char *sname;

		for (j = buckets[i]; j; j = chain[j]) {
			if (!gelf_getsym(data, j, &sym)) {
				return -1;
			}
			if (!(sname = elf_strptr(elf, symsh.sh_link, sym.st_name))) {
				return -1;
			}
			sprintf(buf, "[%lu]", (unsigned long)j);
			if (count++ == 0) {
				fprintf(fp, "%10u  %-10s  %s\n", (unsigned)i, buf, sname);
			}
			else {
				fprintf(fp, "%10s  %-10s  %s\n", "", buf, sname);
			}
		}
		/* collect statistics */
		if (count >= maxcount) {
			j = count + 7u;
			j -= j % 8u;
			stats = xrealloc(stats, j * sizeof(size_t));
			while (maxcount <= count) {
				stats[maxcount++] = 0;
			}
		}
		stats[count]++;
	}
	fprintf(fp, "\n");
	for (i = j = 0; i < maxcount; i++) {
		fprintf(fp, "%10u  buckets contain %8u symbols\n", stats[i], i);
		j += i * stats[i];
	}
	fprintf(fp, "%10u  buckets         %8u symbols (globals)\n",
		(unsigned)nbuckets, j);
	return 0;
}

int
dump_symbols(FILE *fp, Elf *elf, Elf_Scn *scn, GElf_Shdr *sh) {
	static const char fmt32[] = "%10s  0x%08" XF64 " 0x%08" XF64 "  ";
	static const char fmt64[] = "%10s  0x%012" XF64 " 0x%012" XF64 "  ";
	static const char hdr32[] = "%10s  %10s %10s  type bind oth %-10s  %s\n";
	static const char hdr64[] = "%10s  %14s %14s  type bind oth %-10s  %s\n";
#if HAVE_SYMBOL_VERSIONING
	GElf_Versym *symvers = NULL;
#endif /* HAVE_SYMBOL_VERSIONING */
	Elf_Data *data;
	Elf_Data *dstr;
	GElf_Ehdr eh;
	GElf_Sym sym;
	char *sname;
	char buf[32];
	size_t i;
	size_t n;

	if (!gelf_getehdr(elf, &eh)) {
		return -1;
	}
	n = gelf_fsize(elf, ELF_T_SYM, 1, eh.e_version);
	if (n == 0 || sh->sh_size % n) {
		return -1;
	}
	n = sh->sh_size / n;
	if (!(data = elf_getdata(scn, NULL))
	 || !(dstr = elf_getdata(elf_getscn(elf, sh->sh_link), NULL))
	 || !(data->d_buf && dstr->d_buf)) {
		return -1;
	}
#if HAVE_SYMBOL_VERSIONING
	if (opt_v || opt_a) {
		Elf_Data *dver;
		Elf_Scn *scn2;
		GElf_Shdr sh2;

		scn2 = NULL;
		while ((scn2 = et_find_section_by_type(elf, SHT_SUNW_versym, scn2))) {
			if (!gelf_getshdr(scn2, &sh2)) {
				return -1;
			}
			if (sh2.sh_link == elf_ndxscn(scn)) {
				if (!(dver = elf_getdata(scn2, NULL))
				 || !dver->d_buf
				 || dver->d_size < n * sizeof(GElf_Versym)) {
					return -1;
				}
				if (dver->d_type == ELF_T_BYTE) {
					Elf_Data src = *dver;

					/* translate first */
					src.d_type = ELF_T_HALF;
					src.d_version = eh.e_version;
					dver->d_version = EV_CURRENT;
					if (!gelf_xlatetom(elf, dver, &src, eh.e_ident[EI_DATA])) {
						return -1;
					}
				}
				else if (dver->d_type != ELF_T_HALF) {
					return -1;
				}
				symvers = (GElf_Versym*)dver->d_buf;
				break;
			}
		}
	}
#endif /* HAVE_SYMBOL_VERSIONING */
	if (eh.e_ident[EI_CLASS] == ELFCLASS32) {
		fprintf(fp, hdr32, "index", "value   ", "size   ", "shndx", "name");
	}
	else {
		fprintf(fp, hdr64, "index", "value   ", "size   ", "shndx", "name");
	}
	for (i = 0; i < n; i++) {
		if (!gelf_getsym(data, i, &sym)) {
			return -1;
		}
		if (sym.st_name < 0 || sym.st_name >= dstr->d_size) {
			return -1;
		}
		sname = (char*)dstr->d_buf + sym.st_name;
		sprintf(buf, "[%u]", (unsigned)i);
		if (eh.e_ident[EI_CLASS] == ELFCLASS32) {
			fprintf(fp, fmt32, buf, (U64)sym.st_value, (U64)sym.st_size);
		}
		else {
			fprintf(fp, fmt64, buf, (U64)sym.st_value, (U64)sym.st_size);
		}
#if HAVE_SYMBOL_VERSIONING
		if ((opt_v || opt_a) && symvers) {
			fprintf(fp, "%-4s %-4s %-3d %-11s %s\n",
				dump_symtype(sym.st_info), dump_symbind(sym.st_info),
				symvers[i], dump_scnname(elf, sym.st_shndx), sname);
		}
		else
#endif /* HAVE_SYMBOL_VERSIONING */
		fprintf(fp, "%-4s %-4s %-3d %-11s %s\n",
			dump_symtype(sym.st_info), dump_symbind(sym.st_info),
			sym.st_other, dump_scnname(elf, sym.st_shndx), sname);
	}
	return 0;
}

int
dump_reloc(FILE *fp, Elf *elf, GElf_Ehdr *eh, Elf_Scn *scn, GElf_Shdr *sh) {
	Elf_Data *data;
	GElf_Addr shstrndx;
	const char *scnname;
	size_t i;
	size_t n;

	if (!(data = elf_getdata(scn, NULL))) {
		return -1;
	}
	if (!(shstrndx = et_get_shstrndx(elf))) {
		return -1;
	}
	if (!(scnname = elf_strptr(elf, shstrndx, sh->sh_name))) {
		return -1;
	}
	n = gelf_fsize(elf, data->d_type, 1, eh->e_version);
	if (n == 0 || sh->sh_size % n != 0) {
		return -1;
	}
	n = sh->sh_size / n;
	if (sh->sh_type == SHT_RELA) {
		GElf_Rela rela;

		assert(data->d_type == ELF_T_RELA);
		for (i = 0; i < n; i++) {
			char buf[32];

			if (!gelf_getrela(data, i, &rela)) {
				return -1;
			}
			if (rela.r_addend < 0) {
				sprintf(buf, "-%#" XF64, (U64)-rela.r_addend);
			}
			else {
				sprintf(buf, "%#" XF64, (U64)rela.r_addend);
			}
			fprintf(fp, "      %-20s %#10" XF64 " %8s  %-14.14s %s\n",
				dump_reltype(elf, GELF_R_TYPE(rela.r_info)),
				(U64)rela.r_offset, buf, scnname,
				dump_symname(elf, sh->sh_link, GELF_R_SYM(rela.r_info), 1));
		}
	}
	else {
		GElf_Rel rel;

		assert(data->d_type == ELF_T_REL);
		for (i = 0; i < n; i++) {
			if (!gelf_getrel(data, i, &rel)) {
				return -1;
			}
			fprintf(fp, "      %-20s %#10" XF64 " %#8" XF64 "  %-14.14s %s\n",
				dump_reltype(elf, GELF_R_TYPE(rel.r_info)),
				(U64)rel.r_offset, (U64)0, scnname,
				dump_symname(elf, sh->sh_link, GELF_R_SYM(rel.r_info), 1));
		}
	}
	return 0;
}

static GElf_Half
__load_lsb_16(const unsigned char *p) {
	return (GElf_Half)p[1] << 8 | p[0];
}

static GElf_Word
__load_lsb_32(const unsigned char *p) {
	return (GElf_Word)__load_lsb_16(p + 2) << 16 | __load_lsb_16(p);
}

static GElf_Half
__load_msb_16(const unsigned char *p) {
	return (GElf_Half)p[0] << 8 | p[1];
}

static GElf_Word
__load_msb_32(const unsigned char *p) {
	return (GElf_Word)__load_msb_16(p) << 16 | __load_msb_16(p + 2);
}

static unsigned
__encoding(Elf *elf) {
	static Elf *lastelf = NULL;
	static GElf_Ehdr eh;

	if (elf != lastelf) {
		/* not in cache */
		if (!gelf_getehdr(lastelf = elf, &eh)) {
			return -1;
		}
	}
	return eh.e_ident[EI_DATA];
}

static int
dump_note(FILE *fp, Elf *elf, Elf_Scn *scn, GElf_Shdr *sh) {
	union {
		Elf32_Addr addr32[3];
		Elf64_Addr addr64[3];
	} buf;
	Elf_Data *data;
	Elf_Data src, dst;
	GElf_Addr descsz;
	GElf_Addr namesz;
	GElf_Addr type;
	size_t i;
	size_t off;
	size_t wsz;
	unsigned char *p;

	if (!(data = elf_rawdata(scn, NULL))
	 || (data->d_size && !data->d_buf)) {
		return -1;
	}
	wsz = gelf_fsize(elf, ELF_T_ADDR, 1, data->d_version);
	assert(wsz);
	fprintf(fp, "%10s  %-26s  %s\n", "type", "name", "desc");
	off = 0;
	src = *data;
	src.d_type = ELF_T_ADDR;
	src.d_size = 3 * wsz;
	dst.d_buf = &buf;
	dst.d_type = ELF_T_ADDR;
	dst.d_size = sizeof(buf);
	dst.d_version = EV_CURRENT;
	p = data->d_buf;
	while (off < data->d_size) {
		src.d_buf = p + off;
		if (off + 3 * wsz > data->d_size) {
			fprintf(stderr, "premature end\n");
			return -1;
		}
		if (!gelf_xlatetom(elf, &dst, &src, __encoding(elf))) {
			fprintf(stderr, "translation error: %s\n", elf_errmsg(-1));
			return -1;
		}
		off += 3 * wsz;
		if (wsz == 8) {
			namesz = buf.addr64[0];
			descsz = buf.addr64[1];
			type   = buf.addr64[2];
		}
		else {
			assert(wsz == 4);
			namesz = buf.addr32[0];
			descsz = buf.addr32[1];
			type   = buf.addr32[2];
		}
#define rnd(x)	((x)+wsz-1-((x)+wsz-1)%wsz)
		if (off + rnd(namesz) + rnd(descsz) > data->d_size) {
			fprintf(stderr, "premature end (2)\n");
			return -1;
		}
		if (namesz && p[off + namesz - 1]) {
			/* not NUL-terminated */
			fprintf(stderr, "missing NUL\n");
			return -1;
		}
		fprintf(fp, "%10u  %-26s ", (unsigned)type,
			namesz ? (char*)p + off : "");
		off += rnd(namesz);
		if (descsz) {
			for (i = 0; i < descsz; i++) {
				if (i && i % 4 == 0) {
					fprintf(fp, "\n%39s", "");
				}
				fprintf(fp, " %02x", p[off + i]);
			}
			off += rnd(descsz);
		}
		fprintf(fp, "\n");
	}
#undef rnd
	return 0;
}

#if HAVE_SYMBOL_VERSIONING

int
get_verdef(Elf *elf, GElf_Verdef *vd, Elf_Data *data, size_t off) {
	unsigned char *p = (unsigned char*)data->d_buf + off;

	if (off + sizeof(*vd) > data->d_size) {
		return -1;
	}
	if (data->d_type == ELF_T_VDEF) {
		/* already translated, just copy */
		memcpy(vd, p, sizeof(*vd));
		return 0;
	}
	if (data->d_type == ELF_T_BYTE) {
		/* raw data, translate first */
		switch (__encoding(elf)) {
			case ELFDATA2LSB:
				vd->vd_version = __load_lsb_16(p +  0);
				if (vd->vd_version != VER_DEF_CURRENT) {
					return -1;
				}
				vd->vd_flags   = __load_lsb_16(p +  2);
				vd->vd_ndx     = __load_lsb_16(p +  4);
				vd->vd_cnt     = __load_lsb_16(p +  6);
				vd->vd_hash    = __load_lsb_32(p +  8);
				vd->vd_aux     = __load_lsb_32(p + 12);
				vd->vd_next    = __load_lsb_32(p + 16);
				return 0;
			case ELFDATA2MSB:
				vd->vd_version = __load_msb_16(p +  0);
				if (vd->vd_version != VER_DEF_CURRENT) {
					return -1;
				}
				vd->vd_flags   = __load_msb_16(p +  2);
				vd->vd_ndx     = __load_msb_16(p +  4);
				vd->vd_cnt     = __load_msb_16(p +  6);
				vd->vd_hash    = __load_msb_32(p +  8);
				vd->vd_aux     = __load_msb_32(p + 12);
				vd->vd_next    = __load_msb_32(p + 16);
				return 0;
		}
		return -1;
	}
	return -1;
}

int
get_verdaux(Elf *elf, GElf_Verdaux *vda, Elf_Data *data, size_t off) {
	char *p = (char*)data->d_buf + off;

	if (off + sizeof(*vda) > data->d_size) {
		return -1;
	}
	if (data->d_type == ELF_T_VDEF) {
		/* already translated, just copy */
		memcpy(vda, p, sizeof(*vda));
		return 0;
	}
	if (data->d_type == ELF_T_BYTE) {
		/* raw data, translate first */
		switch (__encoding(elf)) {
			case ELFDATA2LSB:
				vda->vda_name = __load_lsb_32(p + 0);
				vda->vda_next = __load_lsb_32(p + 4);
				return 0;
			case ELFDATA2MSB:
				vda->vda_name = __load_msb_32(p + 0);
				vda->vda_next = __load_msb_32(p + 4);
				return 0;
		}
		return -1;
	}
	return -1;
}

int
dump_verdef(FILE *fp, Elf *elf, Elf_Scn *scn, GElf_Shdr *sh) {
	static char **names = NULL;
	Elf_Data *data;
	Elf_Data *dstr;
	GElf_Verdaux vda;
	GElf_Verdef vd;
	char buf[32];
	char ibuf[32];
	size_t i;
	size_t off2;
	size_t off;

	if (!(data = elf_getdata(scn, NULL))) {
		return -1;
	}
	if (!(dstr = elf_getdata(elf_getscn(elf, sh->sh_link), NULL))) {
		return -1;
	}
	fprintf(fp, "%10s  %-26s  %s\n", "index", "version", "dependency");
	off = 0;
	do {
		if (get_verdef(elf, &vd, data, off)) {
			return -1;
		}
		if (vd.vd_version != VER_DEF_CURRENT) {
			return -1;
		}
		sprintf(ibuf, "[%u]", vd.vd_ndx);
		off2 = off + vd.vd_aux;
		if (vd.vd_cnt < 1) {
			return -1;
		}
		names = xrealloc(names, (vd.vd_cnt + 1) * sizeof(char*));
		i = 0;
		do {
			if (get_verdaux(elf, &vda, data, off2)) {
				return -1;
			}
			if (vda.vda_name >= dstr->d_size) {
				return -1;
			}
			names[i] = (char*)dstr->d_buf + vda.vda_name;
			off2 += vda.vda_next;
		}
		while (++i < vd.vd_cnt && vda.vda_next);
		if (vda.vda_next || i < vd.vd_cnt) {
			return -1;
		}
		names[i] = "";
		buf[0] = '\0';
		if (vd.vd_flags & ~(VER_FLG_BASE | VER_FLG_WEAK)) {
			sprintf(buf, "%#x", vd.vd_flags);
		}
		else if (vd.vd_flags & (VER_FLG_BASE | VER_FLG_WEAK)) {
			strcat(buf, "[");
			if (vd.vd_flags & VER_FLG_BASE) {
				strcat(buf, " BASE ");
			}
			if (vd.vd_flags & VER_FLG_WEAK) {
				strcat(buf, " WEAK ");
			}
			strcat(buf, "]");
		}
		fprintf(fp, "%10s  %-26s  %-20s  %s\n",
			ibuf, names[0], names[1], buf);
		for (i = 2; i < vd.vd_cnt; i++) {
			fprintf(fp, "%10s  %-26s  %s\n", "", "", names[i]);
		}
		off += vd.vd_next;
	}
	while (vd.vd_next);
	return 0;
}

int
get_verneed(Elf *elf, GElf_Verneed *vn, Elf_Data *data, size_t off) {
	unsigned char *p = (unsigned char*)data->d_buf + off;

	if (off + sizeof(*vn) > data->d_size) {
		return -1;
	}
	if (data->d_type == ELF_T_VNEED) {
		/* already translated, just copy */
		memcpy(vn, p, sizeof(*vn));
		return 0;
	}
	if (data->d_type == ELF_T_BYTE) {
		/* raw data, translate first */
		switch (__encoding(elf)) {
			case ELFDATA2LSB:
				vn->vn_version = __load_lsb_16(p +  0);
				if (vn->vn_version != VER_NEED_CURRENT) {
					return -1;
				}
				vn->vn_cnt     = __load_lsb_16(p +  2);
				vn->vn_file    = __load_lsb_32(p +  4);
				vn->vn_aux     = __load_lsb_32(p +  8);
				vn->vn_next    = __load_lsb_32(p + 12);
				return 0;
			case ELFDATA2MSB:
				vn->vn_version = __load_msb_16(p +  0);
				if (vn->vn_version != VER_NEED_CURRENT) {
					return -1;
				}
				vn->vn_cnt     = __load_msb_16(p +  2);
				vn->vn_file    = __load_msb_32(p +  4);
				vn->vn_aux     = __load_msb_32(p +  8);
				vn->vn_next    = __load_msb_32(p + 12);
				return 0;
		}
		return -1;
	}
	return -1;
}

int
get_vernaux(Elf *elf, GElf_Vernaux *vna, Elf_Data *data, size_t off) {
	char *p = (char*)data->d_buf + off;

	if (off + sizeof(*vna) > data->d_size) {
		return -1;
	}
	if (data->d_type == ELF_T_VNEED) {
		/* already translated, just copy */
		memcpy(vna, p, sizeof(*vna));
		return 0;
	}
	if (data->d_type == ELF_T_BYTE) {
		/* raw data, translate first */
		switch (__encoding(elf)) {
			case ELFDATA2LSB:
				vna->vna_hash  = __load_lsb_32(p +  0);
				vna->vna_flags = __load_lsb_16(p +  4);
				vna->vna_other = __load_lsb_16(p +  6);
				vna->vna_name  = __load_lsb_32(p +  8);
				vna->vna_next  = __load_lsb_32(p + 12);
				return 0;
			case ELFDATA2MSB:
				vna->vna_hash  = __load_msb_32(p +  0);
				vna->vna_flags = __load_msb_16(p +  4);
				vna->vna_other = __load_msb_16(p +  6);
				vna->vna_name  = __load_msb_32(p +  8);
				vna->vna_next  = __load_msb_32(p + 12);
				return 0;
		}
		return -1;
	}
	return -1;
}

int
dump_verneed(FILE *fp, Elf *elf, Elf_Scn *scn, GElf_Shdr *sh) {
	static const char format[] = "%10s  %-26s  %s\n";
	static char **names = NULL;
	Elf_Data *data;
	Elf_Data *dstr;
	GElf_Vernaux vna;
	GElf_Verneed vn;
	char *file;
	size_t i;
	size_t off2;
	size_t off;

	if (!(data = elf_getdata(scn, NULL))) {
		return -1;
	}
	if (!(dstr = elf_getdata(elf_getscn(elf, sh->sh_link), NULL))) {
		return -1;
	}
	fprintf(fp, format, "", "file", "version");
	off = 0;
	do {
		if (get_verneed(elf, &vn, data, off)) {
			return -1;
		}
		if (vn.vn_version != VER_NEED_CURRENT) {
			return -1;
		}
		if (vn.vn_file >= dstr->d_size) {
			return -1;
		}
		file = (char*)dstr->d_buf + vn.vn_file;
		off2 = off + vn.vn_aux;
		if (vn.vn_cnt < 1) {
			return -1;
		}
		names = xrealloc(names, vn.vn_cnt * sizeof(char*));
		i = 0;
		do {
			if (get_vernaux(elf, &vna, data, off2)) {
				return -1;
			}
			if (vna.vna_name >= dstr->d_size) {
				return -1;
			}
			names[i] = (char*)dstr->d_buf + vna.vna_name;
			off2 += vna.vna_next;
		}
		while (++i < vn.vn_cnt && vna.vna_next);
		if (vna.vna_next || i < vn.vn_cnt) {
			return -1;
		}
		fprintf(fp, format, "", file, names[0]);
		for (i = 1; i < vn.vn_cnt; i++) {
			fprintf(fp, format, "", "", names[i]);
		}
		off += vn.vn_next;
	}
	while (vn.vn_next);
	return 0;
}

#endif /* HAVE_SYMBOL_VERSIONING */

static int
dump_stabs(FILE *fp, Elf *elf, Elf_Scn *scn, GElf_Shdr *sh, const char *sname) {
	struct my_nlist {
		GElf_Word		n_name;
		unsigned char	n_type;
		unsigned char	n_other;
		GElf_Half		n_desc;
		GElf_Word		n_value;
	} fnl;
	Elf_Data *data;
	Elf_Data *dstr;
	Elf_Scn *sstr;
	GElf_Ehdr eh;
	GElf_Shdr sh2;
	size_t i;
	size_t off;

	/*
	 * Find string table
	 */
	if (!gelf_getehdr(elf, &eh)) {
		return -1;
	}
	if (sh->sh_link) {
		if (!gelf_getshdr(sstr = elf_getscn(elf, sh->sh_link), &sh2)) {
			return -1;
		}
	}
	else {
		char *name;
		size_t len;
		GElf_Addr shstrndx;

		if (!(shstrndx = et_get_shstrndx(elf))) {
			return -1;
		}
		len = strlen(sname);
		sstr = NULL;
		while (gelf_getshdr(sstr = elf_nextscn(elf, sstr), &sh2)) {
			if (!(name = elf_strptr(elf, shstrndx, sh2.sh_name))) {
				return -1;
			}
			if (memcmp(name, sname, len) == 0) {
				if (strcmp(name + len, "str") == 0) {
					break;
				}
			}
		}
		if (!sstr) {
			return -1;
		}
	}
	if (sh2.sh_type != SHT_STRTAB) {
		return -1;
	}
	/*
	 * Get section data
	 */
	if (!(data = elf_getdata(scn, NULL))
	 || !(dstr = elf_getdata(sstr, NULL))) {
		return -1;
	}
	fprintf(fp, "%10s  %10s  type  desc   %s\n", "index", "value  ", "name");
	fnl.n_value = 0;
	for (off = i = 0; i < data->d_size / 12; i++) {
		unsigned char *p = (unsigned char*)data->d_buf + 12 * i;
		const char *name;
		static char buf[32];
		struct my_nlist nl;

		if (eh.e_ident[EI_DATA] == ELFDATA2LSB) {
			nl.n_name  = __load_lsb_32(p + 0);
			nl.n_type  = *(p + 4);
			nl.n_other = *(p + 5);
			nl.n_desc  = __load_lsb_16(p + 6);
			nl.n_value = __load_lsb_32(p + 8);
		}
		else {
			nl.n_name  = __load_msb_32(p + 0);
			nl.n_type  = *(p + 4);
			nl.n_other = *(p + 5);
			nl.n_desc  = __load_msb_16(p + 6);
			nl.n_value = __load_msb_32(p + 8);
		}
		if (nl.n_type == 0) {
			off += fnl.n_value;
			fnl = nl;
		}
		if (off + nl.n_name >= dstr->d_size) {
			return -1;
		}
		name = (char*)dstr->d_buf + off + nl.n_name;
		sprintf(buf, "[%u]", (unsigned)i);
		fprintf(fp, "%10s  0x%08x  %#4x %6d  %s\n",
			buf, (unsigned)nl.n_value, nl.n_type, nl.n_desc, name);
	}
	return 0;
}

static int
elf_dump(FILE *fp, Elf *elf) {
    GElf_Ehdr eh;
    GElf_Shdr sh;
    GElf_Addr shstrndx;
    Elf_Scn *scn;
    char *sname;

    if (!gelf_getehdr(elf, &eh)) {
        return -1;
    }
    if (opt_e || (opt_a && !opt_N)) {
        fprintf(fp, "\nELF Header:\n");
        fprintf(fp, "  ei_magic:   { 0x7f, E, L, F }\n");
        fprintf(fp, "  ei_class:   %-18s  ", dump_classname(eh.e_ident[EI_CLASS]));
        fprintf(fp, "ei_data:      %s\n", dump_dataname(eh.e_ident[EI_DATA]));
		fprintf(fp, "  e_machine:  %-18s  ", dump_machname(eh.e_machine));
		fprintf(fp, "e_version:    %s\n", dump_versname(eh.e_version));
		fprintf(fp, "  e_type:     %s\n", dump_typename(eh.e_type));
		fprintf(fp, "  e_flags:    %18x\n", (unsigned)eh.e_flags);
		fprintf(fp, "  e_entry:    %#18" XF64 "  ", (U64)eh.e_entry);
		fprintf(fp, "e_ehsize:     %2u  ", eh.e_ehsize);
		fprintf(fp, "e_shstrndx:   %2u\n", eh.e_shstrndx);
		fprintf(fp, "  e_shoff:    %#18" XF64 "  ", (U64)eh.e_shoff);
		fprintf(fp, "e_shentsize:  %2u  ", eh.e_shentsize);
		fprintf(fp, "e_shnum:      %2u\n", eh.e_shnum);
		fprintf(fp, "  e_phoff:    %#18" XF64 "  ", (U64)eh.e_phoff);
		fprintf(fp, "e_phentsize:  %2u  ", eh.e_phentsize);
		fprintf(fp, "e_phnum:      %2u\n", eh.e_phnum);
    }
    if (opt_p || (opt_a && !opt_N)) {
		GElf_Phdr ph;
		unsigned i;

		for (i = 0; i < eh.e_phnum; i++) {
			if (!gelf_getphdr(elf, i, &ph)) {
				return -1;
			}
			fprintf(fp, "\nProgram Header[%u]:\n", i);
			fprintf(fp, "    p_vaddr:      %#-12" XF64, (U64)ph.p_vaddr);
			fprintf(fp, "    p_flags:    [%s]\n", dump_pflags(ph.p_flags));
			fprintf(fp, "    p_paddr:      %#-12" XF64, (U64)ph.p_paddr);
			fprintf(fp, "    p_type:     [ %s ]\n", dump_ptype(ph.p_type));
			fprintf(fp, "    p_filesz:     %#-12" XF64, (U64)ph.p_filesz);
			fprintf(fp, "    p_memsz:    %#" XF64 "\n", (U64)ph.p_memsz);
			fprintf(fp, "    p_offset:     %#-12" XF64, (U64)ph.p_offset);
			fprintf(fp, "    p_align:    %#" XF64 "\n", (U64)ph.p_align);
		}
    }
    if (!(shstrndx = et_get_shstrndx(elf))) {
		return -1;
    }
    scn = NULL;
    elf_errno();
    while (gelf_getshdr(scn = elf_nextscn(elf, scn), &sh)) {
        if (!(sname = elf_strptr(elf, shstrndx, sh.sh_name))) {
			return -1;
        }
        if (opt_N && strcmp(sname, opt_N) != 0) {
			/* skip section */
			continue;
        }
        if (opt_c || opt_a) {
			unsigned long i = elf_ndxscn(scn);

			fprintf(fp, "\nSection Header[%lu]:  sh_name: %s\n", i, sname);
			fprintf(fp, "    sh_addr:      %#-12" XF64, (U64)sh.sh_addr);
			fprintf(fp, "    sh_flags:   %s\n", dump_shflags(sh.sh_flags));
			fprintf(fp, "    sh_size:      %#-12" XF64, (U64)sh.sh_size);
			fprintf(fp, "    sh_type:    [ %s ]\n", dump_shtype(sh.sh_type));
			fprintf(fp, "    sh_offset:    %#-12" XF64, (U64)sh.sh_offset);
			fprintf(fp, "    sh_entsize: %#" XF64 "\n", (U64)sh.sh_entsize);
			fprintf(fp, "    sh_link:      %-12u", (unsigned)sh.sh_link);
			fprintf(fp, "    sh_info:    %u\n", (unsigned)sh.sh_info);
			fprintf(fp, "    sh_addralign: %#" XF64 "\n", (U64)sh.sh_addralign);
        }
        switch (sh.sh_type) {
			case SHT_NULL:
				break;
			case SHT_PROGBITS:
				if (opt_i || opt_a) {
					Elf_Data *data;

					if (opt_N || strcmp(sname, ".interp") == 0) {
						/* dump the contents of the .interp section */
						fprintf(fp, "\nInterpreter Section:  %s\n", sname);
						if (!(data = elf_getdata(scn, NULL))) {
							return -1;
						}
						assert(!elf_getdata(scn, data));
						if (data->d_size) {
							assert(data->d_buf);
							fprintf(fp, "\t%.*s\n",
								(int)data->d_size, (char*)data->d_buf);
						}
					}
				}
#if 0 /* XXX: processor-specific */
				if (opt_G || opt_a) {
					if (opt_N || strcmp(sname, ".got") == 0) {
						/* XXX: dump the contents of the GOT section */
					}
				}
#endif
				if (opt_S || opt_a) {
					if (opt_N
					 || strcmp(sname, ".stab") == 0
					 || strcmp(sname, ".stab.excl") == 0
					 || strcmp(sname, ".stab.index") == 0) {
						fprintf(fp, "\nDebugging Information:  %s\n", sname);
						if (dump_stabs(fp, elf, scn, &sh, sname)) {
							return -1;
						}
					}
				}
				break;
			case SHT_SYMTAB:
			case SHT_DYNSYM:
				if (opt_s || opt_a) {
					fprintf(fp, "\nSymbol Table:  %s\n", sname);
					if (dump_symbols(fp, elf, scn, &sh)) {
						return -1;
					}
				}
				break;
			case SHT_RELA:
			case SHT_REL:
				if (opt_r || opt_a) {
					fprintf(fp, "\nRelocation: %s\n", sname);
					fprintf(fp, "\t%s\t\t%13s%9s%9s%23s\n", "type",
						"offset", "addend", "section", "with respect to");
					if (dump_reloc(fp, elf, &eh, scn, &sh)) {
						return -1;
					}
				}
				break;
			case SHT_HASH:
				if (opt_h || opt_a) {
					Elf_Data *data;
					size_t esz, n;

					if (!(data = elf_getdata(scn, NULL))) {
						return -1;
					}
					esz = gelf_fsize(elf, data->d_type, 1, eh.e_version);
					switch (esz) {
						case 4: break;
						case 8: warn("non-standard hash table"); break;
						default: return -1;
					}
					if ((n = sh.sh_size / esz) < 2) {
						return -1;
					}
					fprintf(fp, "\nHash Section:  %s\n", sname);
					fprintf(fp, "%10s  %10s  %s\n", "bucket", "symndx  ", "name");
					if (esz == 4) {
						if (dump_hash(fp, elf, &sh, data->d_buf, n)) {
							return -1;
						}
					}
					else {
						if (dump_hash64(fp, elf, &sh, data->d_buf, n)) {
							return -1;
						}
					}
				}
				break;
			case SHT_DYNAMIC:
				if (opt_d || opt_a) {
					Elf_Data *data;
					Elf_Data *dstr;
					GElf_Dyn dyn;
					size_t i;
					size_t n;

					n = gelf_fsize(elf, ELF_T_DYN, 1, eh.e_version);
					if (n == 0 || sh.sh_size % n) {
						return -1;
					}
					n = sh.sh_size / n;
					if (!(data = elf_getdata(scn, NULL))) {
						return -1;
					}
					if (!(dstr = elf_getdata(elf_getscn(elf, sh.sh_link), NULL))) {
						return -1;
					}
					fprintf(fp, "\nDynamic Section:  %s\n", sname);
					fprintf(fp, "%10s  %-12s  %s\n", "index", "tag", "value");
					for (i = 0; i < n; i++) {
						if (!gelf_getdyn(data, i, &dyn)) {
							return -1;
						}
						if (dyn.d_tag == DT_NULL) {
							break;
						}
						dump_dyn(fp, &dyn, i, dstr);
					}
				}
				break;
			case SHT_NOTE:
				if (opt_n || opt_a) {
					fprintf(fp, "Note Section:  %s\n", sname);
					if (dump_note(fp, elf, scn, &sh)) {
						return -1;
					}
				}
				break;
#if HAVE_SYMBOL_VERSIONING
			case SHT_SUNW_verdef:
				if (opt_v || opt_a) {
					fprintf(fp, "\nVersion Definition Section:  %s\n", sname);
					if (dump_verdef(fp, elf, scn, &sh)) {
						return -1;
					}
				}
				break;
			case SHT_SUNW_verneed:
				if (opt_v || opt_a) {
					fprintf(fp, "\nVersion Needed Section:  %s\n", sname);
					if (dump_verneed(fp, elf, scn, &sh)) {
						return -1;
					}
				}
				break;
#endif /* HAVE_SYMBOL_VERSIONING */
#ifdef SHT_SUNW_move
			case SHT_SUNW_move:
				if (opt_m || opt_a) {
					/* XXX: dump move section */
				}
				break;
#endif /* SHT_SUNW_move */
#ifdef SHT_SUNW_syminfo
			case SHT_SUNW_syminfo:
				if (opt_y || opt_a) {
					/* XXX: dump syminfo section */
				}
				break;
#endif /* SHT_SUNW_syminfo */
			case SHT_STRTAB:
			case SHT_NOBITS:
			case SHT_SHLIB:
			default:
				break;
        }
    }
    if (elf_errmsg(0)) {
		return -1;
    }
    if (opt_k) {
		fprintf(fp, "\nelf checksum: %#lx\n", gelf_checksum(elf));
    }
    return 0;
}

static const char*
member_name(Elf *elf, int fd, size_t off) {
	static char *buf = NULL;
	Elf *elf2 = NULL;
	Elf_Arhdr *ah;
	const char *s;

	if (elf_rand(elf, off) != off
	 || !(elf2 = elf_begin(fd, ELF_C_READ, elf))
	 || !(ah = elf_getarhdr(elf2))
	 || !(s = ah->ar_name)) {
		s = "*ERROR*";
	}
	else {
		buf = xrealloc(buf, strlen(s) + 1);
		s = strcpy(buf, s);
	}
	elf_end(elf2);	/* NULL argument is allowed */
	return s;
}

static int
process_file(FILE *fp, const char *fname, int print_names) {
	Elf *elf;
	Elf *elf2;
	Elf_Arhdr *ah;
	Elf_Cmd cmd;
	int err;
	int fd;

	err = 0;
	if ((fd = open(fname, O_RDONLY)) != -1) {
		if ((elf = elf_begin(fd, ELF_C_READ, NULL))) {
			if ((opt_s || opt_a) && !opt_N && elf_kind(elf) == ELF_K_AR) {
				const char *member = NULL;
				size_t off = 0;
				Elf_Arsym *as;
				char buf[32];
				size_t i;
				size_t n;

				if ((as = elf_getarsym(elf, &n))) {
					fprintf(fp, "\nSymbol Table: %s\n", "(archive)");
					fprintf(fp, "%10s  %-10s  %s\n",
						"index", "  offset", "member name and symbol");
					for (i = 0; i < n; i++, as++) {
						sprintf(buf, "[%u]", (unsigned)i);
						if (as->as_name) {
							if (!member || off != as->as_off) {
								off = as->as_off;
								member = member_name(elf, fd, off);
							}
							fprintf(fp, "%10s  0x%08x  (%s):%s\n",
								buf, (unsigned)as->as_off, member,
								as->as_name);
						}
						else {
							fprintf(fp, "%10s  0x%08x\n",
								buf, (unsigned)as->as_off);
						}
					}
				}
				/* rewind archive */
				if (elf_rand(elf, SARMAG) != SARMAG) {
					file_error(fname, "elf_rand: %s", elf_errmsg(-1));
					err = -1;
				}
			}
			cmd = ELF_C_READ;
			while ((elf2 = elf_begin(fd, cmd, elf))) {
				if (!(ah = elf_getarhdr(elf2))) {
					elf_errno();
					if (print_names) {
						fprintf(fp, "\n%s:\n", fname);
					}
					err |= elf_dump(fp, elf2);
				}
				else if (ah->ar_name && *ah->ar_name != '/') {
					fprintf(fp, "\n%s(%s):\n", fname, ah->ar_name);
					err |= elf_dump(fp, elf2);
				}
				cmd = elf_next(elf2);
				elf_end(elf2);
			}
			elf_end(elf);
		}
		else {
			file_error(fname, "elf_begin: %s", elf_errmsg(-1));
			err = -1;
		}
		close(fd);
	}
	else {
		file_error(fname, "open: %s", strerror(errno));
		err = -1;
	}
	return err;
}

static void
usage(void) {
    fprintf(stderr,
        "usage: %s [-acdeGhikmnprSsVvy] [-w file] [-N name] file...\n",
        progname);
	fprintf(stderr,
        "options:\n"
        "  -a        dump all sections (same as -cdGhimnrsvy)\n"
        "  -c        dump section header information\n"
        "  -d        dump the contents of the .dynamic section\n"
        "  -e        dump the elf header\n"
        "  -G        dump the contents of the GOT section (*)\n"
        "  -h        dump the contents of the .hash section\n"
        "  -i        dump the contents of the .interp section\n"
        "  -k        calculate elf checksum\n"
        "  -m        dump the contents of the move section (*)\n"
        "  -n        dump the contents of the .note section\n"
        "  -p        dump the program headers\n"
        "  -r        dump the contents of the relocation sections\n"
        "  -S        dump `stabs' debugging information sections\n"
        "  -s        dump the contents of the symbol table sections\n"
        "  -V        display program version and exit\n"
        "  -v        dump the contents of the version sections (*)\n"
        "  -y        dump the contents of the syminfo section (*)\n"
        "  -w file   write the contents of specified section to file\n"
        "  -N name   qualify an option with a name\n"
        "(options marked with (*) may not work on some systems)\n"
        );
	exit(1);
}

int
main(int argc, char **argv) {
	FILE *outfp;
	int c;
	int err;
	int multi;
	int opts;

    setprogname(progname);

    opts = 0;
    while ((c = getopt(argc, argv, "acdehikmnprsvw:yGN:SV")) != EOF) {
        switch (c) {
            case 'a': opts = opt_a = 1; break;
            case 'c': opts = opt_c = 1; break;
            case 'd': opts = opt_d = 1; break;
            case 'e': opts = opt_e = 1; break;
            case 'h': opts = opt_h = 1; break;
            case 'i': opts = opt_i = 1; break;
            case 'k': opts = opt_k = 1; break;
            case 'm': opts = opt_m = 1; break;
            case 'n': opts = opt_n = 1; break;
            case 'p': opts = opt_p = 1; break;
            case 'r': opts = opt_r = 1; break;
            case 's': opts = opt_s = 1; break;
            case 'v': opts = opt_v = 1; break;
            case 'w': opt_w = optarg; break;
            case 'y': opts = opt_y = 1; break;
            case 'G': opts = opt_G = 1; break;
            case 'N': opt_N = optarg; break;
            case 'S': opts = opt_S = 1; break;
            case 'V': opt_V = 1; break;
            case '?': usage();
        }
    }
    if (opt_V) {
        show_version("elfdump");
        exit(0);
    }
    if (!opts) {
		error("please specify what to dump");
		usage();
    }
    if (!opt_w || (opt_w[0] == '-' && !opt_w[1])) {
		outfp = stdout;
    }
    else if (!(outfp = fopen(opt_w, "w"))) {
		file_error(opt_w, "open: %s", strerror(errno));
		exit(1);
    }
    if (elf_version(EV_CURRENT) == EV_NONE) {
        error("libelf version mismatch");
        exit(1);
    }
    err = 0;
    multi = optind + 1 < argc;
    while (optind < argc) {
        if (process_file(outfp, argv[optind++], multi)) {
			err = 1;
        }
    }
    exit(err);
}
