/*
 * ldout.c -- ld(1) output routines
 * Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ldout.c,v 1.10 2003/02/08 13:09:28 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <gelf.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

#define elf_error(x)	error("%s: %s", (x), elf_errmsg(-1))

Elf_Data ashstrtab = { NULL, };

extern int ld_output_relocs(int apply);

int
ld_define_internal_symbols(void) {
	struct outscn *lasttext = NULL;
	struct outscn *lastdata = NULL;
	struct outscn *lastbss = NULL;
	struct outscn *sect;
	GElf_Sym sym;

	/*
	 * Find last section(s).
	 * Add .text and .data sections if they're missing
	 */
	if (!text_segment->scns) {
		sect = ld_dummy_section(".text", SHT_PROGBITS, SHF_ALLOC | SHF_EXECINSTR);
		sect->next = text_segment->scns;
		text_segment->scns = sect;
	}
	for (sect = text_segment->scns; sect; sect = sect->next) {
		lasttext = sect;
	}
	assert(lasttext);
	if (!(sect = data_segment->scns) || sect->shdr.sh_type == SHT_NOBITS) {
		sect = ld_dummy_section(".data", SHT_PROGBITS, SHF_ALLOC | SHF_WRITE);
		sect->next = data_segment->scns;
		data_segment->scns = sect;
	}
	for (sect = data_segment->scns; sect; sect = sect->next) {
		if (sect->shdr.sh_type != SHT_NOBITS) {
			lastdata = sect;
		}
		lastbss = sect;
	}
	assert(lastdata);
	assert(lastbss);

	sym.st_name = 0;	/* filled in by ld_resolve_symbol */
	sym.st_size = 0;
	sym.st_info = GELF_ST_INFO(STB_GLOBAL, STT_OBJECT);
	sym.st_other = 0;
	sym.st_shndx = 42;	/* doesn't matter, but must not be SHN_UNDEF */

	sym.st_value = lasttext->shdr.sh_size;
	ld_resolve_symbol("(internal)", "_etext", &sym, lasttext);

	sym.st_value = lastdata->shdr.sh_size;
	ld_resolve_symbol("(internal)", "_edata", &sym, lastdata);

	sym.st_value = lastbss->shdr.sh_size;
	ld_resolve_symbol("(internal)", "_end", &sym, lastbss);

	if (errors) {
		return -1;
	}
	return 0;
}

void
ld_transform_common_symbols(void) {
	struct outscn *lastbss = NULL;
	GElf_Xword align;
	size_t i;
	struct gsym *gsym;
	struct outscn *sect;

	for (sect = data_segment->scns; sect; sect = sect->next) {
		lastbss = sect;
	}
	assert(lastbss);
	if (lastbss->shdr.sh_type != SHT_NOBITS) {
		sect = ld_dummy_section(".bss", SHT_NOBITS, SHF_ALLOC | SHF_WRITE);
		lastbss = lastbss->next = sect;
	}

	for (i = 1; i < anglobs; i++) {
		gsym = &aglobals[i];
		if (gsym->sym.st_shndx != SHN_COMMON) {
			continue;
		}
		align = gsym->sym.st_value;
		if (align > 1 && lastbss->shdr.sh_size % align) {
			lastbss->shdr.sh_size += align - lastbss->shdr.sh_size % align;
		}
		if (lastbss->shdr.sh_addralign < align) {
			lastbss->shdr.sh_addralign = align;
		}
		gsym->sym.st_value = lastbss->shdr.sh_size;
		gsym->sym.st_shndx = 42;	/* value doesn't matter */
		gsym->sect = lastbss;
		lastbss->shdr.sh_size += gsym->sym.st_size;
		debug("moved common symbol `%s' to 0x%Lx:0x%Lx",
			ld_symbol_name(gsym->sym.st_name),
			(unsigned long long)gsym->sym.st_value,
			(unsigned long long)gsym->sym.st_size);
	}
}

int
ld_output_sections(void) {
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Shdr shdr;
	struct outscn *sect;
	struct segment *seg;

	/*
	 * Create and fill ELF sections
	 */
	ld_add_string(&ashstrtab, NULL);
	for (seg = segments; seg; seg = seg->next) {
		for (sect = seg->scns; sect; sect = sect->next) {
			/*
			 * Copy section to ELF output file
			 */
			assert(!sect->scn);
			debug("* %s : %s (%Lu bytes)", seg->name ? seg->name : "-",
				sect->name, (unsigned long long)sect->shdr.sh_size);
			if (!gelf_getshdr(scn = elf_newscn(aelf), &shdr)) {
				elf_error("gelf_getshdr");
				return -1;
			}
			if (!(data = elf_newdata(scn))) {
				elf_error("elf_newdata");
				return -1;
			}
			data->d_buf = NULL;
			if (sect->shdr.sh_type != SHT_NOBITS) {
				data->d_buf = sect->buf;
			}
			data->d_size = sect->shdr.sh_size;
			data->d_align = sect->shdr.sh_addralign;
			shdr.sh_name = ld_add_string(&ashstrtab, sect->name);
			shdr.sh_type = sect->shdr.sh_type;
			shdr.sh_flags = sect->shdr.sh_flags;
			if (!gelf_update_shdr(scn, &shdr)) {
				elf_error("gelf_update_shdr");
				return -1;
			}
			sect->scn = scn;
		}
	}
	return 0;
}

int
ld_output_symtab(void) {
	int use_escapes = 0;
	Elf_Data *data;
	Elf_Scn *scn;
	Elf_Scn *strs;
	GElf_Shdr shdr;
	GElf_Word *p;
	size_t i;
	size_t index;
	size_t j;
	struct outscn *sect;
	struct segment *seg;

	/*
	 * Add symbol table
	 */
	if (!(scn = elf_newscn(aelf))
	 || !(strs = elf_newscn(aelf))) {
		elf_error("elf_newscn");
		return -1;
	}
	/*
	 * First, process the string table
	 */
	if (!gelf_getshdr(strs, &shdr)) {
		elf_error("gelf_getshdr");
		return -1;
	}
	if (!(data = elf_newdata(strs))) {
		elf_error("elf_newdata");
		return -1;
	}
	*data = astrtab;
	shdr.sh_name = ld_add_string(&ashstrtab, ".strtab");
	shdr.sh_type = SHT_STRTAB;
	if (!gelf_update_shdr(strs, &shdr)) {
		elf_error("gelf_update_shdr");
		return -1;
	}
	/*
	 * Second, fill the symbol table
	 */
	if (!gelf_getshdr(scn, &shdr)) {
		elf_error("gelf_getshdr");
		return -1;
	}
	if (!(data = elf_newdata(scn))) {
		elf_error("elf_newdata");
		return -1;
	}
	switch (aeh.e_ident[EI_CLASS]) {
		case ELFCLASS32: i = sizeof(Elf32_Sym); break;
		case ELFCLASS64: i = sizeof(Elf64_Sym); break;
		default: i = 0;
	}
	if (i == 0) {
		fatal("unknown ELF class `%u'", aeh.e_ident[EI_CLASS]);
	}
	j = anlocs - (anlocs != 0) + anglobs - (anglobs != 0) + 1;
	data->d_buf = xmalloc(i * j);
	data->d_size = i * j;
	memset(data->d_buf, 0, i);
	data->d_align = gelf_fsize(aelf, ELF_T_ADDR, 1, aeh.e_version);
	data->d_type = ELF_T_SYM;
	data->d_version = aeh.e_version;
	/*
	 * Copy locals
	 */
	for (i = j = 1; i < anlocs; i++) {
		if (alocals[i].shndx) {
			use_escapes = 1;
		}
		if (!gelf_update_sym(data, j++, &alocals[i].sym)) {
			elf_error("gelf_update_sym");
			return -1;
		}
	}
	shdr.sh_info = j;
	/*
	 * Copy globals
	 */
	for (i = 1; i < anglobs; i++) {
		if (aglobals[i].shndx) {
			use_escapes = 1;
		}
		if (!gelf_update_sym(data, j++, &aglobals[i].sym)) {
			elf_error("gelf_update_sym");
			return -1;
		}
	}
	/*
	 * Update section header
	 */
	shdr.sh_name = ld_add_string(&ashstrtab, ".symtab");
	shdr.sh_type = SHT_SYMTAB;
	shdr.sh_link = elf_ndxscn(strs);
	shdr.sh_entsize = gelf_fsize(aelf, ELF_T_SYM, 1, aeh.e_version);
	if (!gelf_update_shdr(scn, &shdr)) {
		elf_error("gelf_update_shdr");
		return -1;
	}
	index = elf_ndxscn(scn);	/* remember the symbol table index */
	/*
	 * Update sh_link of relocation sections
	 */
	for (seg = segments; seg; seg = seg->next) {
		for (sect = seg->scns; sect; sect = sect->next) {
			if (sect->shdr.sh_type == SHT_REL
			 || sect->shdr.sh_type == SHT_RELA) {
				if (!gelf_getshdr(sect->scn, &sect->shdr)) {
					elf_error("gelf_getshdr");
					return -1;
				}
				assert(sect->shdr.sh_link == 0);
				sect->shdr.sh_link = index;
				if (!gelf_update_shdr(sect->scn, &sect->shdr)) {
					elf_error("gelf_update_shdr");
					return -1;
				}
			}
		}
	}
	/*
	 * If there are less than SHN_LORESERVE sections, we're done
	 */
	if (!use_escapes) {
		return 0;
	}
	/*
	 * Write .symtab_shndx
	 */
	if (!gelf_getshdr(scn = elf_newscn(aelf), &shdr)) {
		elf_error("gelf_getshdr");
		return -1;
	}
	if (!(data = elf_newdata(scn))) {
		elf_error("elf_newdata");
		return -1;
	}
	data->d_size = j * sizeof(GElf_Word);
	data->d_buf = xmalloc(data->d_size);
	data->d_type = ELF_T_WORD;
	data->d_align = gelf_fsize(aelf, ELF_T_WORD, 1, aeh.e_version);
	data->d_version = aeh.e_version;
	p = data->d_buf;
	p[0] = 0;
	for (i = j = 1; i < anlocs; i++) {
		p[j++] = alocals[i].shndx;
	}
	for (i = 1; i < anglobs; i++) {
		p[j++] = aglobals[i].shndx;
	}
	shdr.sh_name = ld_add_string(&ashstrtab, ".symtab_shndx");
	shdr.sh_type = SHT_SYMTAB_SHNDX;
	shdr.sh_link = index;
	shdr.sh_entsize = data->d_align;
	if (!gelf_update_shdr(scn, &shdr)) {
		elf_error("gelf_update_shdr");
		return -1;
	}
	return 0;
}

int
ld_output_shstrtab(void) {
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Shdr shdr;
	size_t index;

	/*
	 * Create section
	 */
	if (!gelf_getshdr(scn = elf_newscn(aelf), &shdr)) {
		elf_error("gelf_getshdr");
		return -1;
	}
	if (!(data = elf_newdata(scn))) {
		elf_error("elf_newdata");
		return -1;
	}
	shdr.sh_name = ld_add_string(&ashstrtab, ".shstrtab");
	shdr.sh_type = SHT_STRTAB;
	shdr.sh_flags = 0;
	if (!gelf_update_shdr(scn, &shdr)) {
		elf_error("gelf_update_shdr");
		return -1;
	}
	*data = ashstrtab;	/* MUST come after `ld_add_string()' above */

	/*
	 * Set e_shstrndx
	 */
	index = elf_ndxscn(scn);
	if (index >= SHN_LORESERVE) {
		if (!gelf_getshdr(scn = elf_getscn(aelf, 0), &shdr)) {
			elf_error("gelf_getshdr");
			return -1;
		}
		shdr.sh_link = index;
		index = SHN_XINDEX;
		if (!gelf_update_shdr(scn, &shdr)) {
			elf_error("gelf_update_shdr");
			return -1;
		}
	}
	aeh.e_shstrndx = index;
	if (!gelf_update_ehdr(aelf, &aeh)) {
		elf_error("gelf_update_ehdr");
		return -1;
	}
	return 0;
}

int
ld_output_phdrs(void) {
	GElf_Addr addr = target->load_address;
	GElf_Addr psiz = target->page_size;
	GElf_Xword flen;
	GElf_Xword max;
	GElf_Xword mlen;
	GElf_Xword off;
	size_t index;
	struct outscn *sect;
	struct segment *seg;

	/* XXX: no support for dynamic linking yet */
	assert(ld_target_type == ET_EXEC);
	/*
	 * Create program headers
	 */
	index = 0;
	for (seg = segments; seg; seg = seg->next) {
		if (seg->phdr.p_type != PT_NULL && seg->scns != NULL) {
			index++;
		}
	}
	if (index == 0) {
		return 0;
	}
	if (!gelf_newphdr(aelf, index)) {
		elf_error("gelf_newphdr");
		return -1;
	}
	/*
	 * Calculate section layout
	 */
	if (elf_update(aelf, ELF_C_NULL) == (off_t)-1) {
		elf_error("elf_update");
		return -1;
	}
	/*
	 * Read back layouted section headers
	 */
	for (seg = segments; seg; seg = seg->next) {
		for (sect = seg->scns; sect; sect = sect->next) {
			if (!gelf_getshdr(sect->scn, &sect->shdr)) {
				elf_error("gelf_getshdr");
				return -1;
			}
		}
	}
	/*
	 * Calculate segment offsets and sizes
	 */
	/* XXX: ignores atttributes given in mapfiles */
	index = 0;
	for (seg = segments; seg; seg = seg->next) {
		if (seg->phdr.p_type == PT_NULL || seg->scns == NULL) {
			continue;
		}
		off = flen = mlen = 0;
		for (sect = seg->scns; sect; sect = sect->next) {
			if (off == 0) {
				if (index /* || (seg->phdr.p_flags & PF_NOHDRS) */) {
					off = sect->shdr.sh_offset;
				}
			}
			if (mlen == 0) {
				mlen = flen = sect->shdr.sh_offset;
			}
			assert(sect->shdr.sh_offset);
			assert(off <= sect->shdr.sh_offset);
			assert(flen <= sect->shdr.sh_offset);
			max = sect->shdr.sh_offset;
			if (sect->shdr.sh_type == SHT_NOBITS) {
				/*
				 * Can't use sh_offset alone because there
				 * might be multiple NOBITS sections at
				 * the same offset.
				 */
				if (max < mlen) {
					max = mlen;
					if (sect->shdr.sh_addralign > 1) {
						max += sect->shdr.sh_addralign - 1;
						max -= max % sect->shdr.sh_addralign;
					}
				}
			}
			else if (flen != mlen) {
				fatal("mapping error: NOBITS section not at end of segment");
			}
			sect->shdr.sh_addr = max - off;	/* offset in segment */
			max += sect->shdr.sh_size;
			if (mlen < max) {
				mlen = max;
			}
			if (sect->shdr.sh_type != SHT_NOBITS) {
				flen = mlen;
			}
		}
		/* XXX: use default defaults :) */
		seg->phdr.p_offset = off;
		seg->phdr.p_filesz = flen - off;
		seg->phdr.p_memsz = mlen - off;
		index++;
	}
	/*
	 * Calculate section addresses
	 */
	if (ld_target_type == ET_DYN) {
		addr = 0;
	}
	for (seg = segments; seg; seg = seg->next) {
		if (seg->phdr.p_type == PT_NULL || seg->scns == NULL) {
			continue;
		}
		if (seg->phdr.p_type == PT_LOAD
		 || seg->phdr.p_type == PT_NOTE) {
			seg->phdr.p_align = psiz;
			if (addr % psiz) {
				addr += psiz - addr % psiz;
			}
			addr += seg->phdr.p_offset % psiz;	/* segment addr */
		}
		seg->phdr.p_vaddr = addr;
		seg->phdr.p_paddr = 0;
		for (sect = seg->scns; sect; sect = sect->next) {
			sect->shdr.sh_addr += addr;
			if (!gelf_update_shdr(sect->scn, &sect->shdr)) {
				elf_error("gelf_update_shdr");
				return -1;
			}
		}
		addr += seg->phdr.p_memsz;	/* XXX: p_filesz? */
	}
	/*
	 * Write program headers
	 */
	index = 0;
	for (seg = segments; seg; seg = seg->next) {
		if (seg->phdr.p_type == PT_NULL || seg->scns == NULL) {
			continue;
		}
		debug("phdr[%u]:", (unsigned)index);
		debug("* p_type   = %#Lx", (GElf_Xword)seg->phdr.p_type);
		debug("* p_offset = %#Lx", (GElf_Xword)seg->phdr.p_offset);
		debug("* p_vaddr  = %#Lx", (GElf_Xword)seg->phdr.p_vaddr);
		debug("* p_paddr  = %#Lx", (GElf_Xword)seg->phdr.p_paddr);
		debug("* p_filesz = %#Lx", (GElf_Xword)seg->phdr.p_filesz);
		debug("* p_memsz  = %#Lx", (GElf_Xword)seg->phdr.p_memsz);
		debug("* p_flags  = %#Lx", (GElf_Xword)seg->phdr.p_flags);
		debug("* p_align  = %#Lx", (GElf_Xword)seg->phdr.p_align);
		debug("image ends at %#Lx",
			(GElf_Xword)seg->phdr.p_vaddr + seg->phdr.p_memsz);
		if (!gelf_update_phdr(aelf, index, &seg->phdr)) {
			elf_error("gelf_update_phdr");
			return -1;
		}
		index++;
	}
	return 0;
}

int
ld_output(const char *entry_point) {
	const char *s;
	size_t i;

	if (ld_target_type != ET_REL) {
		/*
		 * Transform common symbols
		 */
		ld_transform_common_symbols();
		/*
		 * Generate _etext, _edata and _end symbols
		 */
		if (ld_define_internal_symbols()) {
			return -1;
		}
	}
	if (ld_undefined_symbols()) {
		return -1;
	}
	/*
	 * Copy data sections to ELF output file
	 */
	if (ld_output_sections()) {
		return -1;
	}
	/*
	 * Prepare symbol tables
	 */
	for (i = 1; i < anlocs; i++) {
		ld_symbol_index(&alocals[i]);
	}
	for (i = 1; i < anglobs; i++) {
		ld_symbol_index(&aglobals[i]);
	}
	/*
	 * Process relocations
	 */
	if (ld_target_type == ET_REL) {
		/*
		 * Copy relocations to output
		 */
		if (ld_output_relocs(0)) {
			return -1;
		}
	}
	else {
		/*
		 * Create program headers first
		 */
		if (ld_output_phdrs()) {
			return -1;
		}
		/*
		 * Calculate symbol values
		 */
		for (i = 1; i < anlocs; i++) {
			ld_symbol_addr(&alocals[i]);
		}
		for (i = 1; i < anglobs; i++) {
			ld_symbol_addr(&aglobals[i]);
		}
		/*
		 * Apply relocations
		 */
		if (ld_output_relocs(1)) {
			return -1;
		}
	}
	/*
	 * Set entry point
	 */
	if (ld_target_type != ET_REL) {
		if ((s = entry_point)) {
			i = ld_find_symbol(s, elf_hash(s));
		}
		else if (ld_target_type == ET_EXEC) {
			s = "_start";
			i = ld_find_symbol(s, elf_hash(s));
			if (i == STN_UNDEF) {
				s = "main";
				i = ld_find_symbol(s, elf_hash(s));
			}
		}
		if (s) {
			if (i != STN_UNDEF) {
				aeh.e_entry = aglobals[i].sym.st_value;
				debug("entry point is `%s' (0x%Lx)", s,
					(unsigned long long)aeh.e_entry);
			}
			else {
				aeh.e_entry = text_segment->scns->shdr.sh_addr;
				warn("entry point `%s' not found; using 0x%Lx", s,
					(unsigned long long)aeh.e_entry);
			}
		}
	}
	/*
	 * Process symbol table
	 */
	if (!(ld_target_type == ET_EXEC && ld_opt_strip)) {
		if (ld_output_symtab()) {
			return -1;
		}
	}
	/*
	 * Add section name table
	 */
	if (ld_output_shstrtab()) {
		return -1;
	}
	if (errors) {
		return -1;
	}
	return 0;
}

int
ld_output_relocs(int apply) {
	int (*func)(struct target*, struct outscn*);
	struct outscn *sect;
	struct segment *seg;

	assert(target);
	assert(target->name);
	for (seg = segments; seg; seg = seg->next) {
		for (sect = seg->scns; sect;sect = sect->next) {
			assert(sect->scn);
			if (sect->nrel) {
				func = apply ? target->apply_rel : target->copy_rel;
				if (!func) {
					error("target `%s' does not support .rel sections",
						target->name);
					return -1;
				}
				else if (func(target, sect)) {
					return -1;
				}
			}
			if (sect->nrela) {
				func = apply ? target->apply_rela : target->copy_rela;
				if (!func) {
					error("target `%s' does not support .rela sections",
						target->name);
					return -1;
				}
				else if (func(target, sect)) {
					return -1;
				}
			}
		}
	}
	return 0;
}
