/*
 * ldmisc.c - miscelleanous functions
 * Copyright (C) 1995 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ldmisc.c,v 1.2 2003/02/07 03:19:24 michael Exp $";

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include <ld/ldmisc.h>

int errors = 0;

static char *progname = NULL;

void
setprogname(const char *newname) {
	progname = xrealloc(progname, strlen(newname) + 1);
	strcpy(progname, newname);
}

void
debug(const char *fmt, ...) {
	va_list ap;

	if (!getenv("LD_DEBUG")) {
		return;
	}
	va_start(ap, fmt);
	fprintf(stderr, "debug: ");
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
	va_end(ap);
}

static void
verror(const char *where, const char *what, const char *fmt, va_list ap) {
	fprintf(stderr, "%s: %s", where, what);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
}

void
warn(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(progname, "warning: ", fmt, ap);
	va_end(ap);
}

void
file_warn(const char *fn, const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(fn, "warning: ", fmt, ap);
	va_end(ap);
}

void
error(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(progname, "", fmt, ap);
	va_end(ap);
	errors++;
}

void
file_error(const char *fn, const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(fn, "", fmt, ap);
	va_end(ap);
	errors++;
}

void
fatal(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(progname, "fatal error: ", fmt, ap);
	va_end(ap);
	abort();
}

void
file_fatal(const char *fn, const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	verror(fn, "fatal error: ", fmt, ap);
	va_end(ap);
	abort();
}

static void
oom(void) {
	error("out of memory");
	exit(1);
}

void*
xmalloc(size_t n) {
	void *p;

	if (!(p = malloc(n ? n : 1))) {
		oom();
	}
	return p;
}

void*
xrealloc(void *p, size_t n) {
	if (p == NULL) {
		return xmalloc(n);
	}
	if (!(p = realloc(p, n ? n : 1))) {
		oom();
	}
	return p;
}

void
xfree(void *p) {
	if (p) {
		free(p);
	}
}

char*
xstrdup(const char *s) {
	char *p;

	if (!s) {
		s = "";
	}
	p = xmalloc(strlen(s) + 1);
	strcpy(p, s);
	return p;
}
