/*
 * generic.c -- generic routines for all targets
 * Copyright (C) 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: generic.c,v 1.3 2003/02/08 13:07:01 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <gelf.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

int
generic_copy_rela(struct target *tgt, struct outscn *sect) {
	static const char prefix[] = ".rela";
	Elf_Data *data;
	GElf_Rela *rela;
	GElf_Shdr shdr;
	GElf_Xword sym;
	char *name;
	size_t i;
	struct outscn *rsect;

	/*
	 * Create relocation section
	 */
	name = xmalloc(sizeof(prefix) + strlen(sect->name));
	strcpy(name, prefix);
	strcat(name, sect->name);
	shdr.sh_type = SHT_RELA;
	shdr.sh_flags = 0;
	rsect = ld_output_section("", name, &shdr);
	if (!gelf_getshdr(rsect->scn = elf_newscn(aelf), &shdr)) {
		error("gelf_getshdr: %s", elf_errmsg(-1));
		return -1;
	}
	shdr.sh_name = ld_add_string(&ashstrtab, name);
	shdr.sh_type = rsect->shdr.sh_type;
	shdr.sh_flags = rsect->shdr.sh_flags;
	shdr.sh_addr = 0;
	shdr.sh_link = 0;	/* XXX: replace with symbol table index later */
	shdr.sh_info = elf_ndxscn(sect->scn);
	assert(shdr.sh_info);
	shdr.sh_entsize = gelf_fsize(aelf, ELF_T_RELA, 1, aeh.e_version);
	if (!gelf_update_shdr(rsect->scn, &shdr)) {
		error("gelf_update_shdr: %s", elf_errmsg(-1));
		return -1;
	}
	rsect->shdr = shdr;
	/*
	 * Create data buffer
	 */
	if (!(data = elf_newdata(rsect->scn))) {
		error("elf_newdata: %s", elf_errmsg(-1));
		return -1;
	}
	data->d_size = sect->nrela * tgt->rela_size;
	data->d_buf = xmalloc(data->d_size);
	data->d_type = ELF_T_RELA;
	data->d_align = gelf_fsize(aelf, ELF_T_ADDR, 1, aeh.e_version);
	data->d_version = aeh.e_version;
	/*
	 * Copy reloc entries one by one
	 */
	for (i = 0; i < sect->nrela; i++) {
		rela = &sect->rela[i];
		sym = GELF_R_SYM(rela->r_info);
		if (sym && sym >= anlocs) {
			/*
			 * Calculate global symbol index
			 */
			sym = SYMBOLS_MAX - sym;
			assert(0 < sym && sym < anglobs);
			if (anlocs) {
				sym += anlocs - 1;
			}
			rela->r_info = GELF_R_INFO(sym, GELF_R_TYPE(rela->r_info));
		}
		assert(rela->r_offset < sect->shdr.sh_size);
		if (!gelf_update_rela(data, i, rela)) {
			error("gelf_update_rela: %s", elf_errmsg(-1));
			return -1;
		}
	}
	return 0;
}

int
generic_copy_rel(struct target *tgt, struct outscn *sect) {
	static const char prefix[] = ".rel";
	Elf_Data *data;
	GElf_Rel *rel;
	GElf_Shdr shdr;
	GElf_Xword sym;
	char *name;
	size_t i;
	struct outscn *rsect;

	/*
	 * Create relocation section
	 */
	name = xmalloc(sizeof(prefix) + strlen(sect->name));
	strcpy(name, prefix);
	strcat(name, sect->name);
	shdr.sh_type = SHT_REL;
	shdr.sh_flags = 0;
	rsect = ld_output_section("", name, &shdr);
	if (!gelf_getshdr(rsect->scn = elf_newscn(aelf), &shdr)) {
		error("gelf_getshdr: %s", elf_errmsg(-1));
		return -1;
	}
	shdr.sh_name = ld_add_string(&ashstrtab, name);
	shdr.sh_type = rsect->shdr.sh_type;
	shdr.sh_flags = rsect->shdr.sh_flags;
	shdr.sh_addr = 0;
	shdr.sh_link = 0;	/* XXX: replace with symbol table index later */
	shdr.sh_info = elf_ndxscn(sect->scn);
	assert(shdr.sh_info);
	shdr.sh_entsize = gelf_fsize(aelf, ELF_T_REL, 1, aeh.e_version);
	if (!gelf_update_shdr(rsect->scn, &shdr)) {
		error("gelf_update_shdr: %s", elf_errmsg(-1));
		return -1;
	}
	rsect->shdr = shdr;
	/*
	 * Create data buffer
	 */
	if (!(data = elf_newdata(rsect->scn))) {
		error("elf_newdata: %s", elf_errmsg(-1));
		return -1;
	}
	data->d_size = sect->nrel * tgt->rel_size;
	data->d_buf = xmalloc(data->d_size);
	data->d_type = ELF_T_REL;
	data->d_align = gelf_fsize(aelf, ELF_T_ADDR, 1, aeh.e_version);
	data->d_version = aeh.e_version;
	/*
	 * Copy reloc entries one by one
	 */
	for (i = 0; i < sect->nrel; i++) {
		rel = &sect->rel[i];
		sym = GELF_R_SYM(rel->r_info);
		if (sym && sym >= anlocs) {
			/*
			 * Calculate global symbol index
			 */
			sym = SYMBOLS_MAX - sym;
			assert(0 < sym && sym < anglobs);
			if (anlocs) {
				sym += anlocs - 1;
			}
			rel->r_info = GELF_R_INFO(sym, GELF_R_TYPE(rel->r_info));
		}
		assert(rel->r_offset < sect->shdr.sh_size);
		if (!gelf_update_rel(data, i, rel)) {
			error("gelf_update_rel: %s", elf_errmsg(-1));
			return -1;
		}
	}
	return 0;
}

GElf_Addr
ld_get_data(const unsigned char *place, size_t size, int be) {
	GElf_Addr value = 0;
	size_t i;

	if (be) {
		for (i = 0; i < size; i++) {
			value = (value << 8) | place[i];
		}
	}
	else {
		for (i = size; i-- > 0; ) {
			value = (value << 8) | place[i];
		}
	}
	return value;
}

void
ld_put_data(unsigned char *place, size_t size, GElf_Addr value, int be) {
	size_t i;

	if (be) {
		for (i = size; i-- > 0; ) {
			place[i] = value;
			value >>= 8;
		}
	}
	else {
		for (i = 0; i < size; i++) {
			place[i] = value;
			value >>= 8;
		}
	}
}
