/*
 * scnxlate.c - adjust section indices after deleting.
 * Copyright (C) 1995 - 2001 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <elftools.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: scnxlate.c,v 1.1 2002/12/27 18:17:36 michael Exp $";
#endif /* lint */

static int
et_symxlate(Elf *elf, Elf_Data *data, int *xlate, size_t n) {
	GElf_Sym sym;
	size_t i;
	size_t nsyms;

#if HAVE_GELF_MSIZE
	nsyms = gelf_msize(elf, ELF_T_SYM, 1, EV_CURRENT);
	if (nsyms == 0 || data->d_size % nsyms) {
		return -1;
	}
#else /* HAVE_GELF_MSIZE */
	switch (gelf_getclass(elf)) {
		case ELFCLASS32: nsyms = sizeof(Elf32_Sym); break;
		case ELFCLASS64: nsyms = sizeof(Elf64_Sym); break;
		default: return -1;
	}
	if (data->d_size % nsyms) {
		return -1;
	}
#endif /* HAVE_GELF_MSIZE */
	nsyms = data->d_size / nsyms;
	for (i = 0; i < nsyms; i++) {
		if (!gelf_getsym(data, i, &sym)) {
			return -1;
		}
		if (sym.st_shndx == SHN_UNDEF || sym.st_shndx >= n) {
			continue;
		}
		sym.st_shndx = xlate[sym.st_shndx];
		if (sym.st_shndx == SHN_UNDEF) {
			/*
			 * symbol points to deleted section
			 */
			if (sym.st_info != GELF_ST_INFO(STB_LOCAL, STT_SECTION)) {
				return -1;
			}
			/*
			 * quick & dirty: convert section symbols
			 */
			sym.st_info = GELF_ST_INFO(STB_LOCAL, STT_NOTYPE);
			sym.st_shndx = SHN_ABS;
		}
		if (!gelf_update_sym(data, i, &sym)) {
			return -1;
		}
	}
	return 0;
}

int
et_scnxlate(Elf *elf, int *xlate, size_t n) {
	GElf_Shdr sh;
	Elf_Scn *scn;
	Elf_Data *data;

	elf_errno();
	scn = NULL;
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			return -1;
		}
		switch (sh.sh_type) {
			case SHT_NULL:
				continue;
			case SHT_REL:
			case SHT_RELA:
#ifdef SHT_SUNW_syminfo
			case SHT_SUNW_syminfo:
#endif /* SHT_SUNW_syminfo */
				if (sh.sh_info < n) {
					sh.sh_info = xlate[sh.sh_info];
				}
				/* flow through */
			case SHT_DYNAMIC:
			case SHT_HASH:
#ifdef SHT_SUNW_move
			case SHT_SUNW_move:
#endif /* SHT_SUNW_move */
#ifdef SHT_SUNW_verdef
			case SHT_SUNW_verdef:
#endif/* SHT_SUNW_verdef */
#ifdef SHT_SUNW_verneed
			case SHT_SUNW_verneed:
#endif/* SHT_SUNW_verneed */
#ifdef SHT_SUNW_versym
			case SHT_SUNW_versym:
#endif /* SHT_SUNW_versym */
				if (sh.sh_link < n) {
					sh.sh_link = xlate[sh.sh_link];
				}
				break;
			case SHT_DYNSYM:
			case SHT_SYMTAB:
				if (sh.sh_link < n) {
					sh.sh_link = xlate[sh.sh_link];
				}
				/*
				 * change section indices in symbol table
				 */
				elf_errno();
				data = NULL;
				while ((data = elf_getdata(scn, data))) {
					if (et_symxlate(elf, data, xlate, n)) {
						return -1;
					}
				}
				if (elf_errmsg(0)) {
					return -1;
				}
				break;
			case SHT_PROGBITS:	/* used by debug sections */
				if (sh.sh_link != SHN_UNDEF && sh.sh_link < n) {
					sh.sh_link = xlate[sh.sh_link];
				}
				break;
			default:
				continue;
		}
		if (!gelf_update_shdr(scn, &sh)) {
			return -1;
		}
	}
	if (elf_errmsg(0)) {
		return -1;
	}
	return 0;
}
