/*
 * nm.c - SVR4 style nm(1).
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <ctype.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: nm.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static int opt_A = 0;	/* prepend pathname */
static int opt_D = 0;	/* display .dynsym section */
static int opt_g = 0;	/* global symbols only */
static int opt_h = 0;	/* skip headers */
static int opt_l = 0;	/* mark weak symbols */
static int opt_n = 0;	/* sort by name */
static int opt_p = 0;	/* BSD format */
static int opt_P = 0;	/* portable format */
static int opt_r = 0;	/* prepend filename */
static int opt_s = 0;	/* symbolic section names */
static int opt_u = 0;	/* undefined symbols only */
static int opt_v = 0;	/* sort by value */

static int fmt = -1;		/* 0 = dec, 1 = oct, 2 = hex */

#if SIZEOF_LONG_LONG == 8 && SIZEOF_LONG != 8
# define I64	long long
# define U64	unsigned long long
# define IF64	"lld"
# define UF64	"llu"
# define OF64	"llo"
# define XF64	"llx"
#else
# define I64	long
# define U64	unsigned long
# define IF64	"ld"
# define UF64	"lu"
# define OF64	"lo"
# define XF64	"lx"
#endif

static const char *const formats[5][3] = {
	/* heading format */
	{
		"%-5s %-10s %-10s %7s %6s %5s %6s %s\n",
		"%-5s %-11s %-11s %7s %6s %5s %6s %s\n",
		"%-5s %-8s %-8s %7s %6s %5s %6s %s\n",
	},
	/* default format */
	{
		"[%03u] %010" UF64 " %010" UF64 " %7s %6s %5u %6s %s\n",
		"[%03u] %011" OF64 " %011" OF64 " %7s %6s %5u %6s %s\n",
		"[%03u] %08"  XF64 " %08"  XF64 " %7s %6s %5u %6s %s\n",
	},
	/* bsd-style heading/undefined */
	{
		"%-10s %c%c%s\n",
		"%-11s %c%c%s\n",
		"%-8s %c%c%s\n",
	},
	/* bsd-style format */
	{
		"%010" UF64 " %c%c%s\n",
		"%011" OF64 " %c%c%s\n",
		"%08"  XF64 " %c%c%s\n",
	},
	/* portable format */
	{
		/* name, type, value, size */
		"%s %c %" UF64 " %" UF64 "\n",
		"%s %c %" OF64 " %" OF64 "\n",
		"%s %c %" XF64 " %" XF64 "\n",
	},
};

static const char*
section_name(Elf *elf, size_t idx, int symbolic) {
	static char buf[32];
	GElf_Shdr sh;
	GElf_Addr shstrndx;
	const char *s;

	switch (idx) {
		case SHN_ABS:    return "ABS";
		case SHN_COMMON: return "COMMON";
		case SHN_UNDEF:  return "UNDEF";
	}
	if (symbolic
	 && (shstrndx = et_get_shstrndx(elf))
	 && gelf_getshdr(elf_getscn(elf, idx), &sh)
	 && (s = elf_strptr(elf, shstrndx, sh.sh_name))) {
		return s;
	}
	sprintf(buf, "%d", (int)idx);
	return buf;
}

static const char*
symbol_type(GElf_Sym *sym) {
	static char buf[32];

	switch (GELF_ST_TYPE(sym->st_info)) {
		case STT_NOTYPE:  return "NOTYPE";
		case STT_OBJECT:  return "OBJECT";
		case STT_FUNC:    return "FUNC";
		case STT_SECTION: return "SECTION";
		case STT_FILE:    return "FILE";
	}
	sprintf(buf, "%u", (unsigned)GELF_ST_TYPE(sym->st_info));
	return buf;
}

static const char*
symbol_binding(GElf_Sym *sym) {
	static char buf[32];

	switch (GELF_ST_BIND(sym->st_info)) {
		case STB_LOCAL:  return "LOCAL";
		case STB_GLOBAL: return "GLOBAL";
		case STB_WEAK:   return "WEAK";
	}
	sprintf(buf, "%u", (unsigned)GELF_ST_BIND(sym->st_info));
	return buf;
}

static unsigned char
bsd_type(GElf_Sym *sym) {
	unsigned char c = '?';

	switch (sym->st_shndx) {
		case SHN_ABS:    c = 'A'; break;
		case SHN_COMMON: c = 'B'; break;
		case SHN_UNDEF:  c = 'U'; break;
		default:
			switch (GELF_ST_TYPE(sym->st_info)) {
				case STT_NOTYPE:  c = 'N'; break;
				case STT_OBJECT:  c = 'D'; break;
				case STT_FUNC:    c = 'T'; break;
				case STT_SECTION: c = 'S'; break;
				case STT_FILE:    c = 'F'; break;
			}
	}
	if (GELF_ST_BIND(sym->st_info) == STB_LOCAL) {
		c = tolower(c);
	}
	return c;
}

static void
print_symbol(const char *fname, Elf *elf, size_t i,
			 const char *name, GElf_Sym *sym) {
	const char *s;

	if (opt_A || opt_r) {
		if (opt_r && (s = strrchr(fname, '/'))) {
			s++;
		}
		else {
			s = fname;
		}
		printf("%s: ", s);
	}
	if (opt_p) {
		int weakind = ' ';
		int c;

		c = bsd_type(sym);
		if (opt_l && GELF_ST_BIND(sym->st_info) == STB_WEAK) {
			weakind = '*';
		}
		if (sym->st_shndx == SHN_UNDEF) {
			printf(formats[2][fmt], "", c, weakind, name);
		}
		else {
			printf(formats[3][fmt], (U64)sym->st_value, c, weakind, name);
		}
	}
	else if (opt_P) {
		printf(formats[4][fmt],
			name, bsd_type(sym), (U64)sym->st_value, (U64)sym->st_size);
	}
	else {
		printf(formats[1][fmt],
			i, (U64)sym->st_value, (U64)sym->st_size, symbol_type(sym),
			symbol_binding(sym), (unsigned)sym->st_other,
			section_name(elf, sym->st_shndx, opt_s), name);
	}
}

struct sym_internal {
	const char *name;
	GElf_Sym sym;
};

static int
valcmp(const void *aa, const void *bb) {
	const struct sym_internal *a = aa;
	const struct sym_internal *b = bb;

	if (GELF_ST_BIND(a->sym.st_info) == STB_LOCAL) {
		if (GELF_ST_BIND(b->sym.st_info) != STB_LOCAL) {
			return -1;
		}
	}
	else if (GELF_ST_BIND(b->sym.st_info) == STB_LOCAL) {
		return 1;
	}
	else if (a->sym.st_value < b->sym.st_value) {
		return -1;
	}
	else if (a->sym.st_value > b->sym.st_value) {
		return 1;
	}
	/* maintain current order */
	return a < b ? -1 : 1;
}

static int
namcmp(const void *aa, const void *bb) {
	const struct sym_internal *a = aa;
	const struct sym_internal *b = bb;
	int i;

	if (GELF_ST_BIND(a->sym.st_info) == STB_LOCAL) {
		if (GELF_ST_BIND(b->sym.st_info) != STB_LOCAL) {
			return -1;
		}
	}
	else if (GELF_ST_BIND(b->sym.st_info) == STB_LOCAL) {
		return 1;
	}
	else if ((i = strcmp(a->name, b->name)) != 0) {
		return i;
	}
	/* maintain current order */
	return a < b ? -1 : 1;
}

static void
do_symbols(const char *fname, Elf *elf, GElf_Shdr *sh, Elf_Data *data) {
	struct sym_internal *syms;
	size_t nsyms;
	size_t i;

	nsyms = sh->sh_size / sh->sh_entsize;
	if (!opt_h && !opt_P) {
		if (opt_p) {
			printf(formats[2][fmt], "Value", 'T', ' ', "Name");
		}
		else {
			printf(formats[0][fmt], "Index", "Value", "Size",
				"Type", "Bind", "Other", "Shndx", "Name");
		}
	}
	syms = xmalloc(nsyms * sizeof(*syms));
	for (i = 1; i < nsyms; ++i) {
		if (!gelf_getsym(data, i, &syms[i].sym)) {
			file_error(fname, "gelf_getsym: %s", elf_errmsg(-1));
			return;
		}
		syms[i].name = elf_strptr(elf, sh->sh_link, syms[i].sym.st_name);
		if (!syms[i].name) {
			syms[i].name = "";
		}
	}
	if (opt_v) {
		/* sort by value */
		qsort(syms + 1, nsyms - 1, sizeof(*syms), valcmp);
	}
	else if (opt_n) {
		/* sort by name */
		qsort(syms + 1, nsyms - 1, sizeof(*syms), namcmp);
	}
	for (i = 1; i < nsyms; ++i) {
		if (opt_g && GELF_ST_BIND(syms[i].sym.st_info) == STB_LOCAL) {
			continue;
		}
		if (opt_u && syms[i].sym.st_shndx != SHN_UNDEF) {
			continue;
		}
		/* XXX: really do this? */
		if ((opt_p || opt_P) && !syms[i].name[0]) {
			continue;
		}
		print_symbol(fname, elf, i, syms[i].name, &syms[i].sym);
	}
	xfree(syms);
}

static int
process_elf(const char *name, Elf *elf, int ofd) {
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	int err;

	assert(ofd == -1);
	if (!gelf_getehdr(elf, &eh)) {
		file_error(name, "gelf_getehdr: %s", elf_errmsg(-1));
		return 1;
	}
	scn = NULL;
	elf_errno();
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
			return 1;
		}
		if (sh.sh_type == (opt_D ? SHT_DYNSYM : SHT_SYMTAB)) {
			break;
		}
	}
	if ((err = elf_errno())) {
		file_error(name, "elf_nextscn: %s", elf_errmsg(err));
		return 1;
	}
	if (!scn) {
		printf("%s: no symbols\n", name);
		return 0;
	}
	if (!opt_A && !opt_r) {
		printf("%s:\n", name);
	}
	data = NULL;
	while ((data = elf_getdata(scn, data))) {
		do_symbols(name, elf, &sh, data);
	}
	if ((err = elf_errno())) {
		file_error(name, "elf_getdata: %s", elf_errmsg(err));
		return 1;
	}
	return 0;
}

static const char usage[] =
	"usage: %s [-ADhlnoPprsVvx] [-g|-u] [-t format] [file...]\n"
	"  -A        prepend pathname to each line\n"
	"  -D        show dynamic section\n"
	"  -g        print global symbols only\n"
	"  -h        do not display header\n"
	"  -l        mark weak symbols with `*'\n"
	"  -n        sort external symbols by name\n"
	"  -o        print values in octal\n"
	"  -p        BSD-style output format\n"
	"  -P        portable output format\n"
	"  -r        prepend filename to each line\n"
	"  -s        print section names, not indices\n"
	"  -t [dox]  print values in decimal/octal/hex\n"
	"  -u        print undefined symbols only\n"
	"  -V        display program version\n"
	"  -v        sort external symbols by value\n"
	"  -x        print values in hexadecimal\n"
	/*
	"  -R        prepend archive and object name to symbol name\n"
	"  -C        demangle C++ names\n"
	*/
	/*
	"  -e        print only external and static symbols (obsolete)\n"
	"  -f        produce full output (obsolete)\n"
	"  -T        truncate symbol name (obsolete)\n"
	*/
	"without any file arguments, a.out will be used\n";

int
main(int argc, char **argv) {
	char *progname = NULL;
	int opt_t = 0;
	int opt_V = 0;
	int c;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		*argv = ++progname;
	}
	else if (!(progname = *argv)) {
		progname = "nm";
	}
	setprogname(progname);

	while ((c = getopt(argc, argv, "ADghlnoPprst:uVvx")) != EOF) {
		switch (c) {
			case 't': if (!opt_t) opt_t = *optarg; break;
			case 'o': if (!opt_t) opt_t = c; break;
			case 'x': if (!opt_t) opt_t = c; break;
			case 'A': if (!opt_A && !opt_r) opt_A = 1; break;
			case 'r': if (!opt_A && !opt_r) opt_r = 1; break;
			case 'g': if (!opt_g && !opt_u) opt_g = 1; break;
			case 'u': if (!opt_g && !opt_u) opt_u = 1; break;
			case 'n': if (!opt_n && !opt_v) opt_n = 1; break;
			case 'v': if (!opt_n && !opt_v) opt_v = 1; break;
			case 'h': opt_h = 1; break;
			case 'l': opt_l = 1; break;
			case 'P': if (!opt_p && !opt_P) opt_P = 1; break;
			case 'p': if (!opt_p && !opt_P) opt_p = 1; break;
			case 's': opt_s = 1; break;
			case 'D': opt_D = 1; break;
			case 'V': opt_V = 1; break;
			case '?': fprintf(stderr, usage, progname); exit(c != 'h');
			default: break;
		}
	}
	switch (opt_t) {
		case 0: fmt = (opt_p || opt_P) ? 2 : 0; break;
		case 'd': fmt = 0; break;
		case 'o': fmt = 1; break;
		case 'x': fmt = 2; break;
		default:
			error("invalid option argument `%s'", optarg);
			fprintf(stderr, usage, progname);
			exit(1);
	}

	if (opt_V) {
		show_version("nm");
		if (optind == argc) {
			exit(0);
		}
	}

	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}

	c = 0;
	if (optind == argc) {
		c = et_read_file("a.out", process_elf);
	}
	else {
		while (optind < argc) {
			c |= et_read_file(argv[optind++], process_elf);
		}
	}
	exit(c ? 1 : 0);
}
