/*
 * awrite.c - write archive files
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <elftools.h>
#include <stdio.h>		/* for rename() */
#include <time.h>
#include <ar.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: awrite.c,v 1.2 2002/12/28 20:29:39 michael Exp $";
#endif /* lint */

struct archive_descriptor {
	size_t	*offsets;
	size_t	noffsets;
	char	*snames;
	size_t	nsnames;
	char	*fnames;
	size_t	nfnames;
	char	*bytes;
	size_t	nbytes;
};

static const size_t blocksize = 1024;

/*
 * Create a new archive descriptor
 */
ardesc_t*
ar_create(void) {
	ardesc_t *ap;

	ap = xmalloc(sizeof(ardesc_t));
	ap->offsets = NULL;
	ap->noffsets = 0;
	ap->snames = NULL;
	ap->nsnames = 0;
	ap->fnames = NULL;
	ap->nfnames = 0;
	ap->bytes = NULL;
	ap->nbytes = 0;
	return ap;
}

/*
 * Destroy an archive descriptor
 */
void
ar_destroy(ardesc_t *ap) {
	if (!ap) {
		return;
	}
	xfree(ap->offsets);
	xfree(ap->snames);
	xfree(ap->fnames);
	xfree(ap->bytes);
	xfree(ap);
}

/*
 * Add an entry to the archive symbol table
 * (offset counted from the beginning of the first regular member)
 */
void
ar_add_symbol(ardesc_t *ap, const char *name, size_t off) {
	size_t n, len = strlen(name) + 1;

	n = ap->nsnames + blocksize - 1, n -= n % blocksize;
	if (ap->nsnames + len > n) {
		n = ap->nsnames + len + blocksize - 1, n -= n % blocksize;
		ap->snames = xrealloc(ap->snames, n);
	}
	strcpy(ap->snames + ap->nsnames, name);
	ap->nsnames += len;
	if (ap->noffsets % 128u == 0) {
		n = ap->noffsets + 128u;
		ap->offsets = xrealloc(ap->offsets, n * sizeof(size_t));
	}
	ap->offsets[ap->noffsets++] = off;
}

/*
 * Add an entry to the archive string table
 * Return the string's offset into the table
 */
size_t
ar_add_fname(ardesc_t *ap, const char *name) {
	size_t n, len = strlen(name);

	n = ap->nfnames + blocksize - 1, n -= n % blocksize;
	if (ap->nfnames + len + 2 > n) {
		n = ap->nfnames + len + 2 + blocksize - 1, n -= n % blocksize;
		ap->fnames = xrealloc(ap->fnames, n);
	}
	memcpy(ap->fnames + ap->nfnames, name, len);
	n = ap->nfnames, ap->nfnames += len;
	ap->fnames[ap->nfnames++] = '/';
	ap->fnames[ap->nfnames++] = '\n';
	return n;
}

/*
 * Add data to the archive
 * Return offset (counted from 1st regular member)
 */
size_t
ar_add_bytes(ardesc_t *ap, const void *data, size_t len) {
	size_t n;

	n = ap->nbytes + blocksize - 1, n -= n % blocksize;
	if (ap->nbytes + len > n) {
		n = ap->nbytes + len + blocksize - 1, n -= n % blocksize;
		ap->bytes = xrealloc(ap->bytes, n);
	}
	memcpy(ap->bytes + ap->nbytes, data, len);
	n = ap->nbytes, ap->nbytes += len;
	return n;
}

/*
 * Align archive data
 */
void
ar_align_bytes(ardesc_t *ap) {
	assert(blocksize % 2u == 0);
	if (ap->nbytes % 2u != 0) {
		ap->bytes[ap->nbytes++] = '\n';
	}
}

/*
 * Fill in numeric archive member header field
 */
static int
setnum(char *dst, size_t len, unsigned long val, unsigned long base) {
	unsigned long v = val;
	size_t i = 1;

	while ((v /= base)) {
		i++;
	}
	if (i > len) {
		return 1;
	}
	while (i > 0) {
		dst[--i] = '0' + val % base;
		val /= base;
	}
	return 0;
}

/*
 * Add archive member header
 * Return it's offset (counted from 1st regular member)
 */
size_t
ar_add_header(ardesc_t *ap, const Elf_Arhdr *ah) {
	struct ar_hdr hdr;
	size_t len;
	size_t off;

	/*
	 * Prepare struct ar_hdr
	 */
	memset(&hdr, ' ', sizeof(hdr));
	len = strlen(ah->ar_name);
	if (len + 1 < sizeof(hdr.ar_name)) {
		memcpy(hdr.ar_name, ah->ar_name, len);
		hdr.ar_name[len] = '/';
	}
	else {
		hdr.ar_name[0] = '/';
		off = ar_add_fname(ap, ah->ar_name);
		setnum(&hdr.ar_name[1], sizeof(hdr.ar_name) - 1, off, 10);
	}
	setnum(hdr.ar_date, sizeof(hdr.ar_date), ah->ar_date, 10);
	setnum(hdr.ar_uid,  sizeof(hdr.ar_uid),  ah->ar_uid,  10);
	setnum(hdr.ar_gid,  sizeof(hdr.ar_gid),  ah->ar_gid,  10);
	setnum(hdr.ar_mode, sizeof(hdr.ar_mode), ah->ar_mode,  8);
	setnum(hdr.ar_size, sizeof(hdr.ar_size), ah->ar_size, 10);
	memcpy(hdr.ar_fmag, ARFMAG, sizeof(hdr.ar_fmag));
	/*
	 * Write header
	 */
	ar_align_bytes(ap);
	return ar_add_bytes(ap, &hdr, sizeof(hdr));
}

/*
 * Get .symtab or .dynsym section from an ELF file, if present
 */
static Elf_Scn*
get_elf_symbol_table(Elf *elf, GElf_Word type) {
	Elf_Scn *scn;
	Elf_Scn *symtab;
	GElf_Shdr sh;

	scn = symtab = NULL;
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			return NULL;
		}
		if (type != SHT_NULL) {
			if (sh.sh_type == type) {
				return scn;
			}
		}
		else if (sh.sh_type == SHT_SYMTAB) {
			return scn;
		}
		else if (sh.sh_type == SHT_DYNSYM) {
			symtab = scn;
		}
	}
	return symtab;
}

/*
 * Add a file to the archive
 * If it is an ELF file, process its symbol table
 */
int
ar_add_elf(ardesc_t *ap, Elf *elf, Elf_Arhdr ah) {
	Elf_Data *data;
	Elf_Scn *symtab;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	GElf_Sym sym;
	char *img;
	int err;
	size_t i;
	size_t len;
	size_t nsyms;
	size_t off;

	/*
	 * Get the file's raw image.
	 * This will work for *all* files, ELF or not.
	 */
	elf_errno();
	if (!(img = elf_rawfile(elf, &len))) {
		if ((err = elf_errno())) {
			file_error(ah.ar_name, "elf_rawfile: %s", elf_errmsg(err));
			return -1;
		}
		/* empty file */
		assert(len == 0);
	}
	ah.ar_size = len;
	/*
	 * Add archive member header.
	 * Keep its offset for the archive symbol table.
	 */
	off = ar_add_header(ap, &ah);
	assert(ap->nbytes % 2u == 0);
	/*
	 * Try to get the ELF header.
	 * If that doesn't work, don't worry.
	 */
	if (gelf_getehdr(elf, &eh)) {
		/*
		 * Load the symbol table
		 */
		/* XXX: use .dynsym if .symtab is not available? */
		elf_errno();
		symtab = get_elf_symbol_table(elf, SHT_SYMTAB);
		if (symtab != NULL) {
			if (!gelf_getshdr(symtab, &sh)) {
				file_error(ah.ar_name, "gelf_getshdr: %s", elf_errmsg(-1));
				return -1;
			}
			if (!(data = elf_getdata(symtab, NULL))) {
				file_error(ah.ar_name, "elf_getdata: %s", elf_errmsg(-1));
				return -1;
			}
			assert(!elf_getdata(symtab, data));
			assert(data->d_type == ELF_T_SYM);
			nsyms = gelf_fsize(elf, ELF_T_SYM, 1, eh.e_version);
			assert(nsyms);
			nsyms = sh.sh_size / nsyms;
			/*
			 * Copy all publically defined symbols
			 */
			for (i = sh.sh_info; i < nsyms; i++) {
				char *symname;

				if (!gelf_getsym(data, i, &sym)) {
					file_error(ah.ar_name, "gelf_getsym: %s", elf_errmsg(-1));
					return -1;
				}
				if (GELF_ST_BIND(sym.st_info) == STB_LOCAL) {
					file_error(ah.ar_name, "broken symbol table");
					return -1;
				}
				if (sym.st_shndx == SHN_UNDEF) {
					continue;
				}
				if (!(symname = elf_strptr(elf, sh.sh_link, sym.st_name))) {
					file_error(ah.ar_name, "elf_strptr: %s", elf_errmsg(-1));
					return -1;
				}
				ar_add_symbol(ap, symname, off);
			}
		}
		else if (elf_errno()) {
			file_error(ah.ar_name, "could not load ELF symbol table");
			return -1;
		}
	}
	/*
	 * Add the file's raw image to the archive
	 */
	if (img && len) {
		ar_add_bytes(ap, img, len);
	}
	ar_align_bytes(ap);
	return 0;
}

/*
 * Add a file to the archive
 */
int
ar_add_file(ardesc_t *ap, const char *mname, const char *fname) {
	Elf *elf;
	Elf_Arhdr ah;
	int err;
	int fd;
	struct stat st;

	err = 0;
	elf_errno();
	if ((fd = open(fname, O_RDONLY)) == -1) {
		file_error(fname, "open: %s", strerror(errno));
		err = 1;
	}
	else {
		if (fstat(fd, &st)) {
			file_error(fname, "stat: %s", strerror(errno));
			err = 1;
		}
		else if (!(elf = elf_begin(fd, ELF_C_READ, NULL))) {
			file_error(fname, "elf_begin: %s", elf_errmsg(-1));
			err = 1;
		}
		else {
			ah.ar_name = (char*)mname;
			ah.ar_date = st.st_mtime;
			ah.ar_uid = st.st_uid;
			ah.ar_gid = st.st_gid;
			ah.ar_mode = st.st_mode;
			ah.ar_size = st.st_size;
			err = ar_add_elf(ap, elf, ah);
			elf_end(elf);
		}
		close(fd);
	}
	return err != 0;
}

/*
 * Write the archive to disk
 */
int
ar_write(ardesc_t *ap, int fd) {
	struct ar_hdr hdr;

	/*
	 * Write archive header
	 */
	if (xwrite(fd, ARMAG, SARMAG)) {
		return -1;
	}
	/*
	 * Write archive symbol table, if any
	 */
	if (ap->noffsets && ap->nsnames) {
		unsigned char *buf, *p;
		size_t i, x, base, len;

		/*
		 * Align
		 */
		if (ap->nsnames % 2u != 0) {
			ap->snames[ap->nsnames++] = '\0';
		}
		/*
		 * Write symbol table header
		 */
		len = 4 * (ap->noffsets + 1) + ap->nsnames;
		assert(len % 2u == 0);
		memset(&hdr, ' ', sizeof(hdr));
		hdr.ar_name[0] = '/';
		setnum(hdr.ar_date, sizeof(hdr.ar_date), time(NULL), 10);
		hdr.ar_uid[0] = '0';
		hdr.ar_gid[0] = '0';
		hdr.ar_mode[0] = '0';
		setnum(hdr.ar_size, sizeof(hdr.ar_size), len, 10);
		memcpy(hdr.ar_fmag, ARFMAG, sizeof(hdr.ar_fmag));
		if (xwrite(fd, &hdr, sizeof(hdr))) {
			return -1;
		}
		/*
		 * Convert offsets to big-endian
		 */
		base = SARMAG + sizeof(hdr) + len;
		if (ap->nfnames) {
			base += sizeof(hdr) + ap->nfnames + (ap->nfnames % 2u);
		}
		i = 0;
		x = ap->noffsets;
		p = buf = xmalloc(4 * (ap->noffsets + 1));
		for (;;) {
			*p++ = (x >> 24) & 0xffu;
			*p++ = (x >> 16) & 0xffu;
			*p++ = (x >>  8) & 0xffu;
			*p++ = (x >>  0) & 0xffu;
			if (i >= ap->noffsets) {
				break;
			}
			x = base + ap->offsets[i++];
		}
		/*
		 * Write symbol table
		 */
		if (xwrite(fd, buf, p - buf) || xwrite(fd, ap->snames, ap->nsnames)) {
			xfree(buf);
			return -1;
		}
		xfree(buf);
	}
	else {
		assert(!ap->noffsets && !ap->nsnames);
	}
	/*
	 * Write archive string table, if any
	 */
	if (ap->nfnames) {
		if (ap->nfnames % 2u != 0) {
			/* XXX: move this? */
			ap->fnames[ap->nfnames++] = '\n';
		}
		memset(&hdr, ' ', sizeof(hdr));
		hdr.ar_name[0] = '/';
		hdr.ar_name[1] = '/';
		setnum(hdr.ar_size, sizeof(hdr.ar_size), ap->nfnames, 10);
		memcpy(hdr.ar_fmag, ARFMAG, sizeof(hdr.ar_fmag));
		if (xwrite(fd, &hdr, sizeof(hdr)) || xwrite(fd, ap->fnames, ap->nfnames)) {
			return -1;
		}
	}
	if (ap->nbytes) {
		ar_align_bytes(ap);
		if (xwrite(fd, ap->bytes, ap->nbytes)) {
			return -1;
		}
	}
	return 0;
}

int
ar_update_archive(ardesc_t *ap, const char *arch, int bak) {
	struct stat st;
	mode_t umsk;
	char *s;
	char *tmpl;
	int err;
	int fd;

	tmpl = xmalloc(strlen(arch) + 9);
	umsk = umask(077);	/* force mode 0600 for mkstemp */
	if (stat(arch, &st)) {
		/* file does not exist, use default mode */
		st.st_mode = 0666 & ~umsk;
		/* don't change uid/gid (that is, use my own) */
		st.st_uid = -1;
		st.st_gid = -1;
	}
	else if (bak) {
		/*
		 * Backup the original archive
		 */
		strcpy(tmpl, arch);
		strcat(tmpl, ".bak");
		unlink(tmpl);
		if (link(arch, tmpl)) {
			file_warn(arch, "can not create backup: %s", strerror(errno));
		}
	}
	/*
	 * Write temporary and rename it
	 */
	err = 0;
	strcpy(tmpl, arch);
	s = (s = strrchr(tmpl, '/')) ? s + 1 : tmpl;
	strcpy(s, "arXXXXXX");
	if ((fd = mkstemp(tmpl)) != -1) {
		if (ar_write(ap, fd)) {
			file_error(tmpl, "could not write archive file");
			unlink(tmpl);
			err = -1;
		}
		else if (rename(tmpl, arch)) {
			file_error(tmpl, "rename: %s", strerror(errno));
			unlink(tmpl);
			err = -1;
		}
		else {
			if (fchown(fd, st.st_uid, st.st_gid) == -1) {
				/* drop setuid/setgid/sticky attributes */
				st.st_mode &= 0777;
			}
			fchmod(fd, st.st_mode & 07777);
		}
		close(fd);
	}
	else {
		file_error(tmpl, "mkstemp: %s", strerror(errno));
		err = -1;
	}
	xfree(tmpl);
	umask(umsk);
	return err;
}

/* vi: set ts=4 sw=4 : */
