/*
 * fcpu_opcodes.l -- Flex-based Scanner/Encoder for F-CPU Instructions
 * Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%{
static const char rcsid[] = "@(#) $Id: fcpu_opcodes.l,v 1.28 2003/01/09 02:24:36 michael Exp $";
%}

/* shorthands for suffixes that are often used */

/* size suffix */
/* XXX: be more specific? */
sz		(("."([a-z]|[0-9]+))?)
/* rounding mode */
rnd		([rtfc]?)
/* load/store stream hints */
sh		([0-7]?)
/* short condition code */
cc		([zmln])
/* long combine mode (currently unused) */
comb	(\.(and|or))
/* long condition code (currently unused) */
cond	(\.(not)?(zero|nan|lsb|msb))

/* cachemm flags (XXX: clarify suffix order) */
fplc	([fp](l?c?|c?l?))
/* serialize flags (XXX: clarify suffix order) */
sxm		(s?x?m?|s?m?x?|x?m?s?|x?s?m?|m?s?x?|m?x?s?)

%{

#include <fcpu_opcodes.h>

/* Exported functions */

/* Encode instruction name (see description below) */
int fcpu_encode_instruction(const char *name);

/* Functions to be supplied by the caller */

/* `warn' and `error' should return, `fatal' not */
/* Arguments are `printf'-style */
extern void warn(const char *fmt, ...);
extern void error(const char *fmt, ...);
extern void fatal(const char *fmt, ...);

/* Local functions */

/* Encode instruction with size/SIMD/suffixes */
static int encode_simd(int args, int sndx, unsigned op, unsigned v,
					   int blen, const char *sfxs, ...);
/* Encode FP instruction with size/SIMD/suffixes */
static int encode_fpop(int args, int sndx, unsigned op, unsigned v,
					   int blen, const char *sfxs, ...);
/* Encode instruction without size/SIMD */
static int encode_nosz(int args, unsigned op, unsigned v,
					   int blen, const char *sfxs, ...);
/* Encode trivial instruction without any flags */
static int encode_triv(int args, unsigned op);
/* Encode loadcons[x] instructions */
static int encode_loadcons(int args, unsigned op, int blen);

/* XXX: add a function that handles `verbose' (.name) suffixes? */

/*
 * The arguments for `encode_*' are rather tricky (and I'm very proud of
 * them! ;)  Here's a short explanation:
 *
 * args     a constant that describes the number and kind of arguments
 *          to expect (see fcpu_opcodes.h for the complete list).
 *
 * sndx     `S index': the character position to check for the s- prefix
 *          = 0  indicates that the instruction does not have an s- prefix
 *          = 1  the instruction does have an s- prefix (first character)
 *          = 2  same as 1, but the `s' to check for is the second character
 *               (because the unprefixed name begins with "s")
 *          = 3  likewise, if the unprefixed name starts with "ss"
 *               (currently unused, but you get the idea now, don't you?)
 *
 * op       the opcode (0...255, will be shifted by the function).
 *
 * v        ("variant") flags that are always set for this instruction
 *          (must be already shifted).
 *
 * blen     size of the fixed portion of the name (without any prefix/suffix).
 *          Indicates where `encode_*' should start looking for
 *          single-character suffixes.  Suffix search ends when the
 *          function reaches the end of the string or encounters a
 *          `.' character (which indicates the start of a size suffix).
 *
 * sfxs     a string containing all possible (single-character) suffixes
 *          (in an order).  Note that the standard size suffixes (both
 *          symbolic and numeric) are handled automatically; they need
 *          not be specified.  If the instruction doesn't have any
 *          other suffixes, use an empty string ("").
 *
 * ...      flags to set for each of the suffixes in `sfxs', in the
 *          same order (must be already shifted).  The number of flag
 *          arguments MUST match the number of characters in `sfxs'.
 *
 * All `encode_*' functions (and also fcpu_encode_instruction) return
 * the (hopefully) correctly encoded instruction, with most register
 * and immediate slots set to 0 (zero).  The exception is the `r1' slot
 * (bits 0...5) which contains the constant passed to `encode_*' in the
 * `args' argument described above.  The caller can extract that value
 * and use it to verify the correctness of the argument list before it
 * fills in the missing parts.
 *
 * If fcpu_encode_instruction does not recognize an instruction, it
 * returns -1.  If it does recognize the name but does not support the
 * instruction (for any reason), it will return -2.  Other negative
 * values probably indicate valid instructions (with opcodes >= 128),
 * depending on the size of an integer.  Thus, please do not check for
 * a negative result but for the exact values -1 and -2.
 */

%}

%option outfile="fcpu_opcodes.c"
%option prefix="fcpu_opcodes_"
%option never-interactive

%%

s?add[cs]?{sz}			return encode_simd(ARG_RRR,  1, OP_ADD,    0, 3, "cs?", ADD_MODE_CARRY, ADD_MODE_SATURATE, ADD_MODE_3);
s?sub[bf]?{sz}			return encode_simd(ARG_RRR,  2, OP_SUB,    0, 3, "bf?", SUB_MODE_BORROW, SUB_MODE_FLOOR, SUB_MODE_3);
s?mulh?s?{sz}			return encode_simd(ARG_RRR,  1, OP_MUL,    0, 3, "hs", MUL_HIGH, MUL_SIGNED);
s?divr?s?{sz}			return encode_simd(ARG_RRR,  1, OP_DIV,    0, 3, "rs", DIV_REMAINDER, DIV_SIGNED);
s?divrems?{sz}			return encode_simd(ARG_RRR,  1, OP_DIV,    DIV_REMAINDER, 6, "s", DIV_SIGNED);

s?addi{sz}				return encode_simd(ARG_U8RR, 1, OP_ADDI,   0, 4, "");
s?subi{sz}				return encode_simd(ARG_U8RR, 2, OP_SUBI,   0, 4, "");
s?muli{sz}				return encode_simd(ARG_S8RR, 1, OP_MULI,   0, 4, "");
s?divi{sz}				return encode_simd(ARG_S8RR, 1, OP_DIVI,   0, 4, "");
s?divri{sz}				return encode_simd(ARG_S8RR, 1, OP_DIVI,   DIV_REMAINDER, 5, "");
s?divremi{sz}			return encode_simd(ARG_S8RR, 1, OP_DIVI,   DIV_REMAINDER, 7, "");
s?rems?{sz}				return encode_simd(ARG_RRR,  1, OP_REM,    0, 3, "s", DIV_SIGNED);
s?remi{sz}				return encode_simd(ARG_S8RR, 1, OP_REMI,   0, 4, "");
s?mac[lh]?s?{sz}		return encode_simd(ARG_RRR,  1, OP_MAC,    0, 3, "lhs", MAC_LOW, MAC_HIGH, MAC_SIGNED);
s?amach?s?{sz}			return encode_simd(ARG_RRR,  1, OP_AMAC,   0, 4, "hs", MUL_HIGH, MUL_SIGNED);
s?addsub{sz}			return encode_simd(ARG_RRR,  1, OP_ADDSUB, 0, 6, "");
s?popcount{sz}			return encode_simd(ARG_ORR,  1, OP_POPC,   0, 8, "");
s?popcounti{sz}			return encode_simd(ARG_U8RR, 1, OP_POPCI,  0, 9, "");

s?inc{sz}				return encode_simd(ARG_RR,   1, OP_INC,      0, 3, "");
s?dec{sz}				return encode_simd(ARG_RR,   1, OP_DEC,      0, 3, "");
s?neg{sz}				return encode_simd(ARG_RR,   1, OP_NEG,      0, 3, "");
s?scann?r?{sz}			return encode_simd(ARG_RR,   2, OP_SCAN,     0, 4, "rn", SCAN_REVERSE, SCAN_NEGATE);
s?lsb[01]{sz}			return encode_simd(ARG_RR,   1, OP_SCAN,            0, 3, "10", 0, SCAN_NEGATE);
s?msb[01]{sz}			return encode_simd(ARG_RR,   1, OP_SCAN, SCAN_REVERSE, 3, "10", 0, SCAN_NEGATE);
s?cmpg{sz}				return encode_simd(ARG_RRR,  1, OP_CMPG,     0, 4, "");
s?cmpgi{sz}				return encode_simd(ARG_U8RR, 1, OP_CMPGI,    0, 5, "");
s?cmple{sz}				return encode_simd(ARG_RRR,  1, OP_CMPLE,    0, 5, "");
s?cmplei{sz}			return encode_simd(ARG_U8RR, 1, OP_CMPLEI,   0, 6, "");
s?abs{sz}				return encode_simd(ARG_RR,   1, OP_ABS,      0, 3, "");
s?nabs{sz}				return encode_simd(ARG_RR,   1, OP_NABS,     0, 4, "");
s?max{sz}				return encode_simd(ARG_RRR,  1, OP_MAX,      0, 3, "");
s?min{sz}				return encode_simd(ARG_RRR,  1, OP_MIN,      0, 3, "");
s?maxi{sz}				return encode_simd(ARG_U8RR, 1, OP_MAXI,     0, 4, "");
s?mini{sz}				return encode_simd(ARG_U8RR, 1, OP_MINI,     0, 4, "");
s?sort{sz}				return encode_simd(ARG_RRR,  2, OP_MINMAX,   0, 4, "");
s?sorti{sz}				return encode_simd(ARG_U8RR, 2, OP_MINMAXI,  0, 5, "");
s?minmax{sz}			return encode_simd(ARG_RRR,  1, OP_MINMAX,   0, 6, "");
s?minmaxi{sz}			return encode_simd(ARG_U8RR, 1, OP_MINMAXI,  0, 7, "");
s?cmpgs{sz}				return encode_simd(ARG_RRR,  1, OP_CMPG,     CMP_SIGNED, 5, "");
s?cmpgsi{sz}			return encode_simd(ARG_S8RR, 1, OP_CMPGI,    CMP_SIGNED, 6, "");
s?cmples{sz}			return encode_simd(ARG_RRR,  1, OP_CMPLE,    CMP_SIGNED, 6, "");
s?cmplesi{sz}			return encode_simd(ARG_S8RR, 1, OP_CMPLEI,   CMP_SIGNED, 7, "");
s?maxs{sz}				return encode_simd(ARG_RRR,  1, OP_MAX,      CMP_SIGNED, 4, "");
s?mins{sz}				return encode_simd(ARG_RRR,  1, OP_MIN,      CMP_SIGNED, 4, "");
s?maxsi{sz}				return encode_simd(ARG_S8RR, 1, OP_MAXI,     CMP_SIGNED, 5, "");
s?minsi{sz}				return encode_simd(ARG_S8RR, 1, OP_MINI,     CMP_SIGNED, 5, "");
s?sorts{sz}				return encode_simd(ARG_RRR,  2, OP_MINMAX,   CMP_SIGNED, 5, "");
s?sortsi{sz}			return encode_simd(ARG_S8RR, 2, OP_MINMAXI,  CMP_SIGNED, 6, "");
s?minmaxs{sz}			return encode_simd(ARG_RRR,  1, OP_MINMAX,   CMP_SIGNED, 7, "");
s?minmaxsi{sz}			return encode_simd(ARG_S8RR, 1, OP_MINMAXI,  CMP_SIGNED, 8, "");

						/* XXX: aliases */
s?cmpl{sz}				return encode_simd(ARG_231,     1, OP_CMPG,   0, 4, "");
s?cmpls{sz}				return encode_simd(ARG_231,     1, OP_CMPG,   CMP_SIGNED, 5, "");
s?cmpgei{sz}			return encode_simd(ARG_U8RRoff, 1, OP_CMPGI,  0, 6, "");
s?cmpgesi{sz}			return encode_simd(ARG_S8RRoff, 1, OP_CMPGI,  CMP_SIGNED, 7, "");
s?cmpge{sz}				return encode_simd(ARG_231,     1, OP_CMPLE,  0, 5, "");
s?cmpges{sz}			return encode_simd(ARG_231,     1, OP_CMPLE,  CMP_SIGNED, 6, "");
s?cmpli{sz}				return encode_simd(ARG_U8RRoff, 1, OP_CMPLEI, 0, 5, "");
s?cmplsi{sz}			return encode_simd(ARG_S8RRoff, 1, OP_CMPLEI, CMP_SIGNED, 6, "");

						/* XXX: possible sizes? */
s?ladd{sz}				return encode_simd(ARG_RRR,  1, OP_LADD,   0, 4, "");
s?lsub{sz}				return encode_simd(ARG_RRR,  1, OP_LSUB,   0, 4, "");
						/* XXX: same size for source and destination? */
s?l2int{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_L2INT,  0, 5, "rtfc", ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);
s?int2l{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_INT2L,  0, 5, "rtfc", ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);

						/* XXX: 6-bit immediate? */
s?shiftlh?{sz}			return encode_simd(ARG_RRR,  2, OP_SHIFTL,   0, 6, "h", SHIFT_HALF_SIMD);
s?shiftli{sz}			return encode_simd(ARG_U8RR, 2, OP_SHIFTLI,  0, 7, "");
s?shiftrh?{sz}			return encode_simd(ARG_RRR,  2, OP_SHIFTR,   0, 6, "h", SHIFT_HALF_SIMD);
s?shiftri{sz}			return encode_simd(ARG_U8RR, 2, OP_SHIFTRI,  0, 7, "");
s?shiftrah?{sz}			return encode_simd(ARG_RRR,  2, OP_SHIFTRA,  0, 7, "h", SHIFT_HALF_SIMD);
s?shiftrai{sz}			return encode_simd(ARG_U8RR, 2, OP_SHIFTRAI, 0, 8, "");
s?rotlh?{sz}			return encode_simd(ARG_RRR,  1, OP_ROTL,     0, 4, "h", SHIFT_HALF_SIMD);
s?rotli{sz}				return encode_simd(ARG_U8RR, 1, OP_ROTLI,    0, 5, "");
s?rotrh?{sz}			return encode_simd(ARG_RRR,  1, OP_ROTR,     0, 4, "h", SHIFT_HALF_SIMD);
s?rotri{sz}				return encode_simd(ARG_U8RR, 1, OP_ROTRI,    0, 5, "");

						/* XXX: 6-bit immediate? */
s?bitrevh?o?{sz}		return encode_simd(ARG_RRR,  1, OP_BITREV,   0, 6, "oh", BITREV_OR, SHIFT_HALF_SIMD);
s?bitrevio?{sz}			return encode_simd(ARG_U8RR, 1, OP_BITREVI,  0, 7, "o", BITREV_OR);

s?dshiftlh?{sz}			return encode_simd(ARG_RRR,  1, OP_DSHIFTL,   0, 7, "h", SHIFT_HALF_SIMD);
s?dshiftli{sz}			return encode_simd(ARG_U8RR, 1, OP_DSHIFTLI,  0, 8, "");
s?dshiftrh?{sz}			return encode_simd(ARG_RRR,  1, OP_DSHIFTR,   0, 7, "h", SHIFT_HALF_SIMD);
s?dshiftri{sz}			return encode_simd(ARG_U8RR, 1, OP_DSHIFTRI,  0, 8, "");
s?dshiftrah?{sz}		return encode_simd(ARG_RRR,  1, OP_DSHIFTRA,  0, 8, "h", SHIFT_HALF_SIMD);
s?dshiftrai{sz}			return encode_simd(ARG_U8RR, 1, OP_DSHIFTRAI, 0, 9, "");
s?dbitrevh?{sz}			return encode_simd(ARG_RRR,  1, OP_DBITREV,   0, 7, "h", SHIFT_HALF_SIMD);
s?dbitrevi{sz}			return encode_simd(ARG_U8RR, 1, OP_DBITREVI,  0, 8, "");

						/* XXX: 6-bit immediate? */
s?btst{sz}				return encode_simd(ARG_RRR,  1, OP_BTST,  0, 4, "");
s?bclr{sz}				return encode_simd(ARG_RRR,  1, OP_BCLR,  0, 4, "");
s?bchg{sz}				return encode_simd(ARG_RRR,  1, OP_BCHG,  0, 4, "");
s?bset{sz}				return encode_simd(ARG_RRR,  1, OP_BSET,  0, 4, "");
s?btsti{sz}				return encode_simd(ARG_U8RR, 1, OP_BTSTI, 0, 5, "");
s?bclri{sz}				return encode_simd(ARG_U8RR, 1, OP_BCLRI, 0, 5, "");
s?bchgi{sz}				return encode_simd(ARG_U8RR, 1, OP_BCHGI, 0, 5, "");
s?bseti{sz}				return encode_simd(ARG_U8RR, 1, OP_BSETI, 0, 5, "");

mix[hl]{sz}				return encode_simd(ARG_RRR,  0, OP_MIX,     0, 3, "hl", MIX_HIGH, MIX_LOW);
expand[hl]{sz}			return encode_simd(ARG_RRR,  0, OP_EXPAND,  0, 6, "hl", EXPAND_HIGH, EXPAND_LOW);
mix{sz}					return encode_simd(ARG_RRR,  0, OP_MIX,     MIX_BOTH, 3, "");
expand{sz}				return encode_simd(ARG_RRR,  0, OP_EXPAND,  EXPAND_BOTH, 6, "");
cshift[lr]{sz}			return encode_simd(ARG_ORR,  0, OP_CSHIFT,  0, 6, "lr", CSHIFT_LEFT, CSHIFT_RIGHT);
s?byterev{sz}			return encode_simd(ARG_RR,   1, OP_BYTEREV, 0, 7, "");
s?vselh?{sz}			return encode_simd(ARG_RRR,  1, OP_VSEL,          0, 4, "h", SHIFT_HALF_SIMD);
vpermh?{sz}				return encode_simd(ARG_RRR,  0, OP_VSEL,  SIMD_FLAG, 5, "h", SHIFT_HALF_SIMD);
sdup{sz}				return encode_simd(ARG_ORR,  0, OP_VSEL,  SIMD_FLAG | SHIFT_HALF_SIMD, 4, "");
s?vseli{sz}				return encode_simd(ARG_U8RR, 1, OP_VSELI,         0, 5, "");
vpermi{sz}				return encode_simd(ARG_U8RR, 0, OP_VSELI, SIMD_FLAG, 6, "");
sdupi{sz}				return encode_simd(ARG_U8RR, 0, OP_VSELI, SIMD_FLAG, 5, "");

						/* logic ops now always have a size */
s?and{sz}				return encode_simd(ARG_RRR,  1, OP_AND,  ROP2_MODE_DIRECT, 3, "");
s?andn{sz}				return encode_simd(ARG_RRR,  1, OP_ANDN, ROP2_MODE_DIRECT, 4, "");
s?xor{sz}				return encode_simd(ARG_RRR,  1, OP_XOR,  ROP2_MODE_DIRECT, 3, "");
s?or{sz}				return encode_simd(ARG_RRR,  1, OP_OR,   ROP2_MODE_DIRECT, 2, "");
s?nor{sz}				return encode_simd(ARG_RRR,  1, OP_NOR,  ROP2_MODE_DIRECT, 3, "");
s?xnor{sz}				return encode_simd(ARG_RRR,  1, OP_XNOR, ROP2_MODE_DIRECT, 4, "");
s?orn{sz}				return encode_simd(ARG_RRR,  1, OP_ORN,  ROP2_MODE_DIRECT, 3, "");
s?nand{sz}				return encode_simd(ARG_RRR,  1, OP_NAND, ROP2_MODE_DIRECT, 4, "");
						/* and now, immediate */
s?andi{sz}				return encode_simd(ARG_S9RR, 1, OP_ANDI,  0, 4, "");
s?andni{sz}				return encode_simd(ARG_S9RR, 1, OP_ANDNI, 0, 5, "");
s?xori{sz}				return encode_simd(ARG_S9RR, 1, OP_XORI,  0, 4, "");
s?ori{sz}				return encode_simd(ARG_S9RR, 1, OP_ORI,   0, 3, "");
s?nori{sz}				return encode_simd(ARG_S9RR, 1, OP_NORI,  0, 4, "");
s?xnori{sz}				return encode_simd(ARG_S9RR, 1, OP_XNORI, 0, 5, "");
s?orni{sz}				return encode_simd(ARG_S9RR, 1, OP_ORNI,  0, 4, "");
s?nandi{sz}				return encode_simd(ARG_S9RR, 1, OP_NANDI, 0, 5, "");
						/* mux */
s?mux{sz}				return encode_simd(ARG_RRR,  1, OP_ROP2, ROP2_MODE_MUX, 3, "");
						/* short combine */
s?and[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_AND,  0, 3, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?andn[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_ANDN, 0, 4, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?xor[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_XOR,  0, 3, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?or[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_OR,   0, 2, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?nor[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_NOR,  0, 3, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?xnor[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_XNOR, 0, 4, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?orn[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_ORN,  0, 3, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
s?nand[ao]{sz}			return encode_simd(ARG_RRR,  1, OP_NAND, 0, 4, "ao", ROP2_MODE_AND, ROP2_MODE_OR);
						/* long combine */
s?and\.and{sz}			return encode_simd(ARG_RRR,  1, OP_AND,  ROP2_MODE_AND, 7, "");
s?andn\.and{sz}			return encode_simd(ARG_RRR,  1, OP_ANDN, ROP2_MODE_AND, 8, "");
s?xor\.and{sz}			return encode_simd(ARG_RRR,  1, OP_XOR,  ROP2_MODE_AND, 7, "");
s?or\.and{sz}			return encode_simd(ARG_RRR,  1, OP_OR,   ROP2_MODE_AND, 6, "");
s?nor\.and{sz}			return encode_simd(ARG_RRR,  1, OP_NOR,  ROP2_MODE_AND, 7, "");
s?xnor\.and{sz}			return encode_simd(ARG_RRR,  1, OP_XNOR, ROP2_MODE_AND, 8, "");
s?orn\.and{sz}			return encode_simd(ARG_RRR,  1, OP_ORN,  ROP2_MODE_AND, 7, "");
s?nand\.and{sz}			return encode_simd(ARG_RRR,  1, OP_NAND, ROP2_MODE_AND, 8, "");
s?and\.or{sz}			return encode_simd(ARG_RRR,  1, OP_AND,  ROP2_MODE_OR,  6, "");
s?andn\.or{sz}			return encode_simd(ARG_RRR,  1, OP_ANDN, ROP2_MODE_OR,  7, "");
s?xor\.or{sz}			return encode_simd(ARG_RRR,  1, OP_XOR,  ROP2_MODE_OR,  6, "");
s?or\.or{sz}			return encode_simd(ARG_RRR,  1, OP_OR,   ROP2_MODE_OR,  5, "");
s?nor\.or{sz}			return encode_simd(ARG_RRR,  1, OP_NOR,  ROP2_MODE_OR,  6, "");
s?xnor\.or{sz}			return encode_simd(ARG_RRR,  1, OP_XNOR, ROP2_MODE_OR,  7, "");
s?orn\.or{sz}			return encode_simd(ARG_RRR,  1, OP_ORN,  ROP2_MODE_OR,  6, "");
s?nand\.or{sz}			return encode_simd(ARG_RRR,  1, OP_NAND, ROP2_MODE_OR,  7, "");

	/*
	 * FP instructions
	 *
	 * Note that the size suffix is different here:
	 *   00   .f   32-bit (IEEE float)
	 *   01   .d   64-bit (IEEE double, default)
	 *   10        (unassigned)
	 *   11        (unassigned)
	 *
	 * XXX: Possible extensions:
	 *   .l   `long double' (with size >= 80-bit)
	 *   .t    80-bit (`ten-byte' Intel extended)
	 *   .q   128-bit (`quad-precision' SPARC extended)
	 */

s?faddx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FADD,    0, 4, "x", FPOP_X);
s?fsubx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FSUB,    0, 4, "x", FPOP_X);
s?fmulx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FMUL,    0, 4, "x", FPOP_X);
						/* XXX: rounding mode not fully documented */
						/* XXX: clarify suffix order */
s?f2intx?{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_F2INT,   0, 5, "xrtfc", FPOP_X, ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);
s?int2fx?{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_INT2F,   0, 5, "xrtfc", FPOP_X, ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);
s?d2intx?{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_D2INT,   0, 5, "xrtfc", FPOP_X, ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);
s?int2dx?{rnd}{sz}		return encode_simd(ARG_RR,   1, OP_INT2D,   0, 5, "xrtfc", FPOP_X, ROUND_NEAREST, ROUND_TRUNC, ROUND_FLOOR, ROUND_CEIL);
s?fiaprxx?{sz}			return encode_fpop(ARG_RR,   1, OP_FIAPRX,  0, 6, "x", FPOP_X);
s?fsqrtiaprxx?{sz}		return encode_fpop(ARG_RR,   1, OP_FSQRTIAPRX, 0, 10, "x", FPOP_X);

s?fdivx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FDIV,    0, 4, "x", FPOP_X);
s?fsqrtx?{sz}			return encode_fpop(ARG_RR,   1, OP_FSQRT,   0, 5, "x", FPOP_X);

s?flogx?{sz}			return encode_fpop(ARG_ORR,  1, OP_FLOG,    0, 4, "x", FPOP_X);
s?fexpx?{sz}			return encode_fpop(ARG_ORR,  1, OP_FEXP,    0, 4, "x", FPOP_X);
s?fmacx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FMAC,    0, 4, "x", FPOP_X);
s?faddsubx?{sz}			return encode_fpop(ARG_RRR,  1, OP_FADDSUB, 0, 7, "x", FPOP_X);

						/* XXX: more relaxed ([ef0-7]*) syntax? */
loade?{sh}{sz}			return encode_simd(ARG_ORR,  0, OP_LOAD,    0, 4, "e01234567", LS_BIG_ENDIAN, LS_STREAM_0, LS_STREAM_1, LS_STREAM_2, LS_STREAM_3, LS_STREAM_4, LS_STREAM_5, LS_STREAM_6, LS_STREAM_7);
storee?{sh}{sz}			return encode_simd(ARG_ORR,  0, OP_STORE,   0, 5, "e01234567", LS_BIG_ENDIAN, LS_STREAM_0, LS_STREAM_1, LS_STREAM_2, LS_STREAM_3, LS_STREAM_4, LS_STREAM_5, LS_STREAM_6, LS_STREAM_7);
loadfe?{sh}{sz}			return encode_simd(ARG_ORR,  0, OP_LOADF,   0, 5, "e01234567", LS_BIG_ENDIAN, LS_STREAM_0, LS_STREAM_1, LS_STREAM_2, LS_STREAM_3, LS_STREAM_4, LS_STREAM_5, LS_STREAM_6, LS_STREAM_7);
storefe?{sh}{sz}		return encode_simd(ARG_ORR,  0, OP_STOREF,  0, 6, "e01234567", LS_BIG_ENDIAN, LS_STREAM_0, LS_STREAM_1, LS_STREAM_2, LS_STREAM_3, LS_STREAM_4, LS_STREAM_5, LS_STREAM_6, LS_STREAM_7);
						/* XXX: there are no stream hints in (load|store)i[f] */
loadie?{sz}				return encode_simd(ARG_S9RR, 0, OP_LOADI,   0, 5, "e", LS_BIG_ENDIAN);
storeie?{sz}			return encode_simd(ARG_S9RR, 0, OP_STOREI,  0, 6, "e", LS_BIG_ENDIAN);
loadife?{sz}			return encode_simd(ARG_S9RR, 0, OP_LOADIF,  0, 6, "e", LS_BIG_ENDIAN);
storeife?{sz}			return encode_simd(ARG_S9RR, 0, OP_STOREIF, 0, 7, "e", LS_BIG_ENDIAN);
						/* XXX: new cstore/cload (subject to change) */
cload{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_CLOAD,       0, 5, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
cloadn{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_CLOAD,  CC_NOT, 6, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
cstore{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_CSTORE,      0, 6, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
cstoren{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_CSTORE, CC_NOT, 7, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);

						/* XXX deprecated
cachemm{fplc}{sz}		return encode_simd(ARG_RR,   0, OP_CACHEMM, 0, 7, "fpcl", CACHEMM_FLUSH, CACHEMM_PREFETCH, CACHEMM_COMPRESS, CACHEMM_LOCK);
						*/

move{sz}				return encode_simd(ARG_RR,   0, OP_MOVE,      0, 4, "");
move{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE,      0, 4, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
moven{cc}{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NOT, 5, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
						/* XXX: long versions */
move\.zero{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE, CC_ZERO,      9, "");
move\.nan{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NAN,       8, "");
move\.msb{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE, CC_MSB,       8, "");
move\.lsb{sz}			return encode_simd(ARG_RRR,  0, OP_MOVE, CC_LSB,       8, "");
move\.notzero{sz}		return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NOT_ZERO, 12, "");
move\.notnan{sz}		return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NOT_NAN,  11, "");
move\.notmsb{sz}		return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NOT_MSB,  11, "");
move\.notlsb{sz}		return encode_simd(ARG_RRR,  0, OP_MOVE, CC_NOT_LSB,  11, "");

widen{sz}				return encode_simd(ARG_RR,   0, OP_WIDEN, 0, 5, "");

loadcons\.[0-3]			return encode_loadcons(ARG_U16R, OP_LOADCONS, 8);
loadconsx\.[0-3]		return encode_loadcons(ARG_S16R, OP_LOADCONSX, 9);
						/* XXX: convenience aliases for large constants (need special handling) */
loadcons				return encode_loadcons(ARG_U64R, OP_LOADCONS, 8);
loadconsx				return encode_loadcons(ARG_S64R, OP_LOADCONSX, 9);

loopentry				return encode_nosz(ARG_R,    OP_LOADADDR,  0, 9, "");
loadaddrd?				return encode_nosz(ARG_RR,   OP_LOADADDR,  0, 8, "d", LOADADDR_DATA);
loadaddrid?				return encode_nosz(ARG_S17R, OP_LOADADDRI, 0, 9, "d", LOADADDR_DATA);

get						return encode_triv(ARG_RR,   OP_GET);
put						return encode_triv(ARG_RR,   OP_PUT);
geti					return encode_triv(ARG_U16R, OP_GETI);
puti					return encode_triv(ARG_U16R, OP_PUTI);

loadm					return encode_triv(ARG_RRR,  OP_LOADM);
storem					return encode_triv(ARG_RRR,  OP_STOREM);

jmp						return encode_triv(ARG_RO,   OP_JMP);
jmp{cc}					return encode_nosz(ARG_RRO,  OP_JMP,      0, 3, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
jmpn{cc}				return encode_nosz(ARG_RRO,  OP_JMP, CC_NOT, 4, "zmln", CC_ZERO, CC_MSB, CC_LSB, CC_NAN);
						/* XXX: long versions */
jmp\.zero				return encode_nosz(ARG_RRO,  OP_JMP, CC_ZERO,      8, "");
jmp\.nan				return encode_nosz(ARG_RRO,  OP_JMP, CC_NAN,       7, "");
jmp\.msb				return encode_nosz(ARG_RRO,  OP_JMP, CC_MSB,       7, "");
jmp\.lsb				return encode_nosz(ARG_RRO,  OP_JMP, CC_LSB,       7, "");
jmp\.notzero			return encode_nosz(ARG_RRO,  OP_JMP, CC_NOT_ZERO, 11, "");
jmp\.notnan				return encode_nosz(ARG_RRO,  OP_JMP, CC_NOT_NAN,  10, "");
jmp\.notmsb				return encode_nosz(ARG_RRO,  OP_JMP, CC_NOT_MSB,  10, "");
jmp\.notlsb				return encode_nosz(ARG_RRO,  OP_JMP, CC_NOT_LSB,  10, "");
loop					return encode_triv(ARG_RR,   OP_LOOP);
syscall					return encode_triv(ARG_U16R, OP_SYSCALL);
trap					return encode_nosz(ARG_U16R, OP_SYSCALL, SYSCALL_TRAP, 4, "");
halt					return encode_triv(ARG_NONE, OP_HALT);
rfe						return encode_triv(ARG_NONE, OP_RFE);

srb_save				return encode_triv(ARG_NONE, OP_SRB_SAVE);
srb_restore				return encode_triv(ARG_NONE, OP_SRB_RESTORE);
						/* XXX: drop this? */
						/* XXX: clarify suffix order */
serialize{sxm}			return encode_nosz(ARG_NONE, OP_SERIALIZE, 0, 9, "sxm", SERIALIZE_SRB, SERIALIZE_EXEC, SERIALIZE_MEMORY);

nop						return encode_triv(ARG_NONE, OP_NOP);

						/* catch-all */
.*						error("unknown instruction \"%s\"", yytext); return -1;

%%

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
char *strchr();
#endif

#include <stdarg.h>
#include <assert.h>

/*
 * Flex wants this
 */
int
yywrap(void) {
	return 1;
}

/*
 * Encode trivial instruction without any flags
 */
static int
encode_triv(int args, unsigned op) {
	return args | (op << 24);
}

/*
 * Encode loadcons[x] instructions
 */
static int
encode_loadcons(int args, unsigned op, int blen) {
	unsigned tmp;

	op <<= 24;
	/* handle size suffix, if any */
	if (yytext[blen] == '.') {
		tmp = atoi(&yytext[blen + 1]);
		if (tmp < 4u) {
			op |= tmp << 22;
		}
		else {
			error("invalid shift count");
			return -1;
		}
	}
	return args | op;
}

/*
 * Encode instruction with size/SIMD/suffixes
 */
static int
encode_simd(int args, int sndx, unsigned op, unsigned v, int blen, const char *sfxs, ...) {
	unsigned suffixes = 0;
	unsigned tmp;
	va_list ap;
	char *s;
	char c;

	op = (op << 24) | v;
	if (sndx > 0 && yytext[sndx - 1] == 's') {
		op |= SIMD_FLAG;
		blen++;
	}
	/* collect flags */
	while ((c = yytext[blen]) && c != '.') {
		s = strchr(sfxs, c);
		assert(s);
		suffixes |= (1u << (s - sfxs));
		blen++;
	}
	/* handle size suffix, if any */
	if (c == '.') {
		switch (yytext[blen + 1]) {
			case 'b': op |= ISIZE_8BIT; break;
			case 'd': op |= ISIZE_16BIT; break;
			case 'q': op |= ISIZE_32BIT; break;
			case 'o': op |= ISIZE_64BIT; break;
			/* XXX: assign single-letter suffix for 64-bit mode? */
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				tmp = atoi(&yytext[blen + 1]);
				switch (tmp) {
					case  0:
					case  1:
					case  2:
					case  3: op |= tmp << 22; break;
					case  8: op |= ISIZE_8BIT; break;
					case 16: op |= ISIZE_16BIT; break;
					case 32: op |= ISIZE_32BIT; break;
					case 64: op |= ISIZE_64BIT; break;
					default:
						error("invalid numeric size suffix");
						return -1;
				}
				break;
			default:
				error("invalid size suffix `%s'", &yytext[blen]);
				return -1;
		}
	}
	else {
		/* default is full width */
		op |= ISIZE_64BIT;
	}
	/* apply flags */
	va_start(ap, sfxs);
	while (*sfxs++) {
		tmp = va_arg(ap, unsigned);
		if (suffixes & 1u) {
			op |= tmp;
		}
		suffixes >>= 1;
	}
	va_end(ap);
	return args | op;
}

/*
 * Encode FP instruction with size/SIMD/suffixes
 */
static int
encode_fpop(int args, int sndx, unsigned op, unsigned v, int blen, const char *sfxs, ...) {
	unsigned suffixes = 0;
	unsigned tmp;
	va_list ap;
	char *s;
	char c;

	op = (op << 24) | v;
	if (sndx > 0 && yytext[sndx - 1] == 's') {
		op |= SIMD_FLAG;
		blen++;
	}
	/* collect flags */
	while ((c = yytext[blen]) && c != '.') {
		s = strchr(sfxs, c);
		assert(s);
		suffixes |= (1u << (s - sfxs));
		blen++;
	}
	/* handle size suffix, if any */
	if (c == '.') {
		switch (c = yytext[blen + 1]) {
			case 'f': op |= FSIZE_SINGLE; break;
			case 'd': op |= FSIZE_DOUBLE; break;
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				tmp = atoi(&yytext[blen + 1]);
				switch (tmp) {
					case  0:
					case  1:
					case  2:
					case  3: op |= tmp << 22; break;
					case 32: op |= FSIZE_SINGLE; break;
					case 64: op |= FSIZE_DOUBLE; break;
					/* XXX: long double? */
					default:
						error("invalid numeric size suffix");
						return -1;
				}
				break;
			default:
				error("invalid size suffix `%s'", &yytext[blen]);
				return -1;
		}
	}
	else {
		/* default is 64-bit */
		op |= FSIZE_DOUBLE;
	}
	/* apply flags */
	va_start(ap, sfxs);
	while (*sfxs++) {
		tmp = va_arg(ap, unsigned);
		if (suffixes & 1u) {
			op |= tmp;
		}
		suffixes >>= 1;
	}
	va_end(ap);
	return args | op;
}

/*
 * Encode instruction without size/SIMD
 */
static int
encode_nosz(int args, unsigned op, unsigned v, int blen, const char *sfxs, ...) {
	unsigned suffixes = 0;
	unsigned tmp;
	va_list ap;
	char *s;
	char c;

	op = (op << 24) | v;
	/* collect flags */
	while ((c = yytext[blen])) {
		s = strchr(sfxs, c);
		assert(s);
		suffixes |= (1u << (s - sfxs));
		blen++;
	}
	/* apply flags */
	va_start(ap, sfxs);
	while (*sfxs++) {
		tmp = va_arg(ap, unsigned);
		if (suffixes & 1u) {
			op |= tmp;
		}
		suffixes >>= 1;
	}
	va_end(ap);
	return args | op;
}

/*
 * Main entry point
 */
int
fcpu_encode_instruction(const char *name) {
	YY_BUFFER_STATE buf;
	int res;

	buf = yy_scan_string(name);
	res = yylex();
	yy_delete_buffer(buf);
	return res;
}

#if TEST	/* this is OBSOLETE! */

void
print_msg(FILE *fp, const char *label, const char *fmt, va_list ap) {
	if (*label) {
		fputs(label, fp);
	}
	vfprintf(fp, fmt, ap);
	fputs("\n", fp);
}

void
warn(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "warning: ", fmt, ap);
	va_end(ap);
}

void
error(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "", fmt, ap);
	va_end(ap);
}

void
fatal(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "fatal error: ", fmt, ap);
	va_end(ap);
	/* no way to recover */
	exit(1);
}

static const char *opcode_table[256] = {
	[OP_ABS] = "abs",
	[OP_ADD] = "add",
	[OP_ADDI] = "addi",
	[OP_ADDSUB] = "addsub",
	[OP_AND] = "and",
	[OP_ANDI] = "andi",
	[OP_ANDN] = "andn",
	[OP_ANDNI] = "andni",
	[OP_BCHG] = "bchg",
	[OP_BCHGI] = "bchgi",
	[OP_BCLR] = "bclr",
	[OP_BCLRI] = "bclri",
	[OP_BITREV] = "bitrev",
	[OP_BITREVI] = "bitrevi",
	[OP_BSET] = "bset",
	[OP_BSETI] = "bseti",
	[OP_BTST] = "btst",
	[OP_BTSTI] = "btsti",
	[OP_BYTEREV] = "byterev",
	[OP_CACHEMM] = "cachemm",
	[OP_CMPG] = "cmpg",
	[OP_CMPGI] = "cmpgi",
	[OP_CMPLE] = "cmple",
	[OP_CMPLEI] = "cmplei",
	[OP_DEC] = "dec",
	[OP_DIV] = "div",
	[OP_DIVI] = "divi",
	[OP_EXPAND] = "expand",
	[OP_F2INT] = "f2int",
	[OP_D2INT] = "d2int",
	[OP_FADD] = "fadd",
	[OP_FADDSUB] = "faddsub",
	[OP_FDIV] = "fdiv",
	[OP_FEXP] = "fexp",
	[OP_FIAPRX] = "fiaprx",
	[OP_FLOG] = "flog",
	[OP_FMAC] = "fmac",
	[OP_FMUL] = "fmul",
	[OP_FSQRT] = "fsqrt",
	[OP_FSQRTIAPRX] = "fsqrtiaprx",
	[OP_FSUB] = "fsub",
	[OP_GET] = "get",
	[OP_GETI] = "geti",
	[OP_HALT] = "halt",
	[OP_INC] = "inc",
	[OP_INT2F] = "int2f",
	[OP_INT2D] = "int2d",
	[OP_INT2L] = "int2l",
	[OP_JMP] = "jmp",
	[OP_L2INT] = "l2int",
	[OP_LADD] = "ladd",
	[OP_LOAD] = "load",
	[OP_LOADADDR] = "loadaddr",
	[OP_LOADADDRI] = "loadaddri",
	[OP_LOADCONS+0] = "loadcons",
	[OP_LOADCONS+1] = "loadcons",
	[OP_LOADCONS+2] = "loadcons",
	[OP_LOADCONS+3] = "loadcons",
	[OP_LOADCONSX+0] = "loadconsx",
	[OP_LOADCONSX+1] = "loadconsx",
	[OP_LOADCONSX+2] = "loadconsx",
	[OP_LOADCONSX+3] = "loadconsx",
	[OP_LOADF] = "loadf",
	[OP_LOADI] = "loadi",
	[OP_LOADIF] = "loadif",
	[OP_LOADM] = "loadm",
	[OP_LOOP] = "loop",
	[OP_LSUB] = "lsub",
	[OP_MAC] = "mac",
	[OP_MAX] = "max",
	[OP_MAXI] = "maxi",
	[OP_MIN] = "min",
	[OP_MINI] = "mini",
	[OP_MIX] = "mix",
	[OP_REM] = "rem",
	[OP_REMI] = "remi",
	[OP_MOVE] = "move",
	[OP_MUL] = "mul",
	[OP_MULI] = "muli",
	[OP_NAND] = "nand",
	[OP_NANDI] = "nandi",
	[OP_NEG] = "neg",
	[OP_NOP] = "nop",
	[OP_NOR] = "nor",
	[OP_NORI] = "nori",
	[OP_OR] = "or",
	[OP_ORI] = "ori",
	[OP_ORN] = "orn",
	[OP_ORNI] = "orni",
	[OP_POPC] = "popcount",
	[OP_POPCI] = "popcounti",
	[OP_PUT] = "put",
	[OP_PUTI] = "puti",
	[OP_RFE] = "rfe",
	[OP_ROTL] = "rotl",
	[OP_ROTLI] = "rotli",
	[OP_ROTR] = "rotr",
	[OP_ROTRI] = "rotri",
	[OP_SCAN] = "scan",
	[OP_VSEL] = "vsel",
	[OP_SERIALIZE] = "serialize",
	[OP_SHIFTL] = "shiftl",
	[OP_SHIFTLI] = "shiftli",
	[OP_SHIFTR] = "shiftr",
	[OP_SHIFTRA] = "shiftra",
	[OP_SHIFTRAI] = "shiftrai",
	[OP_SHIFTRI] = "shiftri",
	[OP_SORT] = "sort",
	[OP_SRB_RESTORE] = "srb_restore",
	[OP_SRB_SAVE] = "srb_save",
	[OP_STORE] = "store",
	[OP_STOREF] = "storef",
	[OP_STOREI] = "storei",
	[OP_STOREIF] = "storeif",
	[OP_STOREM] = "storem",
	[OP_SUB] = "sub",
	[OP_SUBI] = "subi",
	[OP_SYSCALL] = "syscall",
	[OP_WIDEN] = "widen",
	[OP_XNOR] = "xnor",
	[OP_XNORI] = "xnori",
	[OP_XOR] = "xor",
	[OP_XORI] = "xori",
};

int
main(int argc, char **argv) {
	char buf[BUFSIZ];
	const char *s;
	unsigned x;

	while (fgets(buf, sizeof(buf), stdin)) {
		s = strtok(buf, " \t\r\f\v\n");
		if (!*s) continue;
		x = fcpu_encode_instruction(s);
		if ((int)x == -2) continue;	/* unsupported */
		if ((int)x == -1) {
			error("unrecognized opcode \"%s\"", s);
			continue;
		}
		printf("%-24s", s);
		s = opcode_table[x >> 24];
		if (!s) {
			error("unknown opcode %02x\n", x >> 24);
			printf("%08x  # unknown\n", x & ~0x3f);
			continue;
		}
		printf("%08x  %-12s", x & ~0x3f, s);
		switch (x & 0x3f) {
			case ARG_NONE:	/* (halt, rfe, srb_save, srb_restore, serialize, nop) */
				break;
			case ARG_R:		/* reg (loopentry) */
				fputs("reg1", stdout);
				break;
			case ARG_RO:		/* reg[,reg] (jmpa) */
				fputs("reg2[,reg1]", stdout);
				break;
			case ARG_RR:		/* reg,reg */
				fputs("reg2,reg1", stdout);
				break;
			case ARG_U16R:	/* uimm16,reg (loadcons, geti, puti, syscall, trap) */
				fputs("uimm16,reg1", stdout);
				break;
			case ARG_S16R:	/* simm16,reg (loadconsx) */
				fputs("simm16,reg1", stdout);
				break;
			case ARG_S17R:	/* simm17,reg (loadaddri) */
				fputs("simm17,reg1", stdout);
				break;
			case ARG_U64R:	/* uimm64,reg (generic loadcons) */
				fputs("uimm64,reg1", stdout);
				break;
			case ARG_S64R:	/* simm64,reg (generic loadconsx) */
				fputs("simm64,reg1", stdout);
				break;
			case ARG_ORR:	/* [reg,]reg,reg (popcount, load[f], store[f]) */
				fputs("[reg3,]reg2,reg1", stdout);
				break;
			case ARG_RRO:	/* reg,reg[,reg] (cjmp) */
				fputs("reg3,reg2[,reg1]", stdout);
				break;
			case ARG_RRR:	/* reg,reg,reg */
				fputs("reg3,reg2,reg1", stdout);
				break;
			case ARG_U8RR:	/* uimm8,reg,reg */
				fputs("uimm8,reg2,reg1", stdout);
				break;
			case ARG_S8RR:	/* simm8,reg,reg (muli, divi) */
				fputs("simm8,reg2,reg1", stdout);
				break;
			case ARG_S9RR:	/* simm9,reg,reg (loadi[f], storei[f]) */
				fputs("simm9,reg2,reg1", stdout);
				break;
			default:
				fputs("# invalid args", stdout);
				error("invalid encoding");
		}
		putchar('\n');
	}
	exit(0);
}

#endif
