/*
 * fcpu_opcodes.h -- F-CPU Opcode Definitions (and more)
 * Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* @(#) $Id: fcpu_opcodes.h,v 1.15 2003/01/08 02:13:36 michael Exp $ */

#ifndef _FCPU_OPCODES_H
#define _FCPU_OPCODES_H

/*
 * ROP2 functions
 */
enum ROP2_function {
	ROP2_FUNC_AND,
	ROP2_FUNC_ANDN,
	ROP2_FUNC_XOR,
	ROP2_FUNC_OR,
	ROP2_FUNC_NOR,
	ROP2_FUNC_XNOR,
	ROP2_FUNC_ORN,
	ROP2_FUNC_NAND,
	ROP2_FUNC_last
};

/*
 * The opcode list
 */
enum fcpu_opcodes {	/* XXX: subject to change! */
    OP_NOP,
    OP_MOVE,
    /*
     * Opcodes 2...3 currently unassigned
     */
    OP_BITOP        = 4,    /* uses 4 opcodes */
    OP_BTST         = OP_BITOP + ROP2_FUNC_AND,
    OP_BCLR         = OP_BITOP + ROP2_FUNC_ANDN,
    OP_BCHG         = OP_BITOP + ROP2_FUNC_XOR,
    OP_BSET         = OP_BITOP + ROP2_FUNC_OR,
    OP_ROP2         = 8,    /* uses 8 opcodes */
    OP_MUX          = OP_ROP2,
    OP_AND          = OP_ROP2 + ROP2_FUNC_AND,
    OP_ANDN         = OP_ROP2 + ROP2_FUNC_ANDN,
    OP_XOR          = OP_ROP2 + ROP2_FUNC_XOR,
    OP_OR           = OP_ROP2 + ROP2_FUNC_OR,
    OP_NOR          = OP_ROP2 + ROP2_FUNC_NOR,
    OP_XNOR         = OP_ROP2 + ROP2_FUNC_XNOR,
    OP_ORN          = OP_ROP2 + ROP2_FUNC_ORN,
    OP_NAND         = OP_ROP2 + ROP2_FUNC_NAND,
    OP_LOADCONS     = 16,   /* uses 4 opcodes */
    OP_LOADCONSX    = 20,   /* uses 4 opcodes */
    OP_LOADCONSX_last = OP_LOADCONSX + 3,   /* do NOT delete this! */
    /*
     * ASU
     */
    OP_ADD,
    OP_SUB,
    OP_ADDSUB,
    /*
     * IMU
     */
    OP_MUL,
    OP_MAC,
    /*
     * IDU
     */
    OP_DIV,
    OP_REM,
    /*
     * SHL
     */
    OP_SHIFTL,
    OP_SHIFTR,
    OP_SHIFTRA,
    OP_BITREV,
    OP_ROTL,
    OP_ROTR,
    OP_MIX,
    OP_EXPAND,
    OP_BYTEREV,
    OP_VSEL,
    /*
     * INC
     */
    OP_INC,
    OP_DEC,
    OP_NEG,
    OP_ABS,
    OP_MAX,
    OP_MIN,
    OP_MINMAX,  /* NEW */
#define OP_SORT OP_MINMAX   /* compatibility */
    OP_CMPG,
    OP_CMPLE,
    OP_SCAN,
    /*
     * L/S
     */
    OP_LOAD,
    OP_STORE,
    OP_LOADF,
    OP_STOREF,
    OP_LOADM,
    OP_STOREM,
    OP_SRB_RESTORE,
    OP_SRB_SAVE,
    OP_CACHEMM,
    /*
     * Control
     */
    OP_JMP,
    OP_LOADADDR,
    OP_LOOP,
    OP_SYSCALL,
    OP_HALT,
    OP_RFE,
    OP_SERIALIZE,
    OP_GET,
    OP_PUT,
    /*
     * POPC
     */
    OP_POPC,
    /*
     * LNS instructions
     */
    OP_LADD,
    OP_LSUB,
    OP_L2INT,
    OP_INT2L,
    /*
     * FP instructions
     */
    OP_FADD,
    OP_FSUB,
    OP_FMUL,
    OP_FMAC,
    OP_FDIV,
    OP_FSQRT,
    OP_FLOG,
    OP_FEXP,
    OP_FADDSUB,
    OP_F2INT,
    OP_INT2F,
    OP_FIAPRX,
    OP_FSQRTIAPRX,
    /*
     * new opcodes
     */
    OP_AMAC,
    OP_WIDEN,
    OP_CSHIFT,
    OP_NABS,
    OP_CLOAD,
    OP_CSTORE,
    OP_D2INT,
    OP_INT2D,
    OP_DSHIFTL,
    OP_DSHIFTR,
    OP_DSHIFTRA,
    OP_DBITREV,
    /*
     * Immediate operations (mirror `normal' instructions above)
     */
#define IMM 128
    OP_BITOPI       = IMM + OP_BITOP,
    OP_BTSTI        = IMM + OP_BTST,
    OP_BCLRI        = IMM + OP_BCLR,
    OP_BCHGI        = IMM + OP_BCHG,
    OP_BSETI        = IMM + OP_BSET,
    OP_ROP2I        = IMM + OP_ROP2,
    OP_ANDI         = IMM + OP_AND,
    OP_ANDNI        = IMM + OP_ANDN,
    OP_XORI         = IMM + OP_XOR,
    OP_ORI          = IMM + OP_OR,
    OP_NORI         = IMM + OP_NOR,
    OP_XNORI        = IMM + OP_XNOR,
    OP_ORNI         = IMM + OP_ORN,
    OP_NANDI        = IMM + OP_NAND,
    OP_ADDI         = IMM + OP_ADD,
    OP_SUBI         = IMM + OP_SUB,
    OP_MULI         = IMM + OP_MUL,
    OP_DIVI         = IMM + OP_DIV,
    OP_REMI         = IMM + OP_REM,
    OP_CMPGI        = IMM + OP_CMPG,
    OP_CMPLEI       = IMM + OP_CMPLE,
    OP_POPCI        = IMM + OP_POPC,
    OP_MAXI         = IMM + OP_MAX,
    OP_MINI         = IMM + OP_MIN,
    OP_SHIFTLI      = IMM + OP_SHIFTL,
    OP_SHIFTRI      = IMM + OP_SHIFTR,
    OP_SHIFTRAI     = IMM + OP_SHIFTRA,
    OP_BITREVI      = IMM + OP_BITREV,
    OP_ROTLI        = IMM + OP_ROTL,
    OP_ROTRI        = IMM + OP_ROTR,
    OP_LOADI        = IMM + OP_LOAD,
    OP_LOADIF       = IMM + OP_LOADF,
    OP_STOREI       = IMM + OP_STORE,
    OP_STOREIF      = IMM + OP_STOREF,
    OP_GETI         = IMM + OP_GET,
    OP_PUTI         = IMM + OP_PUT,
    OP_LOADADDRI    = IMM + OP_LOADADDR,
    /*
     * new opcodes
     */
    OP_MINMAXI      = IMM + OP_MINMAX,
    OP_DSHIFTLI     = IMM + OP_DSHIFTL,
    OP_DSHIFTRI     = IMM + OP_DSHIFTR,
    OP_DSHIFTRAI    = IMM + OP_DSHIFTRA,
    OP_DBITREVI     = IMM + OP_DBITREV,
    OP_VSELI        = IMM + OP_VSEL,
#undef IMM
    OP_last
};

/*
 * Mode and flag bits for individual instructions
 */
enum modes_and_flags {	/* XXX: verify shifts */
	/*
	 * SIMD flag
	 */
	SIMD_FLAG         = 1 << 21,
	/*
	 * Shifter flags
	 */
	SHIFT_HALF_SIMD   = 1 << 20,
	/*
	 * Integer size flags
	 */
	ISIZE_8BIT        = 0 << 22,
	ISIZE_16BIT       = 1 << 22,
	ISIZE_32BIT       = 2 << 22,
	ISIZE_64BIT       = 3 << 22,
	ISIZE_MASK        = 3 << 22,
	/*
	 * FP size flags
	 */
	FSIZE_SINGLE      = 0 << 22,
	FSIZE_DOUBLE      = 1 << 22,
	FSIZE_2           = 2 << 22,	/* unassigned */
	FSIZE_3           = 3 << 22,	/* unassigned */
	FSIZE_MASK        = 3 << 22,
	/*
	 * ROP2 combine modes
	 */
	ROP2_MODE_DIRECT  = 0 << 18,
	ROP2_MODE_AND     = 1 << 18,
	ROP2_MODE_OR      = 2 << 18,
	ROP2_MODE_MUX     = 3 << 18,
	ROP2_MODE_MASK    = 3 << 18,
	/*
	 * Rounding modes (int2l, l2int, int2f, f2int)
	 */
	ROUND_NEAREST     = 0 << 18,	/* round to nearest/even */
	ROUND_TRUNC       = 1 << 18,	/* round towards zero */
	ROUND_FLOOR       = 2 << 18,	/* round towards -infinity (down) */
	ROUND_CEIL        = 3 << 18,	/* round towards +infinit (up) */
	ROUND_MASK        = 3 << 18,
	/*
	 * Condition Codes (move, jmp)
	 */
	CC_ZERO           = 0 << 19,
	CC_NAN            = 1 << 19,
	CC_MSB            = 2 << 19,
	CC_LSB            = 3 << 19,
	CC_MASK           = 3 << 19,
	CC_NOT            = 4 << 19,
	CC_NOT_ZERO       = CC_NOT + CC_ZERO,
	CC_NOT_NAN        = CC_NOT + CC_NAN,
	CC_NOT_MSB        = CC_NOT + CC_MSB,
	CC_NOT_LSB        = CC_NOT + CC_LSB,
	/*
	 * OP_ADD modes
	 */
	ADD_MODE_NORMAL   = 0 << 18,
	ADD_MODE_CARRY    = 1 << 18,
	ADD_MODE_SATURATE = 2 << 18,
	ADD_MODE_3        = 3 << 18,	/* unassigned */
	ADD_MODE_MASK     = 3 << 18,
	/*
	 * OP_SUB modes
	 */
	SUB_MODE_NORMAL   = 0 << 18,
	SUB_MODE_BORROW   = 1 << 18,
	SUB_MODE_FLOOR    = 2 << 18,
	SUB_MODE_3        = 3 << 18,	/* unassigned */
	SUB_MODE_MASK     = 3 << 18,
	/*
	 * OP_MUL flags
	 */
	MUL_HIGH          = 1 << 18,
	MUL_SIGNED        = 1 << 19,
	/*
	 * OP_MAC flags
	 */
	MAC_LOW           = 0 << 18,
	MAC_HIGH          = 1 << 18,
	MAC_SIGNED        = 1 << 19,
	/*
	 * OP_DIV flags
	 */
	DIV_SIGNED        = 1 << 19,
	DIV_REMAINDER     = 1 << 20,
	/*
	 * OP_SCAN flags
	 */
	SCAN_REVERSE      = 1 << 18,
	SCAN_NEGATE       = 1 << 19,
	/*
	 * OP_BITREV[I] flags
	 */
	BITREV_OR         = 1 << 20,
	/*
	 * OP_CMP flags
	 */
	CMP_SIGNED        = 1 << 20,	/* NEW */
	/*
	 * OP_MIX flags
	 */
	MIX_BOTH          = 0 << 18,	/* NEW */
	MIX_LOW           = 1 << 18,	/* NEW */
	MIX_HIGH          = 2 << 18,	/* NEW */
	/*
	 * OP_EXPAND flags
	 */
	EXPAND_BOTH       = 0 << 18,	/* NEW */
	EXPAND_LOW        = 1 << 18,	/* NEW */
	EXPAND_HIGH       = 2 << 18,	/* NEW */
	/*
	 * OP_F* `x' flag
	 */
	FPOP_X            = 1 << 20,
	/*
	 * OP_LOAD/OP_STORE flags
	 */
	LS_BIG_ENDIAN     = 1 << 21,
	LS_STREAM_0       = 0 << 18,
	LS_STREAM_1       = 1 << 18,
	LS_STREAM_2       = 2 << 18,
	LS_STREAM_3       = 3 << 18,
	LS_STREAM_4       = 4 << 18,
	LS_STREAM_5       = 5 << 18,
	LS_STREAM_6       = 6 << 18,
	LS_STREAM_7       = 7 << 18,
	/*
	 * OP_LOADADDR flags
	 */
	LOADADDR_DATA     = 1 << 23,
	/*
	 * OP_SYSCALL flags
	 */
	SYSCALL_TRAP      = 1 << 23,
	/*
	 * OP_CACHEMM flags
	 */
	CACHEMM_FLUSH     = 0 << 21,
	CACHEMM_PREFETCH  = 1 << 21,
	CACHEMM_LOCK      = 1 << 20,
	CACHEMM_COMPRESS  = 1 << 19,
	CACHEMM_LEVEL_0   = 0 << 16,
	CACHEMM_LEVEL_1   = 1 << 16,
	CACHEMM_LEVEL_2   = 2 << 16,
	CACHEMM_LEVEL_3   = 3 << 16,
	CACHEMM_LEVEL_4   = 4 << 16,
	CACHEMM_LEVEL_5   = 5 << 16,
	CACHEMM_LEVEL_6   = 6 << 16,
	CACHEMM_LEVEL_7   = 7 << 16,
	/*
	 * OP_SERIALIZE flags
	 */
	SERIALIZE_MEMORY  = 1 << 23,
	SERIALIZE_EXEC    = 1 << 22,
	SERIALIZE_SRB     = 1 << 21,
	/*
	 * OP_CSHIFT flags
	 */
	CSHIFT_LEFT       = 0 << 19,
	CSHIFT_RIGHT      = 1 << 19,
};

/*
 * Argument encoding types
 * (values don't matter much -- used only internally)
 */
enum instruction_variants {
    /* 0 arguments */
    ARG_NONE,       /* (halt, rfe, srb_save, srb_restore, serialize, nop) */
    /* 1 argument */
    ARG_R,          /* reg (loopentry) */
    /* 1...2 argument(s) */
    ARG_RO,         /* reg[,reg] (jmpa) */
    /* 2 arguments */
    ARG_RR,         /* reg,reg (common) */
    ARG_U16R,       /* uimm16,reg (loadcons.n, geti, puti, syscall, trap) */
    ARG_S16R,       /* simm16,reg (loadconsx.n) */
    ARG_S17R,       /* simm17,reg (loadaddri) */
    ARG_U64R,		/* uimm64,reg (generic loadcons) */
    ARG_S64R,		/* simm64,reg (generic loadconsx) */
    /* 2...3 arguments */
    ARG_ORR,        /* [reg,]reg,reg (popcount, load[f], store[f]) */
    ARG_RRO,        /* reg,reg[,reg] (jmpa{cc}) */
    /* 3 arguments */
    ARG_RRR,        /* reg,reg,reg (common) */
    ARG_U8RR,       /* uimm8,reg,reg (common) */
    ARG_S8RR,       /* simm8,reg,reg (muli, divi, modi?) */
    ARG_S9RR,       /* simm9,reg,reg (loadi[f], storei[f]) */
    /* 3 register arguments, r2 <-> r3 swapped */
    ARG_231,        /* reg,reg,reg */
    /* off-by-one immediate compare */
    ARG_S8RRoff,	/* simm8,reg,reg */
    ARG_U8RRoff,	/* uimm8,reg,reg */
    /* The End */
    ARG_last
};

/*
 * 32-bit Instruction words are stored in big-endian format
 */
#define INSTRUCTION_BIG_ENDIAN  1

/*
 * Functions
 */
extern int fcpu_encode_instruction(const char *name);
extern int fcpu_decode_instruction(char *buf, size_t len, unsigned op);

#endif /* _FCPU_OPCODES_H */
