/*
 * append.c - append a (named) section to an ELF file.
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: append.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

Elf_Scn*
et_append_section(Elf *elf, const char *name, unsigned type, size_t pos) {
	Elf_Data *data;
	Elf_Scn *scn;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	GElf_Addr shstrtab;
	size_t off;

	/*
	 * get ELF header for the string table index
	 */
	if (!gelf_getehdr(elf, &eh)) {
		return NULL;
	}
	off = 0;
	if (name && *name) {
		/*
		 * get string table section
		 */
		if (!(shstrtab = et_get_shstrndx(elf))) {
			return NULL;
		}
		if (!(scn = elf_getscn(elf, shstrtab))) {
			return NULL;
		}
		if (!gelf_getshdr(scn, &sh)) {
			return NULL;
		}
		/*
		 * calculate string offset (== current section size)
		 */
		data = NULL;
		elf_errno();
		while ((data = elf_getdata(scn, data))) {
			if (data->d_align > 1) {
				off += data->d_align - 1;
				off -= off % data->d_align;
			}
			off += data->d_size;
		}
		if (elf_errmsg(0)) {
			return NULL;
		}
		/*
		 * add new string
		 */
		if (!(data = elf_newdata(scn))) {
			return NULL;
		}
		data->d_buf = xstrdup(name);
		data->d_size = strlen(name) + 1;
		data->d_off = off;
		data->d_align = 1;
		/*
		 * update section header
		 */
		sh.sh_size = off + data->d_size;
		if (!gelf_update_shdr(scn, &sh)) {
			return 0;
		}
	}
	if (pos == 0) {
		/*
		 * find offset for new section
		 */
		elf_errno();
		pos = gelf_fsize(elf, ELF_T_EHDR, 1, eh.e_version);
		scn = NULL;
		while ((scn = elf_nextscn(elf, scn))) {
			if (!gelf_getshdr(scn, &sh)) {
				return NULL;
			}
			if (sh.sh_type == SHT_NOBITS) {
				if (pos < sh.sh_offset) {
					pos = sh.sh_offset;
				}
			}
			else if (sh.sh_type != SHT_NULL) {
				if (pos < sh.sh_offset + sh.sh_size) {
					pos = sh.sh_offset + sh.sh_size;
				}
			}
		}
		if (elf_errmsg(0)) {
			return NULL;
		}
	}
	/*
	 * create new section
	 */
	if (!(scn = elf_newscn(elf))) {
		return NULL;
	}
	if (!gelf_getshdr(scn, &sh)) {
		return NULL;
	}
	/*
	 * fill in the section header
	 */
	sh.sh_name = off;
	sh.sh_type = type;
	sh.sh_flags = 0;			/* to be set by application */
	sh.sh_addr = 0;				/* to be set by application */
	sh.sh_offset = pos;
	sh.sh_size = 0;				/* to be set by application */
	sh.sh_link = SHN_UNDEF;		/* to be set by application */
	sh.sh_info = 0;				/* to be set by application */
	sh.sh_addralign = 1;		/* to be set by application */
	sh.sh_entsize = 0;			/* to be set by application */
	if (!gelf_update_shdr(scn, &sh)) {
		return NULL;
	}
	return scn;
}
