/*
 * strip.c - SVR4 style strip(1).
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <sys/stat.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: strip.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static int strip_symbols = 1;
static int strip_debug = 1;

#define mark_section(i) \
	do{if(!flags[i]){flags[i]=1;count++;}}while(0)

static int
strip_elf(const char *name, Elf *elf, Elf *nelf) {
	static Elf_Scn **table = NULL;
	static int *flags = NULL;
	GElf_Ehdr eh;
	GElf_Shdr sh;
	GElf_Addr shnum;
	GElf_Addr shstrndx;
	char *scnname;
	int count;
	unsigned i, j, k;

	if (!gelf_getehdr(elf, &eh)) {
		file_error(name, "file format not recognized");
		return -1;
	}
	if (!(shnum = et_get_shnum(elf)) && eh.e_shoff) {
		file_error(name, "no section header table at offset 0x%lx",
			(unsigned long)eh.e_shoff);
		return -1;
	}
	if (!shnum) {
		file_error(name, "no sections");
		return -1;
	}
	shstrndx = et_get_shstrndx(elf);
	table = (Elf_Scn**)xrealloc(table, shnum * sizeof(*table));
	flags = (int*)xrealloc(flags, shnum * sizeof(*flags));
	for (i = 0; i < shnum; ++i) {
		flags[i] = 0;
		if (!(table[i] = elf_getscn(elf, i))) {
			file_error(name, "elf_getscn: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_getshdr(table[i], &sh)) {
			file_error(name, "gelf_getshdr: %s", elf_errmsg(-1));
			return -1;
		}
		if (sh.sh_type == SHT_NULL || (sh.sh_flags & SHF_ALLOC)) {
			mark_section(i);
		}
	}
	mark_section(0);
	do {
		count = 0;
		for (i = 1; i < shnum; ++i) {
			if (!gelf_getshdr(table[i], &sh)) {
				return -1;
			}
			switch (sh.sh_type) {
				case SHT_PROGBITS:
					if (!(scnname = elf_strptr(elf, shstrndx, sh.sh_name))) {
						return -1;
					}
					else if (!strcmp(scnname, ".line")) {
						/* always strip .line section */
						break;
					}
					else if (strip_debug && !strncmp(scnname, ".debug", 6)) {
						/* dwarf format */
						break;
					}
					else if (!strcmp(scnname, ".stab")) {
						/* gcc stabs format */
						if (strip_debug) {
							break;
						}
						/* do not strip .stabstr section */
						mark_section(sh.sh_link);
					}
					else if (!strcmp(scnname, ".stab.excl")) {
						/* solaris format */
						if (strip_debug) {
							break;
						}
						/* do not strip .stab.exclstr section */
						mark_section(sh.sh_link);
					}
					else if (!strcmp(scnname, ".stab.index")) {
						/* solaris format */
						if (strip_debug) {
							break;
						}
						/* do not strip .stab.indexstr section */
						mark_section(sh.sh_link);
					}
					mark_section(i);
					break;
				case SHT_REL:
				case SHT_RELA:
					if (!flags[i] && !flags[sh.sh_info]) {
						break;
					}
					/* flow through */
				case SHT_DYNAMIC:
				case SHT_HASH:
					mark_section(sh.sh_link);
					mark_section(i);
					/* flow through */
				case SHT_STRTAB:
					break;
				case SHT_SYMTAB:
					if (strip_symbols) {
						if (eh.e_type != ET_REL && eh.e_type != ET_DYN) {
							break;
						}
					}
					/* flow through */
				case SHT_DYNSYM:
					mark_section(sh.sh_link);
					/* flow through */
				default:
					mark_section(i);
					break;
			}
		}
	}
	while (count);
	for (i = j = k = 0; i < shnum; i++) {
		if (flags[i]) {
			if (i != shstrndx) {
				flags[i] = k++;
			}
		}
		else {
			flags[i] = SHN_UNDEF;
			table[j++] = table[i];
		}
	}
	flags[shstrndx] = k;
	if (j) {
		if (!et_delete_sections(elf, nelf, table, j)) {
			file_error(name, "can't delete sections");
			return -1;
		}
		if (et_scnxlate(nelf, flags, i)) {
			file_error(name, "can't change section references");
			return -1;
		}
		if (et_layout(nelf)) {
			file_error(name, "can't change file layout: %s", elf_errmsg(-1));
			return -1;
		}
	}
	else if (!et_copy(elf, nelf)) {
		file_error(name, "can't copy file: %s", elf_errmsg(-1));
		return -1;
	}
	elf_flagelf(nelf, ELF_C_SET, ELF_F_LAYOUT);
	if (elf_update(nelf, ELF_C_WRITE) == (off_t)-1) {
		file_error(name, "write failed: %s", elf_errmsg(-1));
		return -1;
	}
	return 0;
}

static int
process_elf(const char *name, Elf *elf, int ofd) {
	Elf *nelf = NULL;
	char *img;
	int err;
	size_t len;

	assert(ofd != -1);
	if (elf_kind(elf) == ELF_K_ELF) {
		/*
		 * Process ELF file
		 */
		if (!(nelf = elf_begin(ofd, ELF_C_WRITE, NULL))) {
			file_error(name, "elf_begin: %s", elf_errmsg(-1));
			return 1;
		}
		err = strip_elf(name, elf, nelf);
		elf_end(nelf);
		return err;
	}
	/*
	 * If this is not an archive member, complain
	 */
	if (!elf_getarhdr(elf)) {
		file_error(name, "not an ELF file");
		return 1;
	}
	/*
	 * Copy file verbatim
	 */
	elf_errno();
	if (!(img = elf_rawfile(elf, &len))) {
		if ((err = elf_errno())) {
			file_error(name, "elf_rawfile: %s", elf_errmsg(err));
			return 1;
		}
	}
	if (len && xwrite(ofd, img, len)) {
		return 1;
	}
	return 0;
}

static const char usage[] =
	"usage: %s [-bBlrxV] [file...]\n"
	"  -b    obsolete (ignored)\n"
	"  -B    backup files before modifying them\n"
	"  -l    strip line number information only\n"
	"  -r    obsolete (ignored)\n"
	"  -V    display program version\n"
	"  -x    do not strip the symbol table\n"
	;

int
main(int argc, char **argv) {
	char *progname = NULL;
	int do_version = 0;
	int do_backup = 0;
	int err;
	int c;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		*argv = ++progname;
	}
	else if (!(progname = *argv)) {
		progname = "strip";
	}
	setprogname(progname);

	while ((c = getopt(argc, argv, "bBlrVx")) != EOF) {
		switch (c) {
			case 'B':
				do_backup = 1;
				break;
			case 'l':
				strip_debug = 0;
				/* flow through */
			case 'x':
				strip_symbols = 0;
				break;
			case 'V':
				do_version++;
				break;
			case '?':
				fprintf(stderr, usage, progname);
				exit(1);
			case 'b':	/* ignored */
			case 'r':	/* ignored */
			default:
				break;
		}
	}
	if (do_version) {
		show_version("strip");
		if (optind == argc) {
			exit(0);
		}
	}
	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}
	if (optind == argc) {
		err = et_modify_file("a.out", process_elf, do_backup);
	}
	else {
		err = 0;
		while (optind < argc) {
			err |= et_modify_file(argv[optind++], process_elf, do_backup);
		}
	}
	exit(err);
}
