/*
 * eascan.l -- Lexical Scanner for ELF Assembler
 * Copyright (C) 2001, 2002 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%{
static const char rcsid[] = "@(#) $Id: eascan.l,v 1.32 2002/12/31 18:40:23 michael Exp $";
%}

/* white space (excluding newline) */
ws			([ \t\r\f\v]+)

/* C-style identifiers w/ extension chars "." and "$" */
digit		([0-9])
letter		([A-Z_a-z])
xdigit		({digit}|[A-Fa-f])
idchars		({letter}|{digit}|[$.])
ident		({letter}{idchars}*|[.]{idchars}+)

/* C-style integers, but w/o size suffix */
binary		(0[Bb][01]+)
octal		(0[0-7]*)
decimal		([1-9]{digit}*)
hex			(0[Xx]{xdigit}+)
integer		({binary}|{octal}|{decimal}|{hex})

/* C-style floating point */
mantissa	({digit}+[.]{digit}*|[.]{digit}+)
exponent	([Ee][-+]?{digit}+)
flonum		({mantissa}{exponent}?|{digit}+{exponent})

xmantissa	({xdigit}+[.]?|{xdigit}*[.]{xdigit}+)
xexponent	([Pp][-+]?{digit}+)
xflonum		([0][Xx]{xmantissa}{xexponent})

/* all numbers */
number		({integer}|{flonum}|{xflonum})

/* C99 universal character names (currently unused) */
xquad		({xdigit}{4})
ucname		(\\(U{xquad}|u){xquad})

/* C-style string and character constants */
esc			(\\(["'\\abfnrtv]|[0-7]{1,3}|x{xdigit}+))
cchar		([^\000-\037'\\]|{esc})
schar		([^\000-\037"\\]|{esc})
cconst		(\'{cchar}\')
sconst		(\"({schar}|\\\n)*\")

/* preprocessor stuff */
cppline		("#"{ws}?("line"{ws})?{decimal}{ws}?{sconst}({ws}[1-4])*)

%{

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#include <stdarg.h>
#include <stdio.h>

#include <libelf.h>

#include <eatypes.h>
#include <eapars.h>

static int yywrap(void);

static int lookup_symbol(void);
static int string_const(void);
static int char_const(void);
static int copytext(int token);

static void parse_line(void);

unsigned long current_line = 1;	/* for error messages (set by parser) */
unsigned long lineno = 1;

static size_t cond_level = 0;
static size_t include_level = 0;

#define add_list()	do{/**/}while(0)

%}

%option pointer
%option stack

%x com
%x sk1 sk2

%%

<sk1>{
<<EOF>>			error("missing \".endif\""); BEGIN(0); return ENDIF;
{ws}			add_list();
\\\n			add_list(); BEGIN(sk2); lineno++;
\n				add_list(); lineno++; return '\n';
".if"			add_list(); BEGIN(sk2); cond_level++;
".ifdef"		add_list(); BEGIN(sk2); cond_level++;
".ifndef"		add_list(); BEGIN(sk2); cond_level++;
".ifnotdef"		add_list(); BEGIN(sk2); cond_level++;
".elif"			add_list(); if (!cond_level) { BEGIN(0); return ELIF; }
".else"			add_list(); if (!cond_level) { BEGIN(0); return ELSE; }
".endif"		add_list(); if (!cond_level) { BEGIN(0); return ENDIF; } BEGIN(sk2); cond_level--;
{ident}			add_list(); BEGIN(sk2);
"/*"			add_list(); yy_push_state(com);
.				add_list(); BEGIN(sk2);
}

<sk2>{
<<EOF>>			error("missing \".endif\""); BEGIN(0); return ENDIF;
\\\n			add_list(); lineno++;
\n				add_list(); BEGIN(sk1); lineno++; return '\n';
"/*"			add_list(); yy_push_state(com);
.				add_list();
}

{ws}			add_list();
\\\n			add_list(); lineno++;
\n				add_list(); lineno++; return '\n';

^"#EOF"			if (yywrap()) yyterminate();	/* YG special exit */
^{cppline}$		parse_line();
^"#".*			add_list();
"//".*			add_list();
"/*"			add_list(); yy_push_state(com);
<com><<EOF>>	add_list(); BEGIN(0); error("end-of-file in comment");
<com>"*/"		add_list(); yy_pop_state();
<com>\n			add_list(); lineno++;
<com>.			add_list();

^[0-9]/":"		add_list(); return copytext(IDENTIFIER);
[0-9][bf]		add_list(); return copytext(IDENTIFIER);

{sconst}		add_list(); return string_const();
{cconst}		add_list(); return char_const();
{number}		add_list(); return copytext(NUMBER);

{ident}			add_list(); return lookup_symbol();

"||"			add_list(); return PAR;
"!="			add_list(); return NEQ;
"=="			add_list(); return EQL;
"<="			add_list(); return LEQ;
">="			add_list(); return GEQ;
"<<"			add_list(); return SHL;
">>"			add_list(); return SHR;

.				add_list(); return *yytext;

%%

void suspend_scanner(void) { BEGIN(sk1); }

#include <assert.h>

static const char *filename = NULL;

struct input_stack {
	struct input_stack	*link;
	const char			*name;
	unsigned long		lineno;
	YY_BUFFER_STATE		buffer;
};

static struct input_stack *inputs = NULL;

void
set_input_file(const char *name, FILE *fp) {
	include_level = 0;
	filename = name;
	BEGIN(0);
	yyrestart(fp);
	lineno = 1;
}

void
push_input_file(const char *name, FILE *fp) {
	struct input_stack *p;

	/*
	 * suspend current input file
	 */
	p = xmalloc(sizeof(struct input_stack));
	p->link = inputs;
	p->name = filename;
	p->lineno = lineno;
	p->buffer = YY_CURRENT_BUFFER;
	inputs = p;
	/*
	 * switch to new file
	 */
	include_level++;
	yy_switch_to_buffer(yy_create_buffer(fp, YY_BUF_SIZE));
	BEGIN(0);
	filename = name;
	lineno = 1;
}

/*
 * handle end-of-file
 */
static int
yywrap(void) {
	struct input_stack *p;

	if (!(p = inputs)) {
		assert(include_level == 0);
		return 1;
	}
	assert(include_level);
	include_level--;
	fclose(yyin);	/* flex apparently never does this itself! */
	yy_delete_buffer(YY_CURRENT_BUFFER);
	yy_switch_to_buffer(p->buffer);
	filename = p->name;
	lineno = p->lineno;
	inputs = p->link;
	xfree(p);
	BEGIN(0);
	return 0;
}

/*
 * Yacc/Bison wants this
 */
void
yyerror(const char *msg) {
	if (yytext && *yytext) {
		error("%s before \"%s\"", msg, yytext);
	}
	else {
		error("unexpected end of file", msg);
	}
}

#ifndef STV_HIDDEN
#define STV_DEFAULT		0
#define STV_INTERNAL	1
#define STV_HIDDEN		2
#define STV_PROTECTED	3
#endif

static struct {
	const char	*name;
	int			token;
	long		value;
} pseudo_table[] = {
	/* NOTE: table MUST BE sorted ASCIIbetically! */
	{ ".addr",        IDATA,      8 },
	{ ".align",       ALIGN,      0 },
	{ ".ascii",       SDATA,      0 },
	{ ".asciz",       SDATA,      1 },
	{ ".balign",      ALIGN,      1 },
	{ ".balignl",     ALIGN,      4 },
	{ ".balignw",     ALIGN,      2 },
	{ ".be_addr",     IDATA,     -8 },
	{ ".be_double",   FDATA,     -8 },
	{ ".be_dword",    IDATA,     -8 },
	{ ".be_float",    FDATA,     -4 },
	{ ".be_half",     IDATA,     -2 },
	{ ".be_int",      IDATA,     -4 },
	{ ".be_ldouble",  FDATA,    -16 },
	{ ".be_llong",    IDATA,    -16 },
	{ ".be_long",     IDATA,     -8 },
	{ ".be_qword",    IDATA,    -16 },
	{ ".be_short",    IDATA,     -2 },
	{ ".be_single",   FDATA,     -4 },
	{ ".be_tfloat",   FDATA,    -10 },
	{ ".be_word",     IDATA,     -4 },
	{ ".block",       SPACE,      0 },
	{ ".bss",         NAMESCN,    0 },
	{ ".byte",        IDATA,      1 },
	{ ".comm",        COMMON,     0 },
	{ ".data",        NAMESCN,    0 },
	{ ".double",      FDATA,      8 },
	{ ".dword",       IDATA,      8 },
	{ ".elif",        ELIF,       0 },
	{ ".else",        ELSE,       0 },
	{ ".endif",       ENDIF,      0 },
	{ ".endm",        ENDM,       0 },
	{ ".endr",        ENDR,       0 },
	{ ".equ",         ASSIGN,     0 },
	{ ".equiv",       ASSIGN,     1 },
	{ ".err",         ERROR,      0 },
	{ ".exitm",       EXITM,      0 },
	{ ".export",      EXPORT,     STB_GLOBAL },	// synonym for .global
	{ ".extern",      EXTERN,     0 },
	{ ".file",        FNAME,      0 },
	{ ".fill",        FILL,       0 },
	{ ".float",       FDATA,      4 },
	{ ".global",      EXPORT,     STB_GLOBAL },
	{ ".globl",       EXPORT,     STB_GLOBAL },
	{ ".half",        IDATA,      2 },
	{ ".hidden",      VISIBILITY, STV_HIDDEN },	// .hidden symbol
	{ ".ident",       IDENT,      0 },
	{ ".if",          IF,         0 },
	{ ".ifdef",       IFDEF,      0 },
	{ ".ifndef",      IFNDEF,     0 },
	{ ".ifnotdef",    IFNDEF,     0 },
	{ ".include",     INCLUDE,    0 },
	{ ".int",         IDATA,      4 },
	{ ".internal",    VISIBILITY, STV_INTERNAL },	// .internal symbol
	{ ".irp",         IRP,        0 },	// .irp symbol { , expr }
	{ ".irpc",        IRPC,       0 },	// .irp symbol [ , characters ]
	{ ".lcomm",       COMMON,     1 },
	{ ".ldouble",     FDATA,     16 },
	{ ".le_addr",     IDATA,      8 },
	{ ".le_double",   FDATA,      8 },
	{ ".le_dword",    IDATA,      8 },
	{ ".le_float",    FDATA,      4 },
	{ ".le_half",     IDATA,      2 },
	{ ".le_int",      IDATA,      4 },
	{ ".le_ldouble",  FDATA,     16 },
	{ ".le_llong",    IDATA,     16 },
	{ ".le_long",     IDATA,      8 },
	{ ".le_qword",    IDATA,     16 },
	{ ".le_short",    IDATA,      2 },
	{ ".le_single",   FDATA,      4 },
	{ ".le_tfloat",   FDATA,     10 },
	{ ".le_word",     IDATA,      4 },
	{ ".llong",       IDATA,     16 },
	{ ".local",       EXPORT,     STB_LOCAL },
	{ ".long",        IDATA,      8 },
	{ ".macro",       MACRO,      0 },	// .macro symbol { [ , ] arg [ = default ] }
	{ ".org",         ORG,        0 },
	{ ".p2align",     ALIGN,     -1 },
	{ ".p2alignl",    ALIGN,     -4 },
	{ ".p2alignw",    ALIGN,     -2 },
	{ ".popsection",  PREVSCN,    1 },
	{ ".previous",    PREVSCN,    0 },
	{ ".protected",   VISIBILITY, STV_PROTECTED },	// .protected symbol
	{ ".purgem",      PURGEM,     0 },	// .purgem macro
	{ ".pushsection", SECTION,    0 },
	{ ".qword",       IDATA,     16 },
	{ ".rept",        REPT,       0 },	// .rept abs_expr
	{ ".section",     SECTION,    0 },
	{ ".set",         ASSIGN,     0 },
	{ ".short",       IDATA,      2 },
	{ ".single",      FDATA,      4 },
	{ ".size",        SIZE,       0 },
	{ ".skip",        SPACE,      0 },
	{ ".sleb128",     LEB128,     1 },
	{ ".space",       SPACE,      0 },
	{ ".stabd",       STABD,      0 },
	{ ".stabn",       STABN,      0 },
	{ ".stabs",       STABS,      0 },
	{ ".string",      SDATA,      2 },
	{ ".symver",      SYMVER,     0 },
	{ ".text",        NAMESCN,    0 },
	{ ".tfloat",      FDATA,     10 },
	{ ".type",        TYPE,       0 },
	{ ".uleb128",     LEB128,     0 },
	{ ".version",     VERS,       0 },
	{ ".weak",        EXPORT,     STB_WEAK },
	{ ".weakext",     WEAKEXT,    0 },
	{ ".word",        IDATA,      4 },
#if 0	/* stuff not implemented in the parser */
	/* listing control */
	{ ".eject",       LISTING,    LIST_EJECT },
	{ ".list",        LISTING,    LIST_ON },
	{ ".nolist",      LISTING,    LIST_OFF },
	{ ".psize",       PSIZE,      0 },	// .psize lines , columns
	{ ".sbttl",       TITLE,      1 },	// .sbttl string
	{ ".title",       TITLE,      0 },	// .title string
	/* target selection */
	{ ".arch",        CPU,        0 },	// .arch symbol (synonym for .cpu)
	{ ".cpu",         CPU,        0 },	// .cpu symbol
	/* misc. stuff */
	{ ".abort",       ABORT,      0 },	// premature abort
	{ ".app-file",    FNAME,      0 },	// not a valid identifier
	{ ".end",         END,        0 },	// .end start_symbol // standalone only
	{ ".exit",        EXIT,       0 },
	{ ".import",      IMPORT,     0 },	// .import symbol (ignored)
	{ ".linkonce",    LINKONCE,   0 },	// .linkonce [ discard | one_only | same_size | same_contents ]
#endif
};

#if ENABLE_MACROS
static macro_t *find_macro(const char *name);
#endif

static int
lookup_symbol(void) {
	size_t l, r, m;
	int i;

	if (yytext[0] == '.') {
		l = 0;
		r = sizeof(pseudo_table) / sizeof(pseudo_table[0]);
		while (l < r) {
			m = l + ((r - l) >> 1);
			i = strcmp(yytext, pseudo_table[m].name);
			if (i < 0) {
				r = m;
			}
			else if (i > 0) {
				l = m + 1;
			}
			else {
				if (pseudo_table[m].token == NAMESCN) {
					yylval.sval = pseudo_table[m].name;
				}
				else {
					yylval.lval = pseudo_table[m].value;
				}
				return pseudo_table[m].token;
			}
		}
	}
#if ENABLE_MACROS
	if (find_macro(yytext)) {
		return copytext(MACRO_NAME);
	}
#endif
	return copytext(IDENTIFIER);
}

static int
get_escape(char **sp, char term) {
	static const char table[26] = "\a\bcde\fghijklm\nopq\rs\tu\vwxyz";
	static const char *what[] = { "string", "character constant" };
	char c, c2, *s = *sp;

	while ((c = *s++) == '\\' && *s == '\n') {
		/* skip backslash-newline sequence */
		*sp = ++s;
	}
	if (c == term) {
		/* regular end of string */
		return EOF;
	}
	if (!c) {
		error("premature end of %s", what[term == '\'']);
		return EOF;
	}
	if (c == '\\') {
		if ((c = *s++) == 'x') {
			/* hex escape */
			c = strtoul(s, &s, 16);
		}
		else if (c - 'a' < 26u) {
			/* backslash-letter */
			c = table[c - 'a'];
		}
		else if (c - '0' < 10u) {
			/* octal escape */
			c = c - '0';
			if ((c2 = *s - '0') < 10u) {
				c = 8 * c + c2;
				s++;
				if ((c2 = *s - '0') < 10u) {
					c = 8 * c + c2;
					s++;
				}
			}
		}
		else if (c != '\'' && c != '\"' && c != '\\') {
			error("invalid escape in %s", what[term == '\'']);
		}
	}
	*sp = s;
	return (unsigned char)c;
}

static int
string_const(void) {
	char *s, *t;
	int c;

	yylval.sval = s = xmalloc(yyleng);
	t = yytext + 1;
	while ((c = get_escape(&t, '\"')) != EOF) {
		*s++ = c;
	}
	*s = '\0';
	assert(s - yylval.sval < yyleng);
	if (t != yytext + yyleng - 1) {
		error("invalid string constant");
	}
	return STRING;
}

static int
char_const(void) {
	char *t;
	int c;

	t = yytext + 1;
	c = get_escape(&t, '\'');
	if (c == EOF || t != yytext + yyleng - 1) {
		error("invalid character constant");
	}
	yylval.lval = c & 0xff;
	return CHARACTER;
}

static int
copytext(int token) {
	yylval.sval = strcpy(xmalloc(yyleng + 1), yytext);
	return token;
}

static void
parse_line(void) {
	char *s, *t;
	int c;

	assert(*yytext = '#');
	for (s = yytext + 1; (c = *s) && (unsigned)(c - '0') >= 10; s++);
	lineno = strtoul(s, &s, 0);
	while ((c = *s++) && c != '"');
	filename = t = xmalloc((yytext + yyleng) - s);
	while ((c = get_escape(&s, '\"')) != EOF) {
		*t++ = c;
	}
	*t = '\0';
}

/*
 * Error reporting
 */

static void
print_msg(FILE *fp, const char *label, const char *fmt, va_list ap) {
	if (!filename) {
		fprintf(fp, "as: %s", label);
	}
	else if (!*filename) {
		fprintf(fp, "%s:%lu: %s", "<stdin>", current_line, label);
	}
	else {
		fprintf(fp, "%s:%lu: %s", filename, current_line, label);
	}
	vfprintf(fp, fmt, ap);
	fputs("\n", fp);
}

int errors = 0;
int warnings = 0;

void
warn(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "warning: ", fmt, ap);
	va_end(ap);
	warnings++;
}

void
error(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "", fmt, ap);
	va_end(ap);
	errors++;
}

void
fatal(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "fatal error: ", fmt, ap);
	va_end(ap);
	/* no way to recover */
	exit(1);
}
