/*
 * eapars.y -- YACC Parser for ELF Assembler
 * Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%{

static const char rcsid[] = "@(#) $Id: eapars.y,v 1.82 2003/01/01 20:00:20 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <libelf.h>
#include <assert.h>

#include <eatypes.h>

/* run-time parameters */

int opt_L = 0;	/* keep local (.L*) symbols */
int opt_n = 0;	/* suppress all warnings */
const char *opt_o = "a.out";
const char *opt_O = "elf";
char opt_Q = 'n';	/* 'y' disables assembler version comment */
const char *source_name = "(standard input)";

int all_start_symbols = 0;	/* always create section start symbols? */

/* target-specific data */

typedef struct target_props target_t;

typedef struct reloc reloc_t;

struct target_props {
	long		(*reloc_type)(const target_t*, reloc_t *rel, long size, int pcrel);
	long		(*encode_register)(const target_t*, const char *name);
	void		(*emit)(const target_t*, const char *name, expr_t *expr);
	void		(*emit_stab)(const target_t*, const char*, word_t, word_t, word_t, word_t);
	expr_t*		(*memory_operand)(const target_t*, expr_t *expr);
	size_t		text_alignment;		/* instruction size <= x <= cache line size */
	size_t		data_alignment;		/* best guess: register size */
	size_t		max_alignment;		/* best guess: the CPU's page size */
	/* ELF stuff */
	unsigned	elf_class;
	unsigned	elf_data;
	unsigned	elf_machine;
	size_t		elf_reloc_size;
	size_t		elf_symbol_size;
	size_t		elf_reloc_type;
	size_t		elf_reloc_section;
	unsigned	elf_alloc_reloc : 1;
	unsigned	elf_alloc_symtab : 1;
};

extern target_t fcpu_target;		/* defined later */
target_t *target = &fcpu_target;

/* generic lookup tables */

typedef struct {
	const char*	name;
	long		value;
} lookup_table_t;

static long table_lookup(lookup_table_t *lt, const char *item, long def, const char *category);

extern lookup_table_t object_types[];	/* defined later */
extern lookup_table_t section_types[];	/* defined later */
extern lookup_table_t section_attrs[];	/* defined later */

/* conditional assembly and include */

static int cond_state = 0;

static void
start_stop_scanner(int flag) {
	if (flag) {
		cond_state = 1;
	}
	else {
		suspend_scanner();
	}
}

static void include_file(const char *name);

/* sections */

extern scn_t abs_section[];		/* defined later */

static void init_sections(void);
static long scn_flags_from_string(const char *str);
static scn_t *find_section(const char *name, const long *flags, const long *type, long sub);
static void scn_entsize(scn_t *scn, word_t entsize);
static void default_section_attrs(scn_t *scn, const long *flags, const long *type);
static void pop_section(int pop);
static void push_section(scn_t *newscn);
static char *extend_section(scn_t *scn, word_t bytes);
static sym_t *dummy_symbol(const char *name, scn_t *scn, long type);
static sym_t *section_start_symbol(scn_t *scn);
static word_t add_string(scn_t *scn, const char *str);
static void file_symbol(const char *name);
static void add_comment(const char *text);
static void alloc_note(const char *str);

/* expressions */

extern expr_t errexpr[];	/* defined later */

static expr_t *unary(int w, expr_t *r);
static expr_t *binary(int w, expr_t *l, expr_t *r);
static expr_t *to_abs(expr_t *e);
static expr_t *numeric_expr(const char *val);
static expr_t *address(scn_t *scn, word_t off);
static expr_t *current_addr(void);
static expr_t *register_operand(const char *reg);
static expr_t *memory_operand(expr_t *e);
static expr_t *make_arglist(expr_t *l, expr_t *r);
static expr_t *symexpr(sym_t *sym, word_t off);
static expr_t *relative_symbol(sym_t *sym, const char *rel);

#define absexpr(v)	address(abs_section, (v))

static word_t to_int(expr_t *e);

static int is_abs(expr_t *e);
static int is_true(expr_t *e);

/* allocation */

static void alloc_bytes(expr_t *size, expr_t *fill);
static void alloc_idata(long size, expr_t *data);
static void alloc_fdata(long size, expr_t *data);
static void alloc_string(int addz, const char *val);
static void alloc_leb128(int signd, expr_t *data);
static void alloc_fill(expr_t *repeat, expr_t *size, expr_t *data);
static void align(int mode, expr_t *alignment, expr_t *fill, expr_t *max);
static void new_org(expr_t *off, expr_t *fill);

/* symbols */

static sym_t *make_symbol(const char *name);
static sym_t *intern_symbol(const char *name);
static void define_symbol(long unique, sym_t *sym, expr_t *val);
static void define_common(long local, sym_t *sym, expr_t *size, expr_t *align);
static void set_symbol_binding(sym_t *sym, long bind);
static void set_symbol_size(sym_t *sym, word_t size);
static void set_symbol_type(sym_t *sym, long type);
static void set_symbol_value(sym_t *sym, expr_t *val);
static void set_symbol_visibility(sym_t *sym, long what);
static int is_defined(sym_t *sym);
static void undefined(sym_t *sym);

/* relocations */

static void add_reloc_entry(scn_t *scn, const reloc_t *rel);

/* versioning */

static void symbol_version(sym_t *sym, const char *alias, const char *version, int def);

/* the hardest part... */

static void emit_instruction(const char *name, expr_t *args);
static void emit_stab(const char *name, expr_t *type, expr_t *other, expr_t *desc, expr_t *value);

%}

%union {
	long		lval;
	const char*	sval;
	expr_t*		expr;
	scn_t*		sect;
	sym_t*		symb;
	macro_t*	macr;
}

/* asm directives */
%token ERROR EXTERN FILL FNAME IDENT ORG SECTION SIZE SPACE SYMVER TYPE
%token VERS WEAKEXT
%token <lval> ALIGN ASSIGN COMMON EXPORT FDATA IDATA LEB128 SDATA
%token <lval> VISIBILITY PREVSCN
%token <sval> NAMESCN	/* .text | .data | .bss | probably others */

/* stabs debugging */
%token STABS STABN STABD

/* conditional assembly and include */
%token IF IFDEF IFNDEF ELIF ELSE ENDIF INCLUDE

/* macros */
%token IRP IRPC REPT MACRO ENDM ENDR EXITM PURGEM

/* other tokens */
%token <lval> CHARACTER
%token <sval> IDENTIFIER
%token <sval> MACRO_NAME
%token <sval> NUMBER
%token <sval> STRING

/* "||" operator */
%token PAR

%type <expr> expr unary primary abs_expr opt_abs_expr operand operand_list
%type <expr> register_name
%type <lval> if_statement if_block conditional elif_statement
%type <lval> integer_storage floating_storage string_storage leb128_storage
%type <lval> object_type section_type solaris_scn_flag scn_flag_list scn_flags
%type <lval> visibility entsize
%type <sect> section_designator
%type <sval> section_name ident_or_string
%type <symb> symbol
%type <sval> identifier

/* operator precedence */
%left '|'
%left '^'
%left '&'
%left EQL NEQ
%left '<' '>' LEQ GEQ
%left SHL SHR
%left '+' '-'
%left '*' '/' '%'

%start file

%%

file : initialize group

initialize :
	{ cond_state = 0; init_sections(); }

group :
group : group set_location eol
group : group set_location line eol
group : group set_location conditional
	{ cond_state = $3; /* return to previous nesting level */ }
group : group set_location INCLUDE ident_or_string eol
	{ include_file($4); }

eol : '\n'
eol : error '\n'
	{ yyerrok; }

set_location :
	{ current_line = lineno; }

/*
 * => Macros <=
 * (not ready yet)
 *
group : group set_location MACRO macro_params eol
group : group set_location REPT abs_expr eol
group : group set_location ENDM eol
group : group set_location ENDR eol
group : group set_location MACRO_NAME
group : group set_location EXITM eol
group : group set_location PURGEM identifier eol
group : group set_location IRPC identifier ',' ident_or_string eol
group : group set_location IRP identifier irp_args eol

macro_params : identifier
macro_params : macro_params ',' identifier

irp_args :
irp_args : ',' expr irp_args
*/

/*
 * => Conditional Assembly <=
 *
 * This is mostly done inside the scanner. When the parser detects a false
 * condition, it puts the scanner in `skip' mode. That is, it will not
 * return tokens (except '\n') until it encounters another conditional
 * directive at the same nesting level. It's still not allowed to fill
 * the skipped parts with random garbage, though - later versions may
 * require that they're syntactically correct.
 */
conditional : if_block endif_statement
conditional : if_block else_statement group endif_statement

if_block : if_statement group
if_block : if_block elif_statement group

if_statement : IF abs_expr eol
	{ $$ = cond_state; cond_state = 0; start_stop_scanner(is_true($2)); }
if_statement : IFDEF symbol eol
	{ $$ = cond_state; cond_state = 0; start_stop_scanner(is_defined($2)); }
if_statement : IFNDEF symbol eol
	{ $$ = cond_state; cond_state = 0; start_stop_scanner(!is_defined($2)); }

elif_statement : set_location ELIF abs_expr eol
	{ start_stop_scanner(is_true($3) && !cond_state); }

else_statement : set_location ELSE eol
	{ start_stop_scanner(!cond_state); }

endif_statement : set_location ENDIF eol

line : instruction
line : instruction ';'			/* last instruction may be empty */
line : instruction ';' line

instruction : label
instruction : parallel_instructions
instruction : pseudo_instruction
instruction : label instruction

label : symbol ':'
	{ define_symbol(1, $1, current_addr()); }

symbol : identifier
	{ $$ = intern_symbol($1); }

/*
 * => Parallel Execution <=
 *
 * Instructions separated by "||" may be executed simultaneously
 * (if supported by the hardware).  The CPU (or the assembler) may
 * reorder them as it sees fit.  Labels, if present, apply to the
 * group as a whole.  If the so-parallelized instructions depend on
 * each other, the behaviour is undefined.  Future versions of this
 * program may detect that case and issue a warning, terminate with
 * or without an error message, silently "sequentialize" the code,
 * or simply ignore the error, at the implementors discretion.
 * You've been warned.
 */
parallel_instructions : machine_instruction
parallel_instructions : parallel_instructions PAR machine_instruction
parallel_instructions : parallel_instructions PAR '\n' machine_instruction

machine_instruction : IDENTIFIER
	{ emit_instruction($1, NULL); }
machine_instruction : IDENTIFIER operand_list
	{ emit_instruction($1, $2); }

operand_list : operand
	{ $$ = make_arglist($1, NULL); }
operand_list : operand ',' operand_list
	{ $$ = make_arglist($1, $3); }

/*
 * => Implementor's Choice <=
 *
 * An identifier may be the name of an instruction, a register name
 * or an ordinary symbol, which causes some ambiguities.  While
 * instruction names can be easily recognized by position (after
 * a newline, semicolon or "||" operator, with no ":" following),
 * there is no way to tell the difference between a register name
 * and a symbol used as an immediate operand.  Prefixing one of
 * them with an otherwise unused character fixes the problem.
 * Some assemblers use "$" or "#" for immediates, others use "%"
 * for registers, some even prefix *both* (like gas for Intel IA32).
 * I decided to prefix only immediate operands because register names
 * are probably more often used than immediate operands.  I chose
 * "$" as the prefix character (which may appear in an identifier,
 * but not at its beginning) because most other characters (including
 * the popular "#", "%", "&", and "@") already serve other purposes.
 */
operand : '$' expr
	{ $$ = $2; }
operand : register_name

register_name : identifier
	{ $$ = register_operand($1); }

/*
 * => Direct/Indirect addressing modes <=
 * (not ready yet)
 *
operand : memory_operand
	{ $$ = memory_operand($1); }

memory_operand : '(' expr ')'
	{ $$ = deref($2); }
memory_operand : '(' indirect ')'
	{ $$ = deref($2); }

indirect : scaled_register
indirect : indirect '+' scaled_register
	{ $$ = addr_add($1, $3); }
indirect : indirect '+' primary
	{ $$ = addr_add($1, $3); }
indirect : indirect '-' primary
	{ $$ = addr_sub($1, $3); }

scaled_register : register_name
scaled_register : register_name '*' NUMBER
	{ $$ = addr_mul($1, $3); }
scaled_register : NUMBER '*' register_name
	{ $$ = addr_mul($3, $1); }
*/

abs_expr : expr			{ $$ = to_abs($1); }

expr : expr '*' expr	{ $$ = binary('*', $1, $3); }
expr : expr '/' expr	{ $$ = binary('/', $1, $3); }
expr : expr '%' expr	{ $$ = binary('%', $1, $3); }
expr : expr '+' expr	{ $$ = binary('+', $1, $3); }
expr : expr '-' expr	{ $$ = binary('-', $1, $3); }
expr : expr SHL expr	{ $$ = binary(SHL, $1, $3); }
expr : expr SHR expr	{ $$ = binary(SHR, $1, $3); }
expr : expr '<' expr	{ $$ = binary('<', $1, $3); }
expr : expr '>' expr	{ $$ = binary('>', $1, $3); }
expr : expr LEQ expr	{ $$ = binary(LEQ, $1, $3); }
expr : expr GEQ expr	{ $$ = binary(GEQ, $1, $3); }
expr : expr EQL expr	{ $$ = binary(EQL, $1, $3); }
expr : expr NEQ expr	{ $$ = binary(NEQ, $1, $3); }
expr : expr '&' expr	{ $$ = binary('&', $1, $3); }
expr : expr '|' expr	{ $$ = binary('|', $1, $3); }
expr : expr '^' expr	{ $$ = binary('^', $1, $3); }
expr : unary			{ $$ = $1; }

unary : '+' primary		{ $$ = unary('+', $2); }
unary : '-' primary		{ $$ = unary('-', $2); }
unary : '~' primary		{ $$ = unary('~', $2); }
unary : '!' primary		{ $$ = unary('!', $2); }
unary : primary			{ $$ = $1; }

primary : '.'			{ $$ = current_addr(); }
primary : NUMBER		{ $$ = numeric_expr($1); }
primary : CHARACTER		{ $$ = absexpr($1); }
primary : '(' expr ')'	{ $$ = $2; }
primary : '(' error ')'	{ $$ = errexpr; yyerrok; }

primary : symbol
	{ $$ = symexpr($1, 0); }
primary : symbol '@' identifier
	{ $$ = relative_symbol($1, $3); }

opt_abs_expr :			{ $$ = NULL; }
opt_abs_expr : abs_expr	{ $$ = $1; }

pseudo_instruction : '.' '=' expr	/* same as `.org expr' */
	{ new_org($3, NULL); }
pseudo_instruction : ORG expr
	{ new_org($2, NULL); }
pseudo_instruction : ORG expr ',' abs_expr
	{ new_org($2, $4); }
pseudo_instruction : symbol '=' expr	/* same as `.set symbol, expr' */
	{ define_symbol(0, $1, $3); }
pseudo_instruction : ASSIGN symbol ',' expr
	{ define_symbol($1, $2, $4); }
pseudo_instruction : COMMON symbol ',' abs_expr
	{ define_common($1, $2, $4, NULL); }
pseudo_instruction : COMMON symbol ',' abs_expr ',' abs_expr
	{ define_common($1, $2, $4, $6); }
pseudo_instruction : EXPORT symbol
	{ set_symbol_binding($2, $1); }
pseudo_instruction : WEAKEXT symbol		/* same as `.weak symbol' */
	{ set_symbol_binding($2, STB_WEAK); }
pseudo_instruction : WEAKEXT symbol ',' expr
	{ set_symbol_binding($2, STB_WEAK); set_symbol_value($2, $4); }
pseudo_instruction : section_designator
	{ push_section($1); }
pseudo_instruction : PREVSCN
	{ pop_section($1); }
pseudo_instruction : integer_storage
	{ /* handled elsewhere */ }
pseudo_instruction : floating_storage
	{ /* handled elsewhere */ }
pseudo_instruction : string_storage
	{ /* handled elsewhere */ }
pseudo_instruction : leb128_storage
	{ /* handled elsewhere */ }
pseudo_instruction : ALIGN abs_expr
	{ align($1, $2, NULL, NULL); }
pseudo_instruction : ALIGN abs_expr ',' abs_expr
	{ align($1, $2, $4, NULL); }
pseudo_instruction : ALIGN abs_expr ',' opt_abs_expr ',' abs_expr
	{ align($1, $2, $4, $6); }
pseudo_instruction : SPACE abs_expr
	{ alloc_bytes($2, NULL); }
pseudo_instruction : SPACE abs_expr ',' abs_expr
	{ alloc_bytes($2, $4); }
pseudo_instruction : FILL abs_expr
	{ alloc_fill($2, NULL, NULL); }
pseudo_instruction : FILL abs_expr ',' abs_expr
	{ alloc_fill($2, $4, NULL); }
pseudo_instruction : FILL abs_expr ',' opt_abs_expr ',' abs_expr
	{ alloc_fill($2, $4, $6); }
pseudo_instruction : ERROR
	{ error("encountered .err directive"); }
pseudo_instruction : EXTERN symbol
	{ /* always ignore */ }
pseudo_instruction : FNAME ident_or_string
	{ file_symbol($2); }
pseudo_instruction : IDENT ident_or_string
	{ add_comment($2); }
pseudo_instruction : VERS ident_or_string
	{ alloc_note($2); }
pseudo_instruction : SIZE symbol ',' abs_expr
	{ set_symbol_size($2, to_int($4)); }
pseudo_instruction : TYPE symbol ',' object_type
	{ set_symbol_type($2, $4); }
pseudo_instruction : SYMVER symbol ',' identifier '@' identifier
	{ symbol_version($2, $4, $6, 0); }
pseudo_instruction : SYMVER symbol ',' identifier '@' '@' identifier
	{ symbol_version($2, $4, $7, 1); }
pseudo_instruction : SYMVER symbol ',' identifier '@' '@' '@' identifier
	{ symbol_version($2, $4, $8, 2); }
pseudo_instruction : STABS STRING ',' abs_expr ',' abs_expr ',' abs_expr ',' abs_expr
	{ emit_stab($2, $4, $6, $8, $10); }
pseudo_instruction : STABN abs_expr ',' abs_expr ',' abs_expr ',' abs_expr
	{ emit_stab(NULL, $2, $4, $6, $8); }
pseudo_instruction : STABD abs_expr ',' abs_expr ',' abs_expr
	{ emit_stab(NULL, $2, $4, $6, NULL); }
pseudo_instruction : visibility
	{ /* handled elsewhere */ }

integer_storage : IDATA expr
	{ alloc_idata($$ = $1, $2); }
integer_storage : integer_storage ',' expr
	{ alloc_idata($$ = $1, $3); }

floating_storage : FDATA abs_expr
	{ alloc_fdata($$ = $1, $2); }
floating_storage : floating_storage ',' abs_expr
	{ alloc_fdata($$ = $1, $3); }

string_storage : SDATA STRING
	{ alloc_string($$ = $1, $2); }
string_storage : string_storage ',' STRING
	{ alloc_string($$ = $1, $3); }

leb128_storage : LEB128 abs_expr
	{ alloc_leb128($$ = $1, $2); }
leb128_storage : leb128_storage ',' abs_expr
	{ alloc_leb128($$ = $1, $3); }

section_designator : NAMESCN
	{ $$ = find_section($1, NULL, NULL, 0); }
section_designator : NAMESCN abs_expr
	{ $$ = find_section($1, NULL, NULL, to_int($2)); }
section_designator : section_name
	{ $$ = find_section($1, NULL, NULL, 0); }
section_designator : section_name ',' section_type
	{ $$ = find_section($1, NULL, &($3), 0); }
section_designator : section_name ',' scn_flags
	{ $$ = find_section($1, &($3), NULL, 0); scn_entsize($$, 0); }
section_designator : section_name ',' scn_flags ',' section_type
	{ $$ = find_section($1, &($3), &($5), 0); scn_entsize($$, 0); }
section_designator : section_name ',' scn_flags ',' section_type ',' entsize
	{ $$ = find_section($1, &($3), &($5), 0); scn_entsize($$, $7); }
section_designator : section_name ',' scn_flag_list
	{ $$ = find_section($1, &($3), NULL, 0); }

section_name : SECTION ident_or_string
	{ $$ = $2; }

scn_flags : ident_or_string			/* awx */
	{ $$ = scn_flags_from_string($1); }

entsize : '@' abs_expr
	{ $$ = to_int($2); }

scn_flag_list : solaris_scn_flag
	{ $$ = $1; }
scn_flag_list : scn_flag_list ',' solaris_scn_flag
	{ $$ = $1 | $3; }

visibility : VISIBILITY symbol
	{ set_symbol_visibility($2, $$ = $1); }
visibility : visibility ',' symbol
	{ set_symbol_visibility($3, $$ = $1); }

ident_or_string : identifier
ident_or_string : STRING

identifier : IDENTIFIER
identifier : MACRO_NAME

object_type : '@' identifier	/* function / object */
	{ $$ = table_lookup(object_types, $2, STT_NOTYPE, "symbol type"); }
object_type : '#' identifier	/* function / object */
	{ $$ = table_lookup(object_types, $2, STT_NOTYPE, "symbol type"); }

section_type : '@' identifier	/* progbits / nobits */
	{ $$ = table_lookup(section_types, $2, SHT_PROGBITS, "section type"); }

solaris_scn_flag : '#' identifier	/* write | alloc | execinstr */
	{ $$ = table_lookup(section_attrs, $2, 0, "section attribute"); }

%%

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

struct sym {
	sym_t		*next;
	const char	*name;
	scn_t		*scn;	/* section this symbol belongs to */
	expr_t		*val;
	long		type;
	long		bind;
	word_t		size;
	word_t		align;
	char		other;	/* hidden/protected/internal */
	unsigned	used : 1;
	unsigned	relocs : 1;
	/* ELF stuff */
	size_t		elf_index;
};

/* generic lookup tables */

static long
table_lookup(lookup_table_t *lt, const char *item, long def, const char *category) {
	int i;

	assert(lt);
	assert(item);
	assert(category);
	for (; lt->name; lt++) {
		i = strcmp(item, lt->name);
		if (i < 0) {
			/* assume ordered table */
			break;
		}
		if (i == 0) {
			return lt->value;
		}
	}
	error("invalid %s \"%s\"", category, item);
	return def;
}

lookup_table_t object_types[] = {
	{ "function", STT_FUNC   },
	{ "object",   STT_OBJECT },
	{ NULL,       STT_NOTYPE }
};

lookup_table_t section_types[] = {
	{ "nobits",   SHT_NOBITS   },
	{ "progbits", SHT_PROGBITS },
	{ NULL,       SHT_NULL     }
};

lookup_table_t section_attrs[] = {
	{ "alloc",     SHF_ALLOC     },
	{ "execinstr", SHF_EXECINSTR },
	{ "write",     SHF_WRITE     },
	{ NULL,        0             }
};

/* standard malloc */

void*
xrealloc(void *p, size_t len) {
	len = len ? len : 1;
	p = p ? realloc(p, len) : malloc(len);
	if (!p) fatal("out of menory");
	return p;
}

void
xfree(void *p) {
	if (p) free(p);
}

/* pool alloc */

struct pool {
	struct pool	*next;
	unsigned	free;
};

static struct pool *palloc_pool;

#define PALLOC_POOL_SIZE	65536U
#define PALLOC_MAX_SIZE		256U
#define PALLOC_ALIGN		8U

void*
palloc(size_t len) {
	struct pool *p = palloc_pool;
	char *result;

	if (len > PALLOC_MAX_SIZE) {
		/* palloc is for small memory blocks only */
		fatal("cannot palloc %u bytes", len);
	}
	len += PALLOC_ALIGN - 1;
	len -= len % PALLOC_ALIGN;
	if (!p || p->free + len > PALLOC_POOL_SIZE) {
		p = xmalloc(PALLOC_POOL_SIZE);
		p->next = palloc_pool;
		palloc_pool = p;
		p->free = sizeof(struct pool) + PALLOC_ALIGN - 1;
		p->free -= p->free % PALLOC_ALIGN;
	}
	result = (char*)p + p->free;
	p->free += len;
	return (void*)result;
}

void
pfreeall(void) {
	struct pool *p, *q;

	for (p = palloc_pool; p; p = q) {
		q = p->next;
		xfree(p);
	}
	palloc_pool = NULL;
}

/* sections */

struct scn {
	scn_t		*next;
	const char	*name;
	char		*data;
	long		flags;
	long		type;
	word_t		size;
	word_t		align;
	sym_t		*dummy;
	reloc_t		*relocs;
	size_t		nrelocs;
	word_t		entsize;
	scn_t		*link;
	/* ELF stuff */
	Elf_Scn		*elf_scn;
	Elf_Scn		*elf_rel;
};

struct reloc {
	sym_t		*sym;
	long		type;
	word_t		off;
	word_t		addend;
};

static scn_t *all_scns = NULL;

static scn_t*
make_section(const char *name) {
	scn_t *scn;

	scn = palloc(sizeof(scn_t));
	assert(scn);
	scn->next = NULL;
	scn->name = name;
	scn->data = NULL;
	scn->flags = 0;
	scn->type = 0;
	scn->size = 0;
	scn->align = 0;
	scn->dummy = NULL;
	scn->relocs = NULL;
	scn->nrelocs = 0;
	scn->entsize = 0;
	scn->link = NULL;
	scn->elf_scn = NULL;
	scn->elf_rel = NULL;
	return scn;
}

static void
scn_entsize(scn_t *scn, word_t entsize) {
	assert(scn);
	assert(scn->name);
	if (scn->entsize) {
		error("attempt to override section entry size for %s", scn->name);
	}
#if defined(SHF_MERGE)
	else if ((scn->flags & SHF_MERGE) && !entsize) {
		error("mergeable section %s lacks entry size", scn->name);
	}
#endif
	else {
		scn->entsize = entsize;
	}
}

static scn_t*
find_section(const char *name, const long *flags, const long *type, long sub) {
	scn_t **p, *scn;

	if (sub) {
		error("subsections not supported");
		sub = 0;
	}
	for (p = &all_scns; (scn = *p); p = &scn->next) {
		if (strcmp(name, scn->name) == 0) {
			if ((flags != NULL && *flags != scn->flags)
			 || (type != NULL && *type != scn->type)) {
				error("attempt to change section attributes for %s", name);
			}
			return scn;
		}
	}
	scn = make_section(name);
	assert(scn);
	default_section_attrs(scn, flags, type);
	if (all_start_symbols) {
		scn->dummy = dummy_symbol("", scn, STT_SECTION);
	}
	scn->next = *p, *p = scn;
	return scn;
}

static int
nobits_section(scn_t *scn) {
	assert(scn);
	return scn->type == SHT_NOBITS;
}

static char*
extend_section(scn_t *scn, word_t bytes) {
	const unsigned PAGESZ = 1024;
	word_t off = scn->size;
	word_t n;

	assert(scn);
	scn->size += bytes;
	if (nobits_section(scn)) {
		return NULL;
	}
	n = scn->size + PAGESZ - 1;
	if (n / PAGESZ > (off + PAGESZ - 1) / PAGESZ) {
		scn->data = xrealloc(scn->data, n - n % PAGESZ);
	}
	return scn->data + off;
}

static struct scntab {
	const char*	name;
	long		type;
	long		flags;
} well_known_sections[] = {
	{ ".bss",            SHT_NOBITS,      SHF_ALLOC+SHF_WRITE     },
	{ ".comment",        SHT_PROGBITS,    0                       },
	{ ".data",           SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     },
	{ ".data1",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     },
	{ ".debug",          SHT_PROGBITS,    0                       },
	{ ".dynamic",        SHT_DYNAMIC,     SHF_ALLOC /* +WRITE */  }, /*ld*/
	{ ".dynstr",         SHT_STRTAB,      SHF_ALLOC               }, /*ld*/
	{ ".dynsym",         SHT_DYNSYM,      SHF_ALLOC               }, /*ld*/
	{ ".fini",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
#ifdef SHT_FINI_ARRAY
	{ ".fini_array",     SHT_FINI_ARRAY,  SHF_ALLOC+SHF_WRITE     },
#endif
	{ ".got",            SHT_PROGBITS,    0 /* ALLOC + WRITE */   }, /*ld*/
	{ ".hash",           SHT_HASH,        SHF_ALLOC               }, /*ld*/
	{ ".init",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
#ifdef SHT_INIT_ARRAY
	{ ".init_array",     SHT_INIT_ARRAY,  SHF_ALLOC+SHF_WRITE     },
#endif
	{ ".interp",         SHT_PROGBITS,    0 /* SHF_ALLOC */       }, /*ld*/
	{ ".line",           SHT_PROGBITS,    0                       },
	{ ".note",           SHT_NOTE,        0                       }, /*auto*/
	{ ".plt",            SHT_PROGBITS,    0 /* ALLOC + EXEC */    }, /*ld*/
#ifdef SHT_PREINIT_ARRAY
	{ ".preinit_array",  SHT_PREINIT_ARRAY, SHF_ALLOC+SHF_WRITE   },
#endif
	{ ".rel",            SHT_REL,         0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rela",           SHT_RELA,        0 /* SHF_ALLOC */       }, /*auto*/
	{ ".rodata",         SHT_PROGBITS,    SHF_ALLOC               },
	{ ".rodata1",        SHT_PROGBITS,    SHF_ALLOC               },
	{ ".sbss",           SHT_NOBITS,      SHF_ALLOC+SHF_WRITE     }, /*short*/
	{ ".sdata",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*short*/
	{ ".shstrtab",       SHT_STRTAB,      0                       }, /*auto*/
	{ ".stab",           SHT_PROGBITS,    0                       },
	{ ".stab.excl",      SHT_PROGBITS,    0                       },
	{ ".stab.exclstr",   SHT_STRTAB,      0,                      },
	{ ".stab.index",     SHT_PROGBITS,    0                       },
	{ ".stab.indexstr",  SHT_STRTAB,      0,                      },
	{ ".stabstr",        SHT_STRTAB,      0                       },
	{ ".strtab",         SHT_STRTAB,      0 /* SHF_ALLOC */       }, /*auto*/
	{ ".symtab",         SHT_SYMTAB,      0 /* SHF_ALLOC */       }, /*auto*/
#ifdef SHT_SYMTAB_SHNDX
	{ ".symtab_shndx",   SHT_SYMTAB_SHNDX, 0 /* SHF_ALLOC */      }, /*auto*/
#endif
#ifdef SHF_TLS	/* ELF thread local storage */
	{ ".tbss",           SHT_NOBITS,      SHF_ALLOC+SHF_WRITE+SHF_TLS },
	{ ".tdata",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE+SHF_TLS },
	{ ".tdata1",         SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE+SHF_TLS },
#endif
	{ ".text",           SHT_PROGBITS,    SHF_ALLOC+SHF_EXECINSTR },
#if 0 /* Linux/GNU special (LSB 1.0) */
	{ ".ctors",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".dtors",          SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".eh_frame",       SHT_PROGBITS,    SHF_ALLOC+SHF_WRITE     }, /*LSB*/
	{ ".gnu.version_d",  SHT_GNU_verdef,  SHF_ALLOC               }, /*LSB*/
	{ ".gnu.version_r",  SHT_GNU_verneed, SHF_ALLOC               }, /*LSB*/
	{ ".gnu.version",    SHT_GNU_versym,  SHF_ALLOC               }, /*LSB*/
	{ ".note.ABI-tag",   ?,               ?,                      }, /*LSB*/
#endif
#if 0 /* GNU junk */
	{ ".gnu.warning.*",  SHT_PROGBITS,    0                       },
#endif
};

static void
default_section_attrs(scn_t *scn, const long *flags, const long *type) {
	const char *name;
	size_t l, r, m;
	int i;

	assert(scn);
	name = scn->name;
	assert(name);
	if (strncmp(name, ".rel.", 5) == 0) {
		name = ".rel";
	}
	else if (strncmp(name, ".rela.", 6) == 0) {
		name = ".rela";
	}
	if (type == NULL || flags == NULL) {
		/* get default attributes */
		l = 0;
		r = sizeof(well_known_sections) / sizeof(well_known_sections[0]);
		while (l < r) {
			m = l + ((r - l) >> 1);
			i = strcmp(name, well_known_sections[m].name);
			if (i < 0) {
				r = m;
			}
			else if (i > 0) {
				l = m + 1;
			}
			else {
				/* found */
				if (type == NULL) {
					type = &well_known_sections[m].type;
				}
				if (flags == NULL) {
					flags = &well_known_sections[m].flags;
				}
				break;
			}
		}
	}
	scn->type = type ? *type : SHT_PROGBITS;
	scn->flags = flags ? *flags : 0;
	assert(scn->type != SHT_NULL);
}

static scn_t *bss_section = NULL;
static scn_t *note_section = NULL;
static scn_t *comment_section = NULL;

static scn_t *current_section = NULL;

static scn_t abs_section[1];
static scn_t undef_section[1];
static scn_t common_section[1];
static scn_t alias_section[1];

static struct scn_stack {
	scn_t *section;
	struct scn_stack *link;
} *section_stack = NULL;

static void
pop_section(int pop) {
	scn_t *tmp;

	if (!section_stack) {
		error("no previous section");
		return;
	}
	tmp = section_stack->section;
	section_stack->section = current_section;
	current_section = tmp;
}

static void
push_section(scn_t *newscn) {
	assert(newscn);
	if (newscn != current_section) {
		struct scn_stack *tmp = palloc(sizeof(struct scn_stack));

		assert(tmp);
		tmp->section = current_section;
		tmp->link = section_stack;
		section_stack = tmp;
		current_section = newscn;
	}
}

static word_t
add_string(scn_t *scn, const char *str) {
	word_t off;
	char *s;

	assert(scn);
	assert(str);
	off = scn->size;
	if (nobits_section(scn)) {
		error("directive is invalid in NOBITS sections");
		/* sanitize */
		return off;
	}
	s = extend_section(scn, strlen(str) + 1);
	strcpy(s, str);
	return off;
}

static void
add_comment(const char *text) {
	if (!comment_section) {
		comment_section = find_section(".comment", NULL, NULL, 0);
	}
	if (comment_section->size == 0) {
		add_string(comment_section, "");
	}
	add_string(comment_section, text);
}

static void
init_sections(void) {
	long flags, type;
	scn_t *scn;

	if (all_scns) return;	/* parsing yet another input file */
	/*
	 * create text section
	 */
	type = SHT_PROGBITS;
	flags = SHF_ALLOC | SHF_EXECINSTR;
	scn = find_section(".text", &flags, &type, 0);
	scn->align = target->text_alignment;
	/*
	 * assembly starts in `.text 0'
	 */
	current_section = scn;
	comment_section = NULL;
	section_stack = NULL;
	/*
	 * create data sections
	 */
	flags = SHF_ALLOC | SHF_WRITE;
	scn = find_section(".data", &flags, &type, 0);
	scn->align = target->data_alignment;
	type = SHT_NOBITS;
	scn = find_section(".bss", &flags, &type, 0);
	scn->align = target->data_alignment;
	bss_section = scn;
}

static long
scn_flags_from_string(const char *str) {
	long flags = 0;
	int err = 0;
	char c;

	while ((c = *str++)) {
		switch (c) {
			case 'a': flags |= SHF_ALLOC; break;
			case 'w': flags |= SHF_WRITE; break;
			case 'x': flags |= SHF_EXECINSTR; break;
#if defined(SHF_MERGE) && defined(SHF_STRINGS)
			case 'M': flags |= SHF_MERGE; break;
			case 'S': flags |= SHF_STRINGS; break;
#endif
			default: err++;
		}
	}
	if (err) {
		error("invalid section attribute");
	}
	return flags;
}

/* expressions */

enum {
	SREF_NONE, SREF_PC, SREF_GOT, SREF_PLT, SREF_GOTOFF, SREF_GOTPC,
	SREF_num
};

union expr {
	int				op;
	struct {
		int			op;		/* unary/binary */
		expr_t		*left;
		expr_t		*right;
	} bin;
	struct {				/* absolute, address, symbol */
		int			op;		/* 'a', 's' */
		sym_t		*sym;
		scn_t		*scn;
		word_t		off;
		int			ref;
	} addr;
	struct {				/* bignum / flonum */
		int			op;		/* 'b', 'f' */
		unsigned	nbits;
		unsigned	*bits;
		int			expt;
		int			sign;
	} fnum;
	struct {				/* register */
		int			op;		/* 'r' */
		long		rnum;
	} reg;
};

static expr_t errexpr[1] = {{ .op = 'E' }};
static expr_t iconst[2] = {
	{ .addr = { 'a', NULL, abs_section, 0, -1 }},
	{ .addr = { 'a', NULL, abs_section, 1, -1 }},
};
static expr_t fconst[] = {
	{ .fnum = { 'f', 0, NULL, 0, 0 }},
};

static expr_t*
make_arglist(expr_t *l, expr_t *r) {
	expr_t *e;

	e = palloc(sizeof(expr_t));
	assert(e);
	e->op = ',';
	e->bin.left = l;
	e->bin.right = r;
	return e;
}

static expr_t*
symbol_expr(sym_t *sym, word_t off, int ref) {
	expr_t *e;

	assert(sym);
	assert(sym->scn);
	sym->used = 1;
	e = palloc(sizeof(expr_t));
	assert(e);
	e->op = 's';
	e->addr.sym = sym;
	e->addr.scn = NULL;
	e->addr.off = off;
	e->addr.ref = ref;
	return e;
}

static expr_t*
symexpr(sym_t *sym, word_t off) {
	return symbol_expr(sym, off, SREF_NONE);
}

static expr_t*
relexpr(sym_t *sym, word_t off) {
	return symbol_expr(sym, off, SREF_PC);
}

static expr_t*
relative_symbol(sym_t *sym, const char *rel) {
	lookup_table_t table[] = {
		/* NOTE: must be sorted ASCIIbetically */
		{ "got",    SREF_GOT },
		{ "gotoff", SREF_GOTOFF },
		{ "gotpc",  SREF_GOTPC },
		{ "pc",     SREF_PC },
		{ "plt",    SREF_PLT },
	};
	size_t i, j, k;
	int m;

	i = 0;
	j = sizeof(table) / sizeof(*table);
	while (i < j) {
		k = i + (j - i) / 2u;
		m = strcasecmp(rel, table[k].name);
		if (m < 0) {
			j = k;
		}
		else if (m > 0) {
			i = k + 1;
		}
		else {
			return symbol_expr(sym, 0, table[k].value);
		}
	}
	error("bad reloc specifier \"%s\"", rel);
	/* sanitize */
	return symbol_expr(sym, 0, SREF_NONE);
}

static expr_t*
add_const(expr_t *e, word_t off) {
	expr_t *f;

	assert(e);
	if (e == errexpr) return errexpr;
	if (!off) return e;
	assert(e->op == 'a' || e->op == 's');
	f = palloc(sizeof(expr_t));
	*f = *e;
	f->addr.off += off;
	return f;
}

static expr_t*
eval_sym(expr_t *e) {
	word_t off = 0;

	assert(e);
	if (e == errexpr) return errexpr;
	while (e->op == 's' && e->addr.ref == SREF_NONE) {
		sym_t *sym = e->addr.sym;

		assert(sym);
		assert(sym->scn);
		assert(sym->used);
		if (!sym->val) break;	/* undefined */
		assert(sym->scn != undef_section);
		assert(sym->scn != common_section);
		off += e->addr.off;		/* collect offsets */
		e = sym->val;
	}
	return add_const(e, off);
}

static expr_t*
address(scn_t *scn, word_t off) {
	expr_t *e;

	assert(scn);
	assert(scn != undef_section);
	assert(scn != common_section);
	assert(scn != alias_section);
	if (scn == abs_section && off < 2u) {
		return &iconst[off];
	}
	e = palloc(sizeof(expr_t));
	assert(e);
	e->op = 'a';
	e->addr.sym = NULL;
	e->addr.scn = scn;
	e->addr.off = off;
	e->addr.ref = -1;
	return e;
}

static expr_t*
current_addr(void) {
	return address(current_section, current_section->size);
}

static int
is_abs(expr_t *e) {
	assert(e);
	switch (e->op) {
		case 's':	/* check smbol values, if any */
			assert(e->addr.sym);
			if (e->addr.ref == SREF_NONE && e->addr.sym->val) {
				return is_abs(e->addr.sym->val);
			}
			break;
		case 'a':	/* check section */
			return e->addr.scn == abs_section;
		case 'b':
		case 'f':
			return 1;
	}
	return 0;
}

static expr_t*
to_abs(expr_t *e) {
	assert(e);
	if (e != errexpr) {
		e = eval_sym(e);
		switch (e->op) {
			case 's':
				undefined(e->addr.sym);
				return errexpr;
			case 'a':
				assert(e->addr.scn);
				if (e->addr.scn != abs_section) break;
				return e;
			case 'b':
			case 'f':
				return e;
		}
		error("absolute expression expected");
		e = errexpr;
	}
	return e;
}

static word_t
to_int(expr_t *e) {
	assert(e);
	e = eval_sym(e);
	if (e->op != 'a' || e->addr.scn != abs_section) {
		error("integer value expected");
		return 0;
	}
	return e->addr.off;
}

static expr_t*
binary(int w, expr_t *l, expr_t *r) {
	sym_t *sym;
	word_t v;

	if (l == errexpr || r == errexpr) return errexpr;
	l = eval_sym(l);
	r = eval_sym(r);
	if (r->op == 'a') {
		if (r->addr.scn == abs_section) {
			if (l->op == 'a' && l->addr.scn == abs_section) {
				/* ordinary integer arithmetics */
				switch (w) {
					case '*': return absexpr(l->addr.off * r->addr.off);
					case '/': return absexpr(l->addr.off / r->addr.off);
					case '%': return absexpr(l->addr.off % r->addr.off);
					case '+': return absexpr(l->addr.off + r->addr.off);
					case '-': return absexpr(l->addr.off - r->addr.off);
					case SHL: return absexpr(l->addr.off << r->addr.off);
					case SHR: return absexpr(l->addr.off >> r->addr.off);
					case '<': return absexpr(l->addr.off < r->addr.off);
					case '>': return absexpr(l->addr.off > r->addr.off);
					case LEQ: return absexpr(l->addr.off <= r->addr.off);
					case GEQ: return absexpr(l->addr.off >= r->addr.off);
					case EQL: return absexpr(l->addr.off == r->addr.off);
					case NEQ: return absexpr(l->addr.off != r->addr.off);
					case '&': return absexpr(l->addr.off & r->addr.off);
					case '^': return absexpr(l->addr.off ^ r->addr.off);
					case '|': return absexpr(l->addr.off | r->addr.off);
				}
				assert(0);
			}
			else if (l->op == 'a' || l->op == 's') {
				/* address  constant */
				if (w == '+') {
					return add_const(l, r->addr.off);
				}
				else if (w == '-') {
					return add_const(l, -r->addr.off);
				}
			}
		}
		else if (l->op == 'a' && l->addr.scn == r->addr.scn) {
			/* address - address (same section) */
			if (w == '-') {
				return absexpr(l->addr.off - r->addr.off);
			}
		}
		else if (r->addr.scn == current_section) {
			if (w == '-') {
				/* address - . */
				if (l->op == 's' && l->addr.ref == SREF_NONE) {
					/* XXX: is this correct? */
					v = l->addr.off + current_section->size - r->addr.off;
					return relexpr(l->addr.sym, v);
				}
				if (l->op == 'a') {
					/* we don't have a symbol, but we can make one */
					assert(l->addr.scn != undef_section);
					assert(l->addr.scn != common_section);
					assert(l->addr.scn != alias_section);
					sym = section_start_symbol(l->addr.scn);
					/* XXX: is this correct? */
					v = l->addr.off + current_section->size - r->addr.off;
					return relexpr(sym, v);
				}
			}
			else if (w == '+' && l->op == 's' && l->addr.ref == SREF_PC) {
				/* (address - .) + . */
				/* XXX: is this correct? */
				v = l->addr.off - current_section->size + r->addr.off;
				return symexpr(l->addr.sym, v);
			}
		}
	}
	else if (l->op == 'a' && w == '+') {
		return binary(w, r, l);
	}
	error("integer expression expected");
	return errexpr;
}

static expr_t*
unary(int w, expr_t *r) {
	expr_t *e;

	if (r == errexpr) return errexpr;
	r = eval_sym(r);
	if (r->op == 'a' && r->addr.scn == abs_section) {
		/* integer arithmetics */
		switch (w) {
			case '+': return r;
			case '-': return absexpr(-r->addr.off);
			case '~': return absexpr(~r->addr.off);
			case '!': return absexpr(!r->addr.off);
		}
		assert(0);
	}
	else if (r->op == 'b' || r->op == 'f') {
		/* bignum and floating-point arithmetics */
		switch (w) {
			case '+':
				return r;
			case '-':
				e = palloc(sizeof(expr_t));
				*e = *r;
				e->fnum.sign = !e->fnum.sign;
				e->fnum.bits = xmalloc(e->fnum.nbits * sizeof(unsigned));
				memcpy(e->fnum.bits, r->fnum.bits, e->fnum.nbits * sizeof(unsigned));
				return e;
			case '!':
				/* bignums are never zero */
				return absexpr(r->op != 'b' && r->fnum.bits == 0);
		}
	}
	error("integer expression expected");
	return errexpr;
}

static unsigned
digit(unsigned c) {
	unsigned x;

	if ((x = c - '0') <= '9' - '0') return x;
	if ((x = c - 'a') <= 'z' - 'a') return x + 10;
	if ((x = c - 'A') <= 'Z' - 'A') return x + 10;
	return ~0u;
}

#define MAX_POWERS 16
#define POWER_BASE 5
#define DIVIDE_CHUNKS 9

static void
multiply(expr_t *y, expr_t *a, expr_t *b) {
	unsigned i, j, nres, *res;
	word_t x;

	assert(a && a->fnum.bits);
	assert(b && b->fnum.bits);
	nres = a->fnum.nbits + b->fnum.nbits;
	res = xmalloc(nres * sizeof(unsigned));
	memset(res, 0, nres * sizeof(unsigned));
	for (i = 0; i < a->fnum.nbits; i++) {
		x = 0;
		for (j = 0; j < b->fnum.nbits; j++) {
			x = (word_t)a->fnum.bits[i] * (word_t)b->fnum.bits[j]
			  + (word_t)res[i + j] + x;
			res[i + j] = x;
			x >>= 32;
		}
		res[i + j] = x;
	}
	while (nres && res[nres - 1] == 0) nres--;
	y->op = 'f';
	y->fnum.nbits = nres;
	y->fnum.bits = res;
	y->fnum.expt = a->fnum.expt + b->fnum.expt;
	y->fnum.sign = a->fnum.sign ^ b->fnum.sign;
}

static void
divide(expr_t *y, expr_t *a, expr_t *b) {
	unsigned nrem, *rem, *tmp, nres, *res;
	word_t ax, bx, yx;
	unsigned i, j;
	int sign;
	int expt;

	assert(a && a->fnum.bits);
	assert(b && b->fnum.bits);
	expt = a->fnum.expt - b->fnum.expt;
	sign = a->fnum.sign ^ b->fnum.sign;
	nres = DIVIDE_CHUNKS;
	res = xmalloc(nres * sizeof(unsigned));
	if (b->fnum.nbits == 1) {
		/*
		 * Fast & simple version
		 */
		ax = 0;
		bx = b->fnum.bits[0];
		for (i = 1; i < nres; i++) {
			ax <<= 32;
			if (i <= a->fnum.nbits) {
				ax |= a->fnum.bits[a->fnum.nbits - i];
			}
			res[nres - i] = ax / bx;
			ax %= bx;
		}
		expt += 32 * (a->fnum.nbits - DIVIDE_CHUNKS);
		while (nres && res[nres - 1] == 0) nres--;
		y->op = 'f';
		y->fnum.nbits = nres;
		y->fnum.bits = res;
		y->fnum.expt = expt;
		y->fnum.sign = sign;
		return;
	}
	/*
	 * Slow version
	 */
	nrem = a->fnum.nbits;
	if (nrem < b->fnum.nbits) {
		nrem = b->fnum.nbits;
	}
	assert(nrem >= 2);

	/*
	 * Normalize A
	 */
	j = nrem - a->fnum.nbits;
	rem = xmalloc((nrem + 1) * sizeof(unsigned));
	if (j) memset(rem, 0, j * sizeof(unsigned));
	memcpy(rem + j, a->fnum.bits, a->fnum.nbits * sizeof(unsigned));
	expt -= 32 * j;
	for (j = 0, i = 16; i; i >>= 1) {
		if (rem[nrem - 1] < (1u << (32 - j - i))) {
			j += i;
		}
	}
	if (j) {
		expt -= j;
		for (ax = 0, i = 0; i < nrem; i++) {
			ax |= ((word_t)rem[i] << j); rem[i] = ax; ax >>= 32;
		}
		assert(!ax);
	}
	rem[nrem] = 0;

	/*
	 * Normalize B
	 */
	j = nrem - b->fnum.nbits;
	tmp = xmalloc((nrem + 1) * sizeof(unsigned));
	if (j) memset(tmp, 0, j * sizeof(unsigned));
	memcpy(tmp + j, b->fnum.bits, b->fnum.nbits * sizeof(unsigned));
	expt += 32 * j;
	for (j = 0, i = 16; i; i >>= 1) {
		if (tmp[nrem - 1] < (1u << (32 - j - i))) {
			j += i;
		}
	}
	if (j) {
		expt += j;
		for (bx = 0, i = 0; i < nrem; i++) {
			bx |= ((word_t)tmp[i] << j); tmp[i] = bx; bx >>= 32;
		}
		assert(!bx);
	}
	tmp[nrem] = 0;

	bx = tmp[nrem - 1];
	assert(bx & 0x80000000ULL);
	for (i = nres; i-- > 0; ) {
		/*
		 * Divide most significant bits
		 */
		assert(rem[nrem] <= (unsigned)bx);
		ax = ((word_t)rem[nrem] << 32) | rem[nrem - 1];
		yx = ax / bx;
		assert(yx <= 0xffffffffull);
		/*
		 * Substract (r -= y * b)
		 */
		if (yx) {
			int corrs = 0;

			ax = 0;
			for (j = 0; j <= nrem; j++) {
				unsigned t = rem[j];

				ax += yx * tmp[j];
				rem[j] = t - (unsigned)ax;
				ax = (ax >> 32) + (t < (unsigned)ax);
			}
			/*
			 * Correct if Y was too big
			 */
			while (rem[nrem]) {
				assert(corrs++ < 2);	/* at most two steps */
				assert(yx);
				for (ax = 0, j = 0; j <= nrem; j++) {
					ax += (word_t)rem[j] + (word_t)tmp[j];
					rem[j] = ax;
					ax >>= 32;
				}
				yx--;
			}
		}
		assert(rem[nrem] == 0);
		res[i] = yx;
		/*
		 * Shift remainder
		 */
		assert(!rem[nrem]);
		for (j = nrem; j > 0; j--) {
			rem[j] = rem[j - 1];
		}
		rem[0] = 0;
	}
	expt -= 32 * (nres - 1);
	while (nres && res[nres - 1] == 0) nres--;
	assert(nres);
	y->op = 'f';
	y->fnum.nbits = nres;
	y->fnum.bits = res;
	y->fnum.expt = expt;
	y->fnum.sign = sign;
}

static void
power(expr_t *res, unsigned exp) {
	static expr_t powers[MAX_POWERS];
	static unsigned npowers = 0;
	void *bits;

	assert(res);
	assert(exp < MAX_POWERS);
	while (npowers < 4) {
		powers[npowers].op = 'f';
		powers[npowers].fnum.nbits = 1;
		powers[npowers].fnum.bits = xmalloc(sizeof(unsigned));
		powers[npowers].fnum.bits[0] = npowers
			? powers[npowers - 1].fnum.bits[0] * powers[npowers - 1].fnum.bits[0]
			: POWER_BASE;
		powers[npowers].fnum.expt = 0;
		powers[npowers].fnum.sign = 0;
		npowers++;
	}
	while (npowers <= exp) {
		multiply(&powers[npowers], &powers[npowers - 1], &powers[npowers - 1]);
		npowers++;
	}
	bits = res->fnum.bits;
	multiply(res, res, &powers[exp]);
	xfree(bits);
}

static void
powers(expr_t *res, unsigned exp) {
	unsigned i, mask;

	for (i = 0; exp; i++) {
		if (i >= MAX_POWERS) {
			error("exponent too large to handle");
			xfree(res->fnum.bits);
			*res = fconst[0];
			break;
		}
		mask = 1u << i;
		assert(mask);
		if (exp & mask) {
			power(res, i);
			exp &= ~mask;
		}
	}
}

static expr_t*
numeric_expr(const char *val) {
	const char dchars[] = "abcdefABCDEF0123456789.";
	unsigned base = 10;
	unsigned dex = 1;
	int haddot = 0;
	const char *p;
	expr_t bn, *e;
	unsigned digits, chunks;

	assert(val);

	if (*val == '0') {
		switch (*++val) {
			case 'X': case 'x': base = 16; dex = 4; val++; break;
			case 'B': case 'b': base = 2; dex = 1; val++; break;
			default:
				digits = strspn(val, "01234567");
				if (val[digits]) break;
				base = 8; dex = 3;
		}
		while (*val == '0') val++;
		if (!*val) {
			/* shortcut */
			return absexpr(0);
		}
	}
	bn.fnum.expt = 0;
	if (*val == '.') {
		assert(base == 10 || base == 16);	/* scanner ok? */
		while (*++val == '0') {
			/* adjust exponent */
			bn.fnum.expt -= dex;
		}
		haddot = 1;
	}
	assert(*val != '0');

	/* mantissa length */
	digits = strspn(val, base == 16 ? dchars : dchars + 12);
	if (!digits) {
		assert(haddot);
		/* FP zero */
		return &fconst[0];
	}

	/* drop trailing zeroes from FP numbers */
	if (haddot || val[digits] || ((p = strchr(val, '.')) && p < val + digits)) {
		while (val[digits - 1] == '0') --digits;
	}
	p = val + digits;

	/* allocate memory for bignum */
	switch (base) {
		default: assert(0);
		case  2: chunks = (1 * digits + 31) / 32; break;
		case  8: chunks = (3 * digits + 31) / 32; break;
		case 16: chunks = (4 * digits + 31) / 32; break;
		case 10: chunks = (digits + 8) / 9; break;
	}
	bn.fnum.bits = xmalloc(chunks * sizeof(unsigned));
	bn.fnum.bits[0] = 0;
	bn.fnum.nbits = 1;
	bn.fnum.sign = 0;

	/* parse */
	for (; val < p; val++) {
		if (*val == '.') {
			assert(base == 10 || base == 16);	/* scanner ok? */
			assert(!haddot);
			haddot = 1;
		}
		else {
			word_t x = digit(*val & 0xff);
			unsigned i;

			assert(x < base);	/* scanner ok? */
			for (i = 0; i < bn.fnum.nbits; i++) {
				x += (word_t)base * (word_t)bn.fnum.bits[i];
				bn.fnum.bits[i] = (unsigned)x;
				x >>= 32;
			}
			if (x) {
				assert(bn.fnum.nbits < chunks);
				bn.fnum.bits[bn.fnum.nbits++] = x;
			}
			if (haddot) {
				/* adjust exponent */
				assert(base == 10 || base == 16);	/* scanner ok? */
				bn.fnum.expt -= dex;
			}
		}
	}
	while (*val == '0') ++val;
	while (bn.fnum.nbits && !bn.fnum.bits[bn.fnum.nbits - 1]) {
		--bn.fnum.nbits;
	}
	assert(bn.fnum.nbits);	/* we caught zeroes earlier! */

	/* set type and return */
	bn.op = 'f';
	if (!haddot && !*val) {
		word_t v;

		/* integer constant */
		assert(bn.fnum.expt == 0);
		switch (bn.fnum.nbits) {
			case 1:
				v = bn.fnum.bits[0];
				xfree(bn.fnum.bits);
				return absexpr(v);
			case 2:
				v = ((word_t)bn.fnum.bits[1] << 32) | bn.fnum.bits[0];
				xfree(bn.fnum.bits);
				return absexpr(v);
		}
		/* it's a bignum */
		bn.op = 'b';
	}
	else if (base == 16) {
		if (*val) {
			assert(*val == 'p' || *val == 'P');
			bn.fnum.expt += atoi(val + 1);
		}
	}
	else {
		assert(base == 10);	/* scanner ok? */
		if (*val) {
			assert(*val == 'e' || *val == 'E');
			bn.fnum.expt += atoi(val + 1);
		}
		if (bn.fnum.expt < 0) {
			void *bits;
			expr_t p;

			p.fnum.bits = xmalloc(sizeof(unsigned));
			p.fnum.bits[0] = 1;
			p.fnum.nbits = 1;
			p.fnum.expt = 0;
			p.fnum.sign = 0;
			powers(&p, -bn.fnum.expt);
			bits = bn.fnum.bits;
			divide(&bn, &bn, &p);
			xfree(bits);
			xfree(p.fnum.bits);
		}
		else if (bn.fnum.expt > 0) {
			powers(&bn, bn.fnum.expt);
		}
	}
	e = palloc(sizeof(expr_t));
	*e = bn;
	return e;
}

static int
is_true(expr_t *e) {
	if (e != errexpr) {
		e = eval_sym(e);
		switch (e->op) {
			case 'a':
				return e->addr.scn == abs_section && e->addr.off != 0;
			case 'b':
				return 1;	/* bignums are never zero, by definition */
			case 'f':
				return e->fnum.bits != NULL;
		}
		error("absolute expression expected");
	}
	return 0;
}

static expr_t*
register_operand(const char *reg) {
	long rnum;
	expr_t *e;

	rnum = target->encode_register(target, reg);
	if (rnum != -1L) {
		e = palloc(sizeof(expr_t));
		assert(e);
		e->op = 'r';
		e->reg.rnum = rnum;
		return e;
	}
	error("register name expected");
	return errexpr;
}

#ifdef notyet
static expr_t*
memory_operand(expr_t *e) {
	if (target->memory_operand) {
		return target->memory_operand(target, e);
	}
	error("target does not support memory operands");
	return errexpr;
}
#endif

/* allocation */

static void
alloc_bytes(expr_t *size, expr_t *fill) {
	word_t len, val;
	char *dest;

	len = to_int(size);
	val = fill ? to_int(fill) : 0;
	dest = extend_section(current_section, len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	memset(dest, val, len);
}

static void
put_data(char *dest, size_t len, word_t val, int big_endian) {
	size_t i;

	assert(dest);
	if (big_endian) {
		for (i = len; i > 0; ) {
			dest[--i] = val & 0xff;
			val >>= 8;
		}
	}
	else {
		for (i = 0; i < len; ) {
			dest[i++] = val & 0xff;
			val >>= 8;
		}
	}
}

static void
alloc_idata(long size, expr_t *data) {
	char *dest;
	word_t val;
	size_t len;
	sym_t *sym;
	int i;

	assert(size);
	assert(data);
	if (data == errexpr) return;
	len = size < 0 ? -size : size;
	data = eval_sym(data);
	if (nobits_section(current_section)) {
		if (data->op != 'a'
		 || data->addr.scn != abs_section
		 || data->addr.off != 0) {
			error("initializer must be zero");
		}
		return;
	}
	if (data->op == 'b') {
		int carry = 1;

		/* store entire value if possible */
		dest = extend_section(current_section, len);
		for (i = 0; 4 * i < len; i++) {
			unsigned x = i < data->fnum.nbits ? data->fnum.bits[i] : 0;
			int n = 4;

			if (data->fnum.sign) {
				/* XXX: assume 2's complement */
				x = ~x + carry;
				carry = x == 0;
			}
			if (n > len - 4 * i) {
				n = len - 4 * i;
			}
			if (size < 0) {
				put_data(dest + len - 4 * i - n, n, x, size < 0);
			}
			else {
				put_data(dest + 4 * i, n, x, size < 0);
			}
		}
		if (len < 4 * data->fnum.nbits) {
			if (!opt_n) warn("bignum truncated to %d bits", 8 * len);
		}
		return;
	}
	if (data->op != 'a' && data->op != 's') {
		error("integer expression expected");
		data = absexpr(0);
	}
	val = 0;
	if (data->op == 'a' && data->addr.scn == abs_section) {
		val = data->addr.off;
	}
#if 1
	else if (data->op == 's' && data->addr.ref == SREF_PC
	      && (sym = data->addr.sym) && sym->val && sym->val->op == 'a'
	      && sym->val->addr.scn == current_section) {
		/* known value, no need to relocate */
		/* XXX: correct? */
		val = sym->val->addr.off - current_section->size + data->addr.off;
	}
#endif
	else {
		int ref = SREF_NONE;
		reloc_t rel;

		/* try to relocate (may fail) */
		if (data->op == 's') {
			rel.sym = data->addr.sym;
			assert(rel.sym);
			ref = data->addr.ref;
			if (ref == SREF_NONE) {
				assert(!rel.sym->val);
				assert(rel.sym->scn == undef_section || rel.sym->scn == common_section);
			}
		}
		else {
			assert(data->op == 'a');
			assert(data->addr.scn);
			assert(data->addr.scn != abs_section);
			assert(data->addr.scn != undef_section);
			assert(data->addr.scn != common_section);
			rel.sym = section_start_symbol(data->addr.scn);
			assert(rel.sym);
		}
		rel.addend = data->addr.off;
		rel.off = current_section->size;
		/* must come last */
		rel.type = target->reloc_type(target, &rel, size, ref);
		if (rel.type == -1) {
			/* target does not support this */
			error("absolute expression expected");
		}
		else {
			add_reloc_entry(current_section, &rel);
			if (target->elf_reloc_type == ELF_T_REL) {
				val = rel.addend;
			}
		}
	}
	dest = extend_section(current_section, len);
	put_data(dest, len, val, size < 0);
}

static void
alloc_fdata(long size, expr_t *data) {
#define NTMP	5	/* increase if you need > 128 bits of precision */
	char *dest;
	expr_t e;
	int bits;
	int explicit;
	int limit;
	int i, j;
	size_t len;
	unsigned mask;
	unsigned rndbit;
	unsigned shift;
	unsigned tmp[NTMP + 1];

	if (nobits_section(current_section)) {
		error("directive is invalid in NOBITS sections");
		return;
	}
	assert(size);
	assert(data);
	e = *data;
	len = size < 0 ? -size : size;
	assert((1u << len) & 0x11510);	/* len is 4, 8, 10, 12 or 16 */
	/* XXX: automatically convert int to float? */
	if (e.op != 'f') {
		error("floating-point constant expected");
		e = fconst[0];
	}
	if (!e.fnum.nbits) {
		/*
		 * It's a (probably negative) zero; use quick exit
		 */
		dest = extend_section(current_section, len);
		memset(dest, 0, len);
		if (e.fnum.sign) {
			dest[size < 0 ? 0 : len - 1] |= 0x80;
		}
		return;
	}
	assert(e.fnum.bits);

	/* make integer mantissa (almost) fractional */
	e.fnum.expt += 32 * e.fnum.nbits - 1;

	/*
	 * Copy mantissa to temporary buffer
	 */
	i = e.fnum.nbits - NTMP;
	if (i >= 0) {
		/* right shift */
		memcpy(tmp, e.fnum.bits + i, sizeof(tmp));
		while (i-- > 0) {
			/* prepare for round to nearest/even */
			if (e.fnum.bits[i]) {
				tmp[0] |= 1;
				break;
			}
		}
	}
	else {
		/* left shift */
		i = -i;
		memset(tmp, 0, i * sizeof(unsigned));
		memcpy(tmp + i, e.fnum.bits, e.fnum.nbits * sizeof(unsigned));
	}
	assert(tmp[NTMP - 1]);
	tmp[NTMP] = 0;

	/*
	 * Normalize mantissa
	 */
	shift = 0;
	for (i = 16; i; i >>= 1) {
		if (tmp[NTMP - 1] < (1u << (32 - shift - i))) {
			shift += i;
		}
	}
	if (shift) {
		unsigned carry = 0;

		/* shift left */
		for (i = 0; i < NTMP; i++) {
			unsigned x = tmp[i];

			tmp[i] = (x << shift) | carry;
			carry = x >> (32 - shift);
		}
		assert(!carry);
		/* correct exponent */
		e.fnum.expt -= shift;
	}
	assert(tmp[NTMP - 1] & 0x80000000);
	assert(tmp[NTMP] == 0);

	/*
	 * Choose format characteristics
	 * Note: new formats can be added here
	 * It could even support VAX floats, I guess ;-)
	 */
	switch (len) {
		default:	/* undefined */
			assert(0);
		case 4:		/* IEEE single */
			bits = 24;
			explicit = 0;
			e.fnum.expt += 0x7f;
			limit = 0xff;
			break;
		case 8:		/* IEEE double */
			bits = 53;
			explicit = 0;
			e.fnum.expt += 0x3ff;
			limit = 0x7ff;
			break;
		case 10:	/* Intel (ia32) extended */
		case 12:	/* Intel (ia32) extended with padding */
			bits = 64;
			explicit = 1;
			e.fnum.expt += 0x3fff;
			limit = 0x7fff;
			break;
		case 16:	/* SPARC double extended */
			bits = 113;
			explicit = 0;
			e.fnum.expt += 0x3fff;
			limit = 0x7fff;
			break;
	}

	/*
	 * Rounding
	 */
	rndbit = NTMP * 32u - 1 - bits;
	if (e.fnum.expt <= 0) {
		/* result is denormalized later */
		rndbit += 1 - e.fnum.expt;
	}
	j = rndbit / 32u;
	mask = 1u << (rndbit % 32u);
	if (j < NTMP && (tmp[j] & mask)) {
		int round = 0;

		/*
		 * We may have to round up now
		 * - if any less-significant bit is set, or
		 * - if the result would be an odd number
		 */
		if (tmp[j] & (mask - 1)) {
			round = 1;
		}
		else {
			for (i = 0; i < j; i++) {
				if (tmp[i]) {
					round = 1;
					break;
				}
			}
		}
		rndbit++;
		j = rndbit / 32u;
		mask = 1u << (rndbit % 32u);
		if (round || (tmp[j] & mask)) {
			/*
			 * Yes, we do it!
			 */
			tmp[j] += mask;
			if (tmp[j] < mask) {
				/*
				 * Propagate carry
				 */
				assert(tmp[NTMP] == 0);
				for (i = j + 1; i <= NTMP; i++) {
					if ((unsigned)++tmp[i] != 0) {
						break;
					}
				}
			}
			if (tmp[NTMP]) {
				assert(tmp[NTMP] == 1);
				/*
				 * Overflow, shift right
				 *
				 * Since we know how the bits
				 * look like, we can go the
				 * short way this time...
				 */
				tmp[j] >>= 1;
#ifndef NDEBUG
				for (i = j + 1; i < NTMP; i++) assert(tmp[i] == 0);
#endif
				tmp[NTMP - 1] |= 0x80000000;
				tmp[NTMP] = 0;
				/*
				 * Correct exponent!
				 */
				e.fnum.expt++;
			}
		}
	}
	assert(tmp[NTMP - 1] & 0x80000000);
	assert(tmp[NTMP] == 0);

	/*
	 * Exception handling
	 */
	if (e.fnum.expt >= limit) {
		/*
		 * Too big - signal an error and convert to INF
		 */
		error("floating-point overflow");
		memset(tmp, 0, sizeof(tmp));
		e.fnum.expt = limit;
	}
	else if (e.fnum.expt <= 0) {
		/*
		 * Too small - try to denormalize
		 */
		shift = 1 - e.fnum.expt;
		if (shift >= bits) {
			/*
			 * No chance - signal an error and convert to zero
			 */
			error("floating-point underflow");
			memset(tmp, 0, sizeof(tmp));
		}
		else {
			/*
			 * Denormalize
			 */
			if (!opt_n) warn("gradual floating-point underflow");
			if ((j = shift / 32u)) {
				assert(j < NTMP);
				for (i = 0; i + j < NTMP; i++) {
					tmp[i] = tmp[i + j];
				}
				while (i < NTMP) {
					tmp[i++] = 0;
				}
			}
			if ((shift %= 32u)) {
				unsigned carry = 0;

				for (i = NTMP - j; i-- > 0; ) {
					unsigned x = tmp[i];

					tmp[i] = (x >> shift) | carry;
					carry = x << (32 - shift);
				}
			}
		}
		/*
		 * In either case, the biased exponent is 0
		 */
		e.fnum.expt = 0;
	}

	/*
	 * Put all the pieces together
	 */
	tmp[NTMP] = e.fnum.expt;
	if (e.fnum.sign) {
		tmp[NTMP] += limit + 1;
	}
	if (!explicit) {
		if (tmp[NTMP] & 1) {
			tmp[NTMP - 1] |= 0x80000000;
		}
		else {
			tmp[NTMP - 1] &= 0x7fffffff;
		}
		tmp[NTMP] >>= 1;
	}

	/*
	 * Shift them in place
	 */
	shift = NTMP * 32u - bits;
	j = shift / 32u;
	shift %= 32u;
	if (shift) {
		tmp[j] >>= shift;
		for (i = j + 1; i <= NTMP; i++) {
			tmp[i - 1] |= tmp[i] << (32 - shift);
			tmp[i] >>= shift;
		}
	}

	/*
	 * Store the result
	 */
	dest = extend_section(current_section, len);
	if (size < 0) {
		for (i = len - 4; i >= 0; i -= 4) {
			put_data(dest + i, 4, tmp[j++], size < 0);
		}
		if (i > -4) {
			assert(len == 10);
			assert(i == -2);
			assert(j == NTMP);
			assert(tmp[j] <= 0xffff);
			put_data(dest, 2, tmp[j], size < 0);
		}
	}
	else {
		for (i = 0; i + 4 <= len; i += 4) {
			put_data(dest + i, 4, tmp[j++], size < 0);
		}
		if (i < len) {
			assert(len == 10);
			assert(i + 2 == len);
			assert(j == NTMP);
			assert(tmp[j] <= 0xffff);
			put_data(dest + i, 2, tmp[j], size < 0);
		}
	}
}

static void
alloc_string(int addz, const char *val) {
	char *dest;
	size_t len;

	assert(val);
	if (nobits_section(current_section)) {
		error("directive is invalid in NOBITS section");
		return;
	}
	len = strlen(val) + (addz != 0);
	if (len) {
		dest = extend_section(current_section, len);
		memcpy(dest, val, len);
	}
}

static void
alloc_fill(expr_t *repeat, expr_t *size, expr_t *fill) {
	word_t count, len, val;
	char *dest;

	assert(repeat);
	count = to_int(repeat);
	len = size ? to_int(size) : 1;
	val = fill ? to_int(fill) : 0;
	if (len > 8) len = 8;	/* compatibility */
	if (!count || !len) {
		return;
	}
	dest = extend_section(current_section, count * len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	while (count > 0) {
		/* only support native-endian mode */
		put_data(dest, len, val, target->elf_data == ELFDATA2MSB);
		dest += len;
		count--;
	}
}

/* variable-length encoding used by DWARF */
static void
alloc_leb128(int signd, expr_t *data) {
	char temp[(8 * sizeof(word_t) + 6) / 7];
	word_t val;
	size_t len;
	char *dest;

	assert(data);
	if (nobits_section(current_section)) {
		error("invalid directive in NOBITS section");
		return;
	}
	/* XXX: handle bignums? */
	val = to_int(data);
	len = 0;
	if (signd) {
		/* XXX: is this correct? */
		while (val + 0x40 >= 0x80) {
			temp[len++] = 0x80 | (val & 0x7f);
			val = (sword_t)val >> 7;
		}
	}
	else {
		while (val >= 0x80) {
			temp[len++] = 0x80 | (val & 0x7f);
			val >>= 7;
		}
	}
	temp[len++] = val & 0x7f;
	dest = extend_section(current_section, len);
	memcpy(dest, temp, len);
}

static void
alloc_note(const char *str) {
	int big_endian = target->elf_data == ELFDATA2MSB;
	size_t wsize = target->elf_class == ELFCLASS64 ? 8 : 4;
	char *dest;
	size_t len;
	size_t bytes;

	assert(str);
	if (!note_section) {
		note_section = find_section(".note", NULL, NULL, 0);
	}
	len = strlen(str) + 1;
	bytes = 4 * wsize + len - 1; bytes -= bytes % wsize;
	dest = extend_section(note_section, bytes);
	put_data(dest, wsize, len, big_endian); dest += wsize;
	put_data(dest, wsize,   0, big_endian); dest += wsize;
	put_data(dest, wsize,   1, big_endian); dest += wsize;
	if (len) {
		strncpy(dest, str, bytes - 3 * wsize); 
	}
}

static void
align(int mode, expr_t *alignment, expr_t *fill, expr_t *max) {
	word_t how, val, lim, off;
	char *dest;

	assert(alignment);
	how = to_int(alignment);
	val = fill ? to_int(fill) : 0;
	lim = max ? to_int(max) : 0;
	if (mode <= 0) {
		/* logarithmic mode */
		how = 1ULL << how;
	}
	else if (how & (how - 1)) {
		error("alignment must be a power of 2");
		return;
	}
	if (how > target->max_alignment) {
		error("alignment too large");
		return;
	}
	if (how <= 1) {
		return;
	}
	off = current_section->size % how;
	if (!off || (lim && (how - off) > lim)) {
		return;
	}
	if (current_section->align < how) {
		current_section->align = how;
	}
	dest = extend_section(current_section, how -= off);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("initializer must be zero");
		}
		return;
	}
	if (mode < 0) {
		mode = -mode;
	}
	if (mode > 1) {
		/* multi-byte fill */
		if (how % mode) {
			memset(dest, 0, how % mode);
			dest += how % mode;
		}
		while (how >= mode) {
			put_data(dest, mode, val, target->elf_data == ELFDATA2MSB);
			dest += mode;
			how -= mode;
		}
	}
	else {
		memset(dest, val, how);
	}
}

static void
new_org(expr_t *off, expr_t *fill) {
	word_t org, val, cur, len;
	char *dest;

	assert(off);
	if (off == errexpr) {
		return;
	}
	off = eval_sym(off);
	switch (off->op) {
		case 's':
			undefined(off->addr.sym);
			return;
		default:
			error("integer expression expected");
			return;
		case 'a':
			break;
	}
	if (off->addr.scn != current_section && off->addr.scn != abs_section) {
		error("new location must be in current (sub)section");
		return;
	}
	org = off->addr.off;
	val = fill ? to_int(fill) : 0;
	cur = current_section->size;
	if (org < cur) {
		error("cannot move location counter backwards");
	}
	if (org <= cur) {
		return;
	}
	len = org - cur;
	dest = extend_section(current_section, len);
	if (nobits_section(current_section)) {
		if (fill && val) {
			error("fill value must be zero");
		}
		return;
	}
	memset(dest, val, len);
}

/* symbols */

static sym_t *all_syms = NULL;
static sym_t *file_syms = NULL;
static sym_t *current_file_sym = NULL;

static sym_t*
make_symbol(const char *name) {
	sym_t *sym;

	assert(name);
	sym = palloc(sizeof(sym_t));
	sym->next = NULL;
	sym->name = name;
	sym->scn = undef_section;
	sym->val = NULL;
	sym->type = STT_NOTYPE;
	sym->bind = -1;
	sym->size = 0;
	sym->align = 0;
	sym->other = 0;
	sym->used = 0;
	sym->relocs = 0;
	sym->elf_index = STN_UNDEF;
	return sym;
}

static sym_t*
dummy_symbol(const char *name, scn_t *scn, long type) {
	sym_t *sym;

	assert(name);
	assert(scn);
	sym = make_symbol(name);
	sym->scn = scn;
	sym->val = address(scn, 0);
	sym->type = type;
	sym->bind = STB_LOCAL;
	sym->used = 1;
	return sym;
}

static void
file_symbol(const char *name) {
	sym_t **p;

	assert(name);
	for (p = &file_syms; *p; p = &(*p)->next);
	current_file_sym = *p = dummy_symbol(name, abs_section, STT_FILE);
	assert((*p)->next == NULL);
}

static sym_t*
section_start_symbol(scn_t *scn) {
	assert(scn);
	/* Pseudo sections (ABS, UNDEF, COMMON) don't have names */
	/* They should not appear in this place */
	assert(scn->name);
	if (!scn->dummy) {
		scn->dummy = dummy_symbol("", scn, STT_SECTION);
	}
	return scn->dummy;
}

static sym_t*
intern_symbol(const char *name) {
	static unsigned table[10] = { 0 };
	char buf[3*sizeof(*table)+5];
	sym_t **p, *sym;

	assert(name);
	/* handle local labels */
	if (*name >= '0' && *name <= '9') {
		unsigned n;

		n = name[0] - '0';
		assert(n < 10);
		switch (name[1]) {
			case 'b': n = table[n]; break;
			case 'f': n = table[n] + 1; break;
			case '\0': n = table[n] += 1; break;
			default: assert(0);
		}
		sprintf(buf, ".L%c-%u", name[0], n);
		name = buf;
	}
	for (p = &all_syms; (sym = *p); p = &sym->next) {
		/* XXX: sort? */
		if (strcmp(name, sym->name) == 0) {
			return sym;
		}
	}
	if (name == buf) {
		name = xstrdup(name);
	}
	sym = make_symbol(name);
	if (name == buf) {
		sym->bind = STB_LOCAL;
	}
	sym->next = *p;
	*p = sym;
	return sym;
}

static void
set_symbol_value(sym_t *sym, expr_t *val) {
	sym_t *s2;

	assert(sym);
	assert(val);
	if (val == errexpr) return;
	val = eval_sym(val);
	switch (val->op) {
		case 's':
			if (val->addr.ref != SREF_NONE) {
				error("value unknown at assembly time");
				val = absexpr(0);
				break;
			}
			s2 = val->addr.sym;
			assert(s2);
			assert(!s2->val);
			assert(s2->scn == undef_section || s2->scn == common_section);
			/* copy value */
			sym->val = val;
			sym->scn = alias_section;
			/* copy attributes */
			sym->type = s2->type;
			sym->size = s2->size;
			sym->align = s2->align;
			set_symbol_binding(sym, s2->bind);
			return;
		case 'a':
			assert(val->addr.scn != undef_section);
			assert(val->addr.scn != common_section);
			break;
		case 'b':
			error("bignum invalid; assuming zero");
			val = absexpr(0);
			break;
		case 'f':
			error("floating-point number invalid; assuming zero");
			val = absexpr(0);
			break;
		default:
			assert(0);
	}
	sym->scn = val->addr.scn;
	sym->val = val;
}

static void
define_symbol(long unique, sym_t *sym, expr_t *val) {
	assert(sym);
	assert(val);
	assert(sym->scn);
	if (unique && sym->val) {
		error("redefinition of symbol \"%s\"", sym->name);
		return;
	}
	set_symbol_value(sym, val);
}

static void
define_common(long local, sym_t *sym, expr_t *size, expr_t *alignment) {
	word_t val, len, algn;

	assert(sym);
	sym->used = 1;
	assert(size);
	len = to_int(size);
	if (len == 0) {
		error("common symbol \"%s\" is zero bytes long", sym->name);
		/* sanitize */
		len = 1;
	}
	if (alignment) {
		algn = to_int(alignment);
		if (algn & (algn - 1)) {
			error("alignment must be a power of 2");
			return;
		}
		if (algn > target->max_alignment) {
			error("alignment too large");
			return;
		}
	}
	else {
		algn = target->data_alignment;
		assert((algn & (algn - 1)) == 0);
		while (algn > 1 && len % algn) {
			algn >>= 1;
		}
	}
	if (sym->val) {
		error("redefinition of symbol \"%s\"", sym->name);
		return;
	}
	if (local || sym->bind == STB_LOCAL || sym->bind == STB_WEAK) {
		/* allocate space in .bss section */
		val = bss_section->size;
		if (algn > 1) {
			val += algn - 1;
			val -= val % algn;
		}
		sym->scn = bss_section;
		sym->val = address(bss_section, val);
		sym->type = STT_OBJECT;		/* gas uses STT_NOTYPE ?! */
		if (sym->bind == -1) {
			sym->bind = STB_LOCAL;
		}
		sym->size = len;
		bss_section->size = val + len;
		if (bss_section->align < algn) {
			bss_section->align = algn;
		}
	}
	else {
		/* put symbol in dummy section */
		sym->scn = common_section;
		/* sym->val = NULL;		/ * already (un)set */
		sym->type = STT_OBJECT;
		sym->bind = STB_GLOBAL;
		sym->size = len;
		sym->align = algn;
	}
}

static void
set_symbol_binding(sym_t *sym, long bind) {
	assert(sym);
	if (sym->bind == -1) {
		sym->bind = bind;
	}
}

static void
set_symbol_type(sym_t *sym, long type) {
	assert(sym);
	sym->type = type;
}

static void
set_symbol_size(sym_t *sym, word_t size) {
	assert(sym);
	sym->size = size;
}

static void
set_symbol_visibility(sym_t *sym, long what) {
	sym->other = what;	/* may overwrite any old value */
}

static int
is_defined(sym_t *sym) {
	return sym->val != NULL;
}

static void
undefined(sym_t *sym) {
	assert(sym);
	assert(sym->name);
	error("undefined symbol \"%s\"", sym->name);
}

/* the hardest part... */

static void
emit_instruction(const char *name, expr_t *args) {
	target->emit(target, name, args);
}

static void
emit_stab(const char *name, expr_t *type, expr_t *other, expr_t *desc, expr_t *value) {
	word_t val;

	if (!target->emit_stab) {
		error("target does not support stabs");
		return;
	}
	val = value ? to_int(value) : current_section->size;
	target->emit_stab(target, name,
		to_int(type), to_int(other), to_int(desc), val);
}

/* relocations */

static void
add_reloc_entry(scn_t *scn, const reloc_t *rel) {
	size_t n;

	assert(scn);
	assert(rel);
	assert(rel->sym);
	assert(rel->sym->scn != alias_section);
	rel->sym->used = 1;
	rel->sym->relocs = 1;
	if (scn->nrelocs % 16u == 0) {
		n = scn->nrelocs + 16u;
		scn->relocs = xrealloc(scn->relocs, n * sizeof(reloc_t));
	}
	scn->relocs[scn->nrelocs++] = *rel;
}

/* versioning */

static void
symbol_version(sym_t *sym, const char *alias, const char *version, int def) {
	sym_t *sym2;
	char *p, *s;
	int i;

	/* XXX: this is only a mock-up */
	assert(sym);
	assert(alias);
	assert(version);
	s = p = xmalloc(strlen(alias) + strlen(version) + def + 2);
	while ((*s = *alias++)) s++;
	for (i = 0; i <= def; i++) *s++ = '@';
	while ((*s = *version++)) s++;
	*s = '\0';
	sym2 = intern_symbol(p);
	define_symbol(1, sym2, symexpr(sym, 0));
	sym2->used = 1;
	xfree(p);
}

/* F-CPU specific stuff */

#include <fcpu_opcodes.h>
#include <elf_FCPU.h>

static long
fcpu_reloc_type(const target_t *tp, reloc_t *rel, long size, int ref) {
	static const int table[SREF_num][2] = {
		[SREF_NONE]   = { R_FCPU_LE_64,   R_FCPU_BE_64 },
		[SREF_PC]     = { R_FCPU_LE_PC64, R_FCPU_BE_PC64 },
		[SREF_GOT]    = { 0, 0 },
		[SREF_PLT]    = { 0, 0 },
		[SREF_GOTOFF] = { 0, 0 },
		[SREF_GOTPC]  = { 0, 0 },
	};
	long x = -1L;

	assert(tp);
	assert(rel);
	if ((unsigned)ref < SREF_num) {
		switch (size) {
			case  8: x = table[ref][0]; break;
			case -8: x = table[ref][1]; break;
		}
	}
	if (x == 0) x = -1L;
	return x;
}

static long
fcpu_regnum(const target_t *tp, const char *reg) {
	unsigned long rn;
	char *s;

	assert(tp);
	assert(reg);
	if (*reg == 'R' || *reg == 'r') {
		rn = strtoul(reg + 1, &s, 10);
		assert(s);
		if (s != reg + 1 && !*s && rn < 64) {
			return (long)rn;
		}
	}
	return -1L;
}

#define more_args(args)	((args) != NULL)

static expr_t*
unchain_arg(expr_t **list) {
	expr_t *e;

	assert(list);
	if (more_args(*list)) {
		assert((*list)->op == ',');
		e = (*list)->bin.left;
		*list = (*list)->bin.right;
		assert(e->op != ',');
		return e;
	}
	return NULL;
}

static int
expect_register(expr_t **list, int *r) {
	expr_t *e;

	if (!(e = unchain_arg(list)) || e->op != 'r') {
		error("register expected");
		return -1;
	}
	if (r) *r = e->reg.rnum & 0x3f;
	return 0;
}

static int
expect_expr(expr_t **list, expr_t **resp, const char *what) {
	expr_t *e;

	assert(resp);
	if (!(e = unchain_arg(list))) {
		error("integer expression expected");
		return -1;
	}
	if (e == errexpr) {
		return -1;
	}
	e = eval_sym(e);
	if (!strchr(what, e->op)) {
		error("integer expression expected");
		return -1;
	}
	*resp = e;
	return 0;
}

static int
expect_integer(expr_t **list, int *r, int bits) {
	word_t v, w;
	expr_t *e;

	assert(r);
	assert(bits);
	if (expect_expr(list, &e, "a")) {
		return -1;
	}
	assert(e->op == 'a');
	if (e->addr.scn != abs_section) {
		error("absolute expression expected");
		return -1;
	}
	v = e->addr.off;
	/* check range */
	if (bits < 0) {
		w = 1ULL << -bits;
		if (v + (w >> 1) >= w) {
			if (!opt_n) warn("integer constant out of range [-%x;%x]",
				(int)(w >> 1), (int)(w >> 1) - 1);
		}
	}
	else {
		w = 1ULL << bits;
		if (v >= w) {
			if (!opt_n) warn("integer constant out of range [0;%x]",
				(unsigned)w - 1);
		}
	}
	*r = v & (w - 1);
	return 0;
}

static void
put_insn(unsigned opcode, unsigned flags) {
	char *dest;

	opcode = (opcode << 24) | flags;
	dest = extend_section(current_section, 4);
	put_data(dest, 4, opcode, INSTRUCTION_BIG_ENDIAN);
}

static void
fcpu_emit_loadaddri(unsigned code, int sig, expr_t *args) {
	expr_t *e;
	word_t v;
	int r;

	if (expect_expr(&args, &e, "as")) return;
	if (expect_register(&args, &r)) return;
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	e = eval_sym(e);
	if (e->op == 'a' && e->addr.scn == abs_section) {
		/* XXX: assumes that operand is a displacement! */
		v = e->addr.off;
	}
	else if (e->op == 'a' && e->addr.scn == current_section) {
		/* address is known, no need to relocate */
		v = e->addr.off - current_section->size - 4;
	}
	else {
		reloc_t rel;

		if (e->op == 's') {
			rel.sym = e->addr.sym;
			assert(rel.sym);
			switch (e->addr.ref) {
				case SREF_NONE:
					assert(!rel.sym->val);
					assert(rel.sym->scn == undef_section || rel.sym->scn == common_section);
					rel.addend = e->addr.off - 4;
					break;
				case SREF_PC:
				case SREF_PLT:
				case SREF_GOTPC:
					rel.addend = e->addr.off;
					break;
				default:
					error("absolute expression expected");
					return;
			}
		}
		else {
			assert(e->op == 'a');
			assert(e->addr.scn);
			assert(e->addr.scn != undef_section);
			assert(e->addr.scn != common_section);
			assert(e->addr.scn != alias_section);
			rel.sym = section_start_symbol(e->addr.scn);
			assert(rel.sym);
			rel.addend = e->addr.off - 4;
		}
		rel.off = current_section->size;
		rel.type = R_FCPU_PC17;
		add_reloc_entry(current_section, &rel);
		v = 0;
	}
	if ((word_t)(v + 0x10000) >= 0x20000) {
		error("displacement out of range");
		/* sanitize */
		v = 0;
	}
	code |= ((unsigned)v & 0x1ffff) << 6;
	put_insn(0, code | r);
}

static void
fcpu_emit_loadcons(unsigned code, int sig, expr_t *args) {
	expr_t *e;
	word_t v;
	int r;
	int i;

	if (expect_expr(&args, &e, "as")) return;
	if (expect_register(&args, &r)) return;
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	e = eval_sym(e);
	if (e->op == 'a' && e->addr.scn == abs_section) {
		/* absolute expression */
		/* XXX: optimize this further? */
		v = e->addr.off;
		if (code == (OP_LOADCONSX << 24)) {
			/* loadconsx */
			for (i = 0; i < 3 && (word_t)(v + 0x8000) >= 0x10000; i++) {
				put_insn(OP_LOADCONS, (i << 22) | (v & 0xffff) << 6 | r);
				v = (sword_t)v >> 16;
			}
			put_insn(OP_LOADCONSX, (i++ << 22) | (v & 0xffff) << 6 | r);
		}
		else {
			assert(code == (OP_LOADCONS << 24));
			/* loadcons */
			/* XXX: what else can we do? */
			for (i = 0; i < 4; i++) {
				put_insn(OP_LOADCONS, (i << 22) | (v & 0xffff) << 6 | r);
				v = v >> 16;
			}
		}
	}
	else {
		/* XXX: make a difference between loadcons and loadconsx? */
		static const int reloc_types[] = {
			R_FCPU_U_16, R_FCPU_U_32, R_FCPU_U_48, R_FCPU_U_64,
			R_FCPU_U_PC16, R_FCPU_U_PC32, R_FCPU_U_PC48, R_FCPU_U_PC64,
		};
		size_t relbase = 0;
		reloc_t rel;

		if (e->op == 's') {
			rel.sym = e->addr.sym;
			assert(rel.sym);
			switch (e->addr.ref) {
				case SREF_NONE:
					assert(!rel.sym->val);
					assert(rel.sym->scn == undef_section || rel.sym->scn == common_section);
					break;
				case SREF_PC:
				case SREF_PLT:
				case SREF_GOTPC:
					relbase = 4;
					break;
				default:
					error("absolute expression expected");
					return;
			}
		}
		else {
			assert(e->op == 'a');
			assert(e->addr.scn);
			assert(e->addr.scn != undef_section);
			assert(e->addr.scn != common_section);
			assert(e->addr.scn != alias_section);
			rel.sym = section_start_symbol(e->addr.scn);
			assert(rel.sym);
		}
		rel.addend = e->addr.off;
		for (i = 0; i < 4; i++) {
			/* emit dummy load */
			rel.off = current_section->size;
			put_insn(0, code | ((3 - i) << 22) | r);
			/* add relocation entry */
			rel.type = reloc_types[relbase + 3 - i];
			add_reloc_entry(current_section, &rel);
			/* use LOADCONS for subsequent instructions */
			code = OP_LOADCONS << 24;
		}
	}
	if (i > 1) {
		if (!opt_n) warn("emitted %d instructions", i);
	}
}

static void
fcpu_emit(const target_t *tp, const char *name, expr_t *args) {
	extern int fcpu_encode_instruction(const char *name);
	int code, sig, tmp, r;

	assert(tp);
	assert(name);
	if (nobits_section(current_section)) {
		error("assembling instructions not allowed in NOBITS section");
		return;
	}
	/* get opcode + instruction `signature' */
	code = fcpu_encode_instruction(name);
	if (code == -1 || code == -2) {
		/* invalid or unsupported instruction */
		return;
	}
	/* check arguments */
	sig = code & 0x3f;
	code &= ~0x3f;
	switch (sig) {
		default: assert(0);
		case ARG_NONE:	/* (halt, rfe, srb_save, srb_restore, serialize, nop) */
			break;
		case ARG_R:		/* reg (loopentry) */
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_RO:	/* reg[,reg] (jmpa) */
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (more_args(args)) {
				if (expect_register(&args, &r)) return;
				code |= r << 0;
			}
			break;
		case ARG_RR:	/* reg,reg (common) */
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_U16R:	/* uimm16,reg (loadcons.n, geti, puti, syscall, trap) */
			if (expect_integer(&args, &r, 16)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S16R:	/* simm16,reg (loadconsx.n) */
			if (expect_integer(&args, &r, -16)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S17R:	/* simm17,reg (loadaddri) */
			fcpu_emit_loadaddri(code, sig, args);
			return;
		case ARG_U64R:	/* uimm64,reg (generic loadcons) */
		case ARG_S64R:	/* simm64,reg (generic loadconsx) */
			fcpu_emit_loadcons(code, sig, args);
			return;
		case ARG_ORR:	/* [reg,]reg,reg (popcount, load[f], store[f]) */
			if (expect_register(&args, &r)) return;
			if (expect_register(&args, &tmp)) return;
			r = (r << 6) | tmp;
			if (more_args(args)) {
				if (expect_register(&args, &tmp)) return;
				r = (r << 6) | tmp;
			}
			code |= r;
			break;
		case ARG_RRO:	/* reg,reg[,reg] (jmpa{cc}) */
			if (expect_register(&args, &r)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (more_args(args)) {
				if (expect_register(&args, &r)) return;
				code |= r << 0;
			}
			break;
		case ARG_RRR:	/* reg,reg,reg (common) */
			if (expect_register(&args, &r)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_U8RR:	/* uimm8,reg,reg (common) */
			if (expect_integer(&args, &r, 8)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S8RR:	/* simm8,reg,reg (muli, divi, modi?) */
			if (expect_integer(&args, &r, -8)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S9RR:	/* simm9,reg,reg (loadi[f], storei[f]) */
			if (expect_integer(&args, &r, -9)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_231:	/* reg,reg,reg */
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_U8RRoff:	/* uimm8,reg,reg (cmpli, cmpgei) */
			if (expect_integer(&args, &r, 8)) return;
			switch ((unsigned)code >> 24) {
				case OP_CMPLEI: r--; break;
				case OP_CMPGI:  r++; break;
				default: assert(0);
			}
			if (r < 0 || r > 255) {
				error("immediate operand out of range");
			}
			r &= 0xff;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
		case ARG_S8RRoff:	/* simm8,reg,reg (cmplsi, cmpgesi) */
			if (expect_integer(&args, &r, -8)) return;
			r = (signed char)r;
			switch ((unsigned)code >> 24) {
				case OP_CMPLEI: r--; break;
				case OP_CMPGI:  r++; break;
				default: assert(0);
			}
			if (r < -128 || r > 127) {
				error("immediate operand out of range");
			}
			r &= 0xff;
			code |= r << 12;
			if (expect_register(&args, &r)) return;
			code |= r << 6;
			if (expect_register(&args, &r)) return;
			code |= r << 0;
			break;
	}
	if (more_args(args)) {
		error("too many arguments");
		return;
	}
	put_insn(0, code);
}

static scn_t *stab_section = NULL;

void
fcpu_emit_stab(const target_t *tp, const char *name, word_t type, word_t other, word_t desc, word_t value) {
	int be = tp->elf_data == ELFDATA2MSB;
	scn_t *stabstr;
	word_t noff;
	char *dest;

	/*
	 * Create sections
	 */
	if (!stab_section) {
		stab_section = find_section(".stab", NULL, NULL, 0);
		stab_section->type = SHT_PROGBITS;
		stab_section->align = 4;
		stab_section->entsize = 12;
		stab_section->link = find_section(".stabstr", NULL, NULL, 0);
		stab_section->link->type = SHT_STRTAB;
		dest = extend_section(stab_section->link, 1);
		*dest = 0;
		/*
		 * Create first entry
		 */
		fcpu_emit_stab(tp,
			current_file_sym ? current_file_sym->name : source_name,
			0, 0, 0, 0);
	}
	stabstr = stab_section->link;
	assert(stabstr);

	/*
	 * Make new entry
	 */
	noff = 0;
	if (name && *name) {
		noff = stabstr->size;
		dest = extend_section(stabstr, strlen(name) + 1);
		strcpy(dest, name);
	}
	dest = extend_section(stab_section, 12);
	put_data(dest + 0, 4,  noff, be);
	put_data(dest + 4, 1,  type, be);
	put_data(dest + 5, 1, other, be);
	put_data(dest + 6, 2,  desc, be);
	put_data(dest + 8, 4, value, be);

	/*
	 * Patch first entry
	 */
	dest = stab_section->data;
	put_data(dest + 6, 2, stab_section->size / 12 - 1, be);
	put_data(dest + 8, 4, stab_section->link->size, be);
}

target_t
fcpu_target = {
	.reloc_type        = fcpu_reloc_type,
	.encode_register   = fcpu_regnum,
	.emit              = fcpu_emit,
	.emit_stab         = fcpu_emit_stab,
	.memory_operand    = NULL,			/* not supported */
	.text_alignment    = 32,			/* L1 cache line size */
	.data_alignment    = 8,
	.max_alignment     = 1024,			/* minimum page size */
	.elf_class         = ELFCLASS64,
	.elf_data          = ELFDATA2LSB,
	.elf_machine       = EM_FCPU,
	.elf_reloc_size    = sizeof(Elf64_Rela),
	.elf_symbol_size   = sizeof(Elf64_Sym),
	.elf_reloc_type    = ELF_T_RELA,
	.elf_reloc_section = SHT_RELA,
	.elf_alloc_reloc   = 0,
	.elf_alloc_symtab  = 0,
};

/* binary output */

struct bincopy {
	char *data;
	word_t origin;
};

static void
bincopy(struct bincopy *p, long type, long flags, long mask) {
	word_t save;
	scn_t *scn;
	sym_t *sym;

	for (scn = all_scns; scn; scn = scn->next) {
		if (scn->type != type) continue;
		if ((scn->flags ^ flags) & mask) continue;
		save = p->origin;
		if (scn->align > 1 && p->origin % scn->align) {
			p->origin += scn->align - 1;
			p->origin -= p->origin % scn->align;
		}
		p->data = xrealloc(p->data, p->origin + scn->size);
		if (p->origin > save) {
			memset(p->data + save, 0xfc, p->origin - save);
		}
		if (scn->size) {
			assert(scn->data);
			memcpy(p->data + p->origin, scn->data, scn->size);
		}
		sym = section_start_symbol(scn);
		sym->val = absexpr(p->origin);
		p->origin += scn->size;
	}
}

static int
write_bin(const char *fname) {
	const unsigned SHF_ALL = SHF_ALLOC|SHF_WRITE|SHF_EXECINSTR;
	struct bincopy bc = { NULL, 0 };
	word_t wrsize;
	scn_t *scn;
	sym_t *sym;
	size_t i;
	int fd;

	/* XXX: write a.out header? */

	/* text sections */
	bincopy(&bc, SHT_PROGBITS, SHF_ALLOC|SHF_EXECINSTR, SHF_ALLOC|SHF_EXECINSTR);

	sym = intern_symbol("_etext");
	define_symbol(1, sym, absexpr(bc.origin));

	/* data sections */
	bincopy(&bc, SHT_PROGBITS, SHF_ALLOC, SHF_ALL);
	bincopy(&bc, SHT_PROGBITS, SHF_ALLOC|SHF_WRITE, SHF_ALL);
	wrsize = bc.origin;

	sym = intern_symbol("_edata");
	define_symbol(1, sym, absexpr(bc.origin));

	/* bss sections */
	for (scn = all_scns; scn; scn = scn->next) {
		if (scn->type != SHT_NOBITS) continue;
		if (!(scn->flags & SHF_ALLOC)) continue;
		if (scn->align > 1 && bc.origin % scn->align) {
			bc.origin += scn->align - 1;
			bc.origin -= bc.origin % scn->align;
		}
		sym = section_start_symbol(scn);
		sym->val = absexpr(bc.origin);
		bc.origin += scn->size;
	}

	/* allocate common symbols */
	for (sym = all_syms; sym; sym = sym->next) {
		if (sym->scn != common_section) continue;
		if (sym->align > 1 && bc.origin % sym->align) {
			bc.origin += sym->align - 1;
			bc.origin -= bc.origin % sym->align;
		}
		/*
		 * Convert into absolute symbol
		 */
		sym->scn = abs_section;
		sym->val = absexpr(bc.origin);
		bc.origin += sym->size;
	}

	sym = intern_symbol("_end");
	define_symbol(1, sym, absexpr(bc.origin));

	/* apply relocs */
	for (scn = all_scns; scn; scn = scn->next) {
		word_t origin;

		if (!scn->dummy || !scn->dummy->val) continue;
		if (!scn->nrelocs) continue;
		assert(scn->relocs);
		assert(scn->type == SHT_PROGBITS);
		assert(scn->dummy->val->op == 'a');
		assert(scn->dummy->val->addr.scn == abs_section);
		origin = scn->dummy->val->addr.off;
		for (i = 0; i < scn->nrelocs; i++) {
			reloc_t *rel = &scn->relocs[i];
			expr_t *sval = NULL;
			word_t place;
			word_t val;

			assert(rel->sym);
			if (rel->sym->val) {
				sval = eval_sym(rel->sym->val);
			}
			if (!sval) {
				undefined(rel->sym);
				/* sanitize */
				sval = rel->sym->val = absexpr(0);
			}
			else if (sval->op != 'a') {
				error("symbol has no absolute value");
				/* sanitize */
				sval = rel->sym->val = absexpr(0);
			}
			else if (sval->addr.scn != abs_section) {
				sym = sval->addr.scn->dummy;
				if (!sym || !sym->val || sym->val->op != 'a' || sym->val->addr.scn != abs_section) {
					error("relocation references missing section");
					sval = absexpr(0);
				}
				else {
					sval = absexpr(sym->val->addr.off + sval->addr.off);
				}
			}
			place = origin + rel->off;
			val = sval->addr.off + rel->addend;
			switch (rel->type) {
				case R_FCPU_NONE:
					break;
				case R_FCPU_PC17:     /* (S + A - P) & 0x1ffff */
					val -= place;
					if ((word_t)(val + 0x10000) >= 0x20000) {
						error("displacement out of range - cannot relocate");
						val = 0;
					}
					val = (val & 0x1ffff) << 6;
#if INSTRUCTION_BIG_ENDIAN
					bc.data[place+3] = (bc.data[place+3] & 0x3f) | val; val >>= 8;
					bc.data[place+2] = (bc.data[place+2] & 0x00) | val; val >>= 8;
					bc.data[place+1] = (bc.data[place+1] & 0x80) | val; val >>= 8;
					bc.data[place+0] = (bc.data[place+0] & 0xff) | val;
#else
					bc.data[place+0] = (bc.data[place+0] & 0x3f) | val; val >>= 8;
					bc.data[place+1] = (bc.data[place+1] & 0x00) | val; val >>= 8;
					bc.data[place+2] = (bc.data[place+2] & 0x80) | val; val >>= 8;
					bc.data[place+3] = (bc.data[place+3] & 0xff) | val;
#endif
					break;

				case R_FCPU_LE_PC64:  /* S + A - P */
					val -= place;
				case R_FCPU_LE_64:    /* S + A */
					put_data(bc.data + place, 8, val, 0);
					break;

				case R_FCPU_BE_PC64:  /* S + A - P */
					val -= place;
				case R_FCPU_BE_64:    /* S + A */
					put_data(bc.data + place, 8, val, 1);
					break;

				case R_FCPU_U_PC16:   /* ((S + A - P) >>  0) & 0xffff */
					val -= place;
				case R_FCPU_U_16:     /* ((S + A) >>  0) & 0xffff */
					val >>= 0;
					goto insn_16;
				case R_FCPU_U_PC32:   /* ((S + A - P) >> 16) & 0xffff */
					val -= place;
				case R_FCPU_U_32:     /* ((S + A) >> 16) & 0xffff */
					val >>= 16;
					goto insn_16;
				case R_FCPU_U_PC48:   /* ((S + A - P) >> 32) & 0xffff */
					val -= place;
				case R_FCPU_U_48:     /* ((S + A) >> 32) & 0xffff */
					val >>= 32;
					goto insn_16;
				case R_FCPU_U_PC64:   /* ((S + A - P) >> 48) & 0xffff */
					val -= place;
				case R_FCPU_U_64:     /* ((S + A) >> 48) & 0xffff */
					val >>= 48;
				insn_16:
					val = (val & 0xffff) << 6;
#if INSTRUCTION_BIG_ENDIAN
					bc.data[place+3] = (bc.data[place+3] & 0x3f) | val; val >>= 8;
					bc.data[place+2] = (bc.data[place+2] & 0x00) | val; val >>= 8;
					bc.data[place+1] = (bc.data[place+1] & 0xc0) | val; val >>= 8;
					bc.data[place+0] = (bc.data[place+0] & 0xff) | val;
#else
					bc.data[place+0] = (bc.data[place+0] & 0x3f) | val; val >>= 8;
					bc.data[place+1] = (bc.data[place+1] & 0x00) | val; val >>= 8;
					bc.data[place+2] = (bc.data[place+2] & 0xc0) | val; val >>= 8;
					bc.data[place+3] = (bc.data[place+3] & 0xff) | val;
#endif
					break;
				default:
					error("unrecognized relocation type (%u)", (unsigned)rel->type);
			}
		}
	}
	if (errors) {
		return -1;
	}
	if ((fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0666)) == -1) {
		fatal("%s: %s", fname, strerror(errno));
	}
	if (write(fd, bc.data, wrsize) != wrsize) {
		fatal("write error");
	}
	if (close(fd)) {
		fatal("on close");
	}
	return errors ? -1 : 0;
}

/* ELF stuff */

#include <gelf.h>

typedef struct strtab {
	char	*data;
	size_t	size;
} strtab_t;

static size_t
addstr(strtab_t *st, const char *str) {
	size_t off, len, n;

	assert(st);
	assert(str);
	off = st->size;
	if (off == 0) {
		st->data = xmalloc(1024);
		*st->data = '\0';
		st->size = off = 1;
	}
	len = strlen(str);
	if (len == 0) {
		return 0;
	}
	st->size = off + len + 1;
	if ((n = st->size + 1023) / 1024 > (off + 1023) / 1024) {
		st->data = xrealloc(st->data, n - n % 1024);
	}
	strcpy(st->data + off, str);
	return off;
}

static size_t
add2str(strtab_t *st, const char *str1, const char *str2) {
	size_t off, len, n;

	assert(st);
	assert(str1);
	assert(str2);
	off = st->size;
	if (off == 0) {
		st->data = xmalloc(1024);
		*st->data = '\0';
		st->size = off = 1;
	}
	len = strlen(str1) + strlen(str2);
	if (len == 0) {
		return 0;
	}
	st->size = off + len + 1;
	if ((n = st->size + 1023) / 1024 > (off + 1023) / 1024) {
		st->data = xrealloc(st->data, n - n % 1024);
	}
	strcpy(st->data + off, str1);
	strcat(st->data + off, str2);
	return off;
}

static void
elf_error(void) {
	fatal("libelf: %s", elf_errmsg(-1));
}

static void
copy_symbol(strtab_t *names, Elf_Data *data, size_t n, sym_t *sym) {
	const size_t NSYMS = 256;
	GElf_Sym esym;

	/*
	 * make space
	 */
	if (n % NSYMS == 0) {
		data->d_buf = xrealloc(data->d_buf,
			(n + NSYMS) * target->elf_symbol_size);
	}
	data->d_size = (n + 1) * target->elf_symbol_size;

	/*
	 * copy symbol structure
	 */
	if (n == 0) {
		/* first member is always empty */
		assert(!sym);
		esym.st_name = addstr(names, "");
		assert(esym.st_name == 0);
		esym.st_info = 0;
		esym.st_other = 0;
		esym.st_shndx = SHN_UNDEF;
		esym.st_value = 0;
		esym.st_size = 0;
	}
	else {
		assert(sym);
		assert(sym->bind != -1);
		esym.st_name = addstr(names, sym->name);
		esym.st_other = sym->other;
		assert(sym->scn);
		assert(sym->scn != alias_section);
		if (sym->scn == abs_section) {
			assert(sym->val);
			assert(sym->val->op == 'a');
			assert(sym->val->addr.scn == abs_section);
			esym.st_shndx = SHN_ABS;
			esym.st_value = to_int(sym->val);
		}
		else if (sym->scn == undef_section) {
			assert(!sym->val);
			esym.st_shndx = SHN_UNDEF;
			esym.st_value = 0;
			assert(sym->bind == STB_GLOBAL || sym->bind == STB_WEAK);
		}
		else if (sym->scn == common_section) {
			assert(!sym->val);
			esym.st_shndx = SHN_COMMON;
			esym.st_value = sym->align;
			assert(sym->bind == STB_GLOBAL);
		}
		else {
			assert(sym->val);
			assert(sym->val->op == 'a');
			assert(sym->val->addr.scn == sym->scn);
			assert(sym->scn->elf_scn);
			esym.st_shndx = elf_ndxscn(sym->scn->elf_scn);
			esym.st_value = sym->val->addr.off;
		}
		esym.st_info = GELF_ST_INFO(sym->bind, sym->type);
		esym.st_size = sym->size;
		sym->elf_index = n;
	}
	if (!gelf_update_sym(data, n, &esym)) elf_error();
}

static int
local_label(sym_t *sym) {
	return sym->name[0] == '.' && sym->name[1] == 'L'
		&& (unsigned)(sym->name[2] - '0') < 10u
		&& sym->name[3] == '-';
}

static void
fix_symbols(void) {
	sym_t *sym;

	for (sym = all_syms; sym; sym = sym->next) {
		if (sym->scn == undef_section) {
			assert(!sym->val);
			if (sym->bind == STB_LOCAL) {
				if (local_label(sym)) {
					error("instance %s of local label \"%c\" is undefined",
						&sym->name[4], sym->name[2]);
				}
				else {
					undefined(sym);
				}
			}
			if (sym->bind < STB_GLOBAL) {
				sym->bind = STB_GLOBAL;
			}
		}
		else if (sym->bind < STB_LOCAL) {
			sym->bind = STB_LOCAL;
		}
	}
}

static void
fix_relocs(void) {
	scn_t *scn;
	reloc_t *r;
	size_t n;

	for (scn = all_scns; scn; scn = scn->next) {
		for (n = 0; n < scn->nrelocs; n++) {
			r = scn->relocs + n;
			assert(r->sym);
			assert(r->sym->scn);
			assert(r->sym->used);
			assert(r->sym->relocs);
			if (!r->sym->val) {
				assert(r->sym->scn == undef_section
					|| r->sym->scn == common_section);
				continue;
			}
			if (r->sym->bind > STB_LOCAL) continue;
			if (r->sym == r->sym->scn->dummy) continue;
			/* change symbol reference */
			assert(r->sym->val);
			assert(r->sym->val->op == 'a');
			assert(r->sym->val->addr.scn == r->sym->scn);
			r->addend += r->sym->val->addr.off;	/* XXX: really? */
			r->sym = section_start_symbol(r->sym->scn);
			assert(r->sym);
		}
	}
}

static void
write_elf_file(const char *fname) {
	strtab_t scn_names = { NULL, 0 };
	strtab_t sym_names = { NULL, 0 };
	GElf_Ehdr etmp, *ehdr;
	GElf_Shdr stmp, *shdr;
	Elf *elf;
	Elf_Data *data;
	Elf_Scn *escn;
	Elf_Scn *symscn;
	Elf_Scn *strscn;
	scn_t *scn;
	sym_t *sym;
	int fd;
	size_t n;

	if (elf_version(EV_CURRENT) == EV_NONE) elf_error();
	elf_fill(0xFC);	/* padding, just for fun */
	assert(fname);
	if ((fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0666)) == -1) {
		fatal("%s: %s", fname, strerror(errno));
	}
	if (!(elf = elf_begin(fd, ELF_C_WRITE, NULL))) elf_error();
	if (!gelf_newehdr(elf, target->elf_class)) elf_error();
	if (!(ehdr = gelf_getehdr(elf, &etmp))) elf_error();
	ehdr->e_ident[EI_DATA] = target->elf_data;
	ehdr->e_type = ET_REL;
	ehdr->e_machine = target->elf_machine;
	ehdr->e_version = EV_CURRENT;
	ehdr->e_entry = 0;
	ehdr->e_flags = 0;
	/* ehdr->e_shstrndx = 0;	/ * filled in later */
	if (!gelf_update_ehdr(elf, ehdr)) elf_error();
	/*
	 * Create data sections
	 */
	for (scn = all_scns; scn; scn = scn->next) {
		assert(scn->name);
		assert(!scn->elf_scn);
		assert(scn->type == SHT_NOBITS
			|| scn->type == SHT_PROGBITS
			|| scn->type == SHT_STRTAB	/* e.g. .stabstr */
			|| scn->type == SHT_NOTE);
		if (!(escn = elf_newscn(elf))) elf_error();
		if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
		shdr->sh_name = addstr(&scn_names, scn->name);
		shdr->sh_type = scn->type;
		shdr->sh_flags = scn->flags;
		shdr->sh_addr = 0;
		shdr->sh_link = 0;
		shdr->sh_info = 0;
		shdr->sh_entsize = scn->entsize;
		if (!gelf_update_shdr(escn, shdr)) elf_error();
		if (!(data = elf_newdata(escn))) elf_error();
		data->d_buf = scn->data;
		assert(scn->type != SHT_NOBITS || data->d_buf == NULL);
		data->d_type = ELF_T_BYTE;
		data->d_size = scn->size;
		data->d_align = scn->align;
		data->d_version = EV_CURRENT;
		scn->elf_scn = escn;
		if (scn->relocs && scn->nrelocs) {
			static const char *pfx[] = { ".rel", ".rela" };

			/*
			 * Create a relocation section
			 */
			if (!(escn = elf_newscn(elf))) elf_error();
			if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
			shdr->sh_name = add2str(&scn_names,
				pfx[target->elf_reloc_type == ELF_T_RELA], scn->name);
			shdr->sh_type = target->elf_reloc_section;
			shdr->sh_flags = target->elf_alloc_reloc ? SHF_ALLOC : 0;
			shdr->sh_addr = 0;
			shdr->sh_link = 0;		/* symbol table index added later */
			shdr->sh_info = elf_ndxscn(scn->elf_scn);
			shdr->sh_entsize = gelf_fsize(elf, target->elf_reloc_type, 1, EV_CURRENT);
			if (!gelf_update_shdr(escn, shdr)) elf_error();
			scn->elf_rel = escn;
		}
	}
	/*
	 * Establish section links
	 */
	for (scn = all_scns; scn; scn = scn->next) {
		if (!scn->link) continue;
		assert(scn->name);
		assert(scn->elf_scn);
		assert(scn->link->name);
		assert(scn->link->elf_scn);
		assert(scn->link->type == SHT_STRTAB);
		if (!(shdr = gelf_getshdr(scn->elf_scn, &stmp))) elf_error();
		shdr->sh_link = elf_ndxscn(scn->link->elf_scn);
		if (!gelf_update_shdr(scn->elf_scn, shdr)) elf_error();
	}
	/*
	 * Create .symtab / .strtab
	 */
	if (!(symscn = elf_newscn(elf))) elf_error();
	if (!(strscn = elf_newscn(elf))) elf_error();
	if (!(shdr = gelf_getshdr(symscn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".symtab");
	shdr->sh_type = SHT_SYMTAB;
	shdr->sh_flags = target->elf_alloc_symtab ? SHF_ALLOC : 0;
	shdr->sh_addr = 0;
	shdr->sh_link = elf_ndxscn(strscn);
	shdr->sh_info = 0;	/* index of first public symbol */
	shdr->sh_entsize = gelf_fsize(elf, ELF_T_SYM, 1, EV_CURRENT);
	if (!gelf_update_shdr(symscn, shdr)) elf_error();
	if (!(data = elf_newdata(symscn))) elf_error();
	data->d_buf = NULL;
	data->d_type = ELF_T_SYM;
	data->d_size = 0;
	data->d_align = gelf_fsize(elf, ELF_T_ADDR, 1, EV_CURRENT);
	data->d_version = EV_CURRENT;
	/*
	 * Copy symbols
	 */
	n = 0;
	/* first of all, ELF wants a NULL symbol */
	copy_symbol(&sym_names, data, n++, NULL);
	/* second, STT_FILE symbols (if any) */
	for (sym = file_syms; sym; sym = sym->next) {
		assert(sym->type == STT_FILE);
		copy_symbol(&sym_names, data, n++, sym);
	}
	/* third, section symbols (if any) */
	for (scn = all_scns; scn; scn = scn->next) {
		if ((sym = scn->dummy)) {
			assert(sym->type == STT_SECTION);
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	/* fourth, local symbols */
	for (sym = all_syms; sym; sym = sym->next) {
		/* doesn't seem to be what other assemblers do
		if (!sym->used) continue;
		*/
		if (sym->scn == alias_section) continue;
		if (!opt_L) {
			/* detect local symbols by name */
			if (sym->name[0] == '.' && sym->name[1] == 'L') continue;
		}
		if (sym->bind <= STB_LOCAL) {
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	shdr->sh_info = n;
	/* and finally, global symbols */
	for (sym = all_syms; sym; sym = sym->next) {
		if (!sym->val && !sym->used) continue;
		if (sym->scn == alias_section) continue;
		if (sym->bind > STB_LOCAL) {
			copy_symbol(&sym_names, data, n++, sym);
		}
	}
	/* update shdr->sh_info */
	if (!gelf_update_shdr(symscn, shdr)) elf_error();
	/*
	 * Create .strtab
	 */
	if (!(shdr = gelf_getshdr(strscn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".strtab");
	shdr->sh_type = SHT_STRTAB;
	shdr->sh_flags = target->elf_alloc_symtab ? SHF_ALLOC : 0;
	shdr->sh_addr = 0;
	shdr->sh_link = 0;
	shdr->sh_info = 0;
	shdr->sh_entsize = 0;
	if (!gelf_update_shdr(strscn, shdr)) elf_error();
	if (!(data = elf_newdata(strscn))) elf_error();
	data->d_buf = sym_names.data;
	data->d_type = ELF_T_BYTE;
	data->d_size = sym_names.size;
	data->d_align = 0;
	data->d_version = EV_CURRENT;
	/*
	 * Create .shstrtab
	 */
	if (!(escn = elf_newscn(elf))) elf_error();
	if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
	shdr->sh_name = addstr(&scn_names, ".shstrtab");
	shdr->sh_type = SHT_STRTAB;
	shdr->sh_flags = 0;
	shdr->sh_addr = 0;
	shdr->sh_link = 0;
	shdr->sh_info = 0;
	shdr->sh_entsize = 0;
	if (!gelf_update_shdr(escn, shdr)) elf_error();
	if (!(data = elf_newdata(escn))) elf_error();
	data->d_buf = scn_names.data;
	data->d_type = ELF_T_BYTE;
	data->d_size = scn_names.size;
	data->d_align = 0;
	data->d_version = EV_CURRENT;
	ehdr->e_shstrndx = elf_ndxscn(escn);
	if (!gelf_update_ehdr(elf, ehdr)) elf_error();
	/*
	 * Copy relocs
	 */
	for (scn = all_scns; scn; scn = scn->next) {
		if (!(escn = scn->elf_rel)) continue;
		/* update header */
		if (!(shdr = gelf_getshdr(escn, &stmp))) elf_error();
		shdr->sh_link = elf_ndxscn(symscn);
		if (!gelf_update_shdr(escn, shdr)) elf_error();
		/* copy data */
		if (!(data = elf_newdata(escn))) elf_error();
		data->d_type = target->elf_reloc_type;
		data->d_size = scn->nrelocs * target->elf_reloc_size;
		data->d_buf = xmalloc(data->d_size);
		data->d_align = gelf_fsize(elf, ELF_T_ADDR, 1, EV_CURRENT);
		data->d_version = EV_CURRENT;
		for (n = 0; n < scn->nrelocs; n++) {
			reloc_t *r = scn->relocs + n;
			union {
				GElf_Rela rela;
				GElf_Rel rel;
			} u;

			/* assume GElf_Rel is contained in GElf_Rela */
			/* ISO C says it should be, but check anyway */
			assert(&u.rel.r_offset == &u.rela.r_offset);
			assert(&u.rel.r_info   == &u.rela.r_info);

			assert(r->sym);
			assert(r->sym->elf_index);
			u.rela.r_offset = r->off;
			u.rela.r_info = GELF_R_INFO((GElf_Xword)r->sym->elf_index,
										(GElf_Xword)r->type);
			u.rela.r_addend = r->addend;
			if (target->elf_reloc_type == ELF_T_RELA) {
				if (!gelf_update_rela(data, n, &u.rela)) elf_error();
			}
			else {
				if (!gelf_update_rel(data, n, &u.rel)) elf_error();
			}
		}
	}
	/*
	 * And finally, write the file
	 */
	if (!elf_update(elf, ELF_C_WRITE)) elf_error();
	if (elf_end(elf) != 0) {
		fatal("libelf: elf_{begin,end} nesting error");
	}
	if (close(fd)) {
		fatal("%s: %s", fname, strerror(errno));
	}
	xfree(scn_names.data);
	xfree(sym_names.data);
	/* XXX: free symbol table / section data */
}

/* main program */

const char **include_path = NULL;
size_t include_path_num = 0;

void
add_include_path(const char *name) {
	if (include_path_num % 8u == 0) {
		include_path = xrealloc(include_path,
			(include_path_num + 8) * sizeof(const char*));
	}
	include_path[include_path_num++] = name;
}

static void
include_file(const char *name) {
	static char *tmp = NULL;
	size_t len, i;
	FILE *fp;

	assert(name);
	if (*name != '/') {
		for (i = 0; i < include_path_num; i++) {
			len = strlen(include_path[i]) + strlen(name) + 2;
			tmp = xrealloc(tmp, len);
			sprintf(tmp, "%s/%s", include_path[i], name);
			if ((fp = fopen(tmp, "r"))) {
				push_input_file(tmp, fp);
				return;
			}
		}
	}
	if ((fp = fopen(name, "r"))) {
		push_input_file(name, fp);
		return;
	}
	error("cannot find include file: %s", name);
}

static void
usage(const char *pname, int retcode) {
	fprintf(stderr,
		"Usage: %s [options] [srcfile...]\n"
		"Options:\n"
		"  -h           show this message and exit\n"
		"  -I dir       search include files in dir\n"
		"  -L           keep local symbols (starting with `.L')\n"
		"  -n           suppress all warnings\n"
		"  -o outfile   write to outfile instead of a.out\n"
		"  -O format    choose output format\n"
		"               Valid formats are \"bin\" and \"elf\" (default)\n"
		"  -V           print version information and exit\n",
		pname);
	exit(retcode);
}

int
main(int argc, char **argv) {
	int opt_V = 0;
	FILE *fp;
	int i;

	errors = warnings = 0;
	while ((i = getopt(argc, argv, "hI:Lno:O:Q:V")) != EOF) {
		switch (i) {
			case 'I': add_include_path(optarg); break;
			case 'L': opt_L = 1; break;
			case 'n': opt_n = 1; break;
			case 'o': opt_o = optarg; break;
			case 'O': opt_O = optarg; break;
			case 'Q': opt_Q = *optarg; break;
			case 'V': opt_V = 1; break;
			case 'h':
			case '?': usage(*argv, i == '?');
		}
	}
	if (opt_V) {
		fputs("ELF assembler " VERSION_STRING "\n"
			"Copyright (C) 2001 - 2003 Michael \"Tired\" Riepe\n"
			"This program is free software; you can redistribute it and/or\n"
			"modify it under the terms of the GNU General Public License.\n"
			"This program is distributed WITHOUT ANY WARRANTY.\n\n",
			stderr);
		exit(0);
	}
	if (optind < argc) {
		do {
			source_name = argv[optind++];
			if (!(fp = fopen(source_name, "r"))) {
				fatal("%s: %s\n", source_name, strerror(errno));
			}
			set_input_file(source_name, fp);
			yyparse();
			fclose(fp);
		}
		while (optind < argc);
	}
	else {
		set_input_file(source_name, stdin);
		yyparse();
	}
	if (strcmp(opt_O, "bin") == 0) {
		if (write_bin(opt_o)) {
			fprintf(stderr, "%d errors, %d warnings\n", errors, warnings);
			unlink(opt_o);
			exit(1);
		}
		if (warnings) {
			fprintf(stderr, "%d warnings\n", warnings);
		}
	}
	else {
		if (strcmp(opt_O, "elf") != 0) {
			fprintf(stderr, "unknown output format \"%s\", assuming ELF\n", opt_O);
		}
		/* prepare output */
		if (opt_Q != 'y') {
			add_comment("Created with ELF assembler "
				VERSION_STRING " by Michael \"Tired\" Riepe");
		}
		fix_symbols();
		fix_relocs();
		if (errors) {
			fprintf(stderr, "%d errors, %d warnings\n", errors, warnings);
			unlink(opt_o);
			exit(1);
		}
		if (warnings) {
			fprintf(stderr, "%d warnings\n", warnings);
		}
		/* write output file */
		write_elf_file(opt_o);
	}
	exit(0);
}
