-- regfile_custom.vhdl -- F-CPU Register File Implementations
-- Copyright (C) 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id$

library IEEE;
use IEEE.std_logic_1164.all;

-- Not necessary but reduces memory consumption
-- Note that a cell row is not identical with a register
entity Cell_Row is
	generic (
		BITS : integer := 64
	);
	port (
		-- data inputs
		D0, D1 : in std_ulogic;
		-- write enable
		W0, W1 : in std_ulogic_vector(BITS-1 downto 0);
		-- read enable
		R0, R1, r2 : in std_ulogic_vector(BITS-1 downto 0);
	--
		-- data outputs
		Q0, Q1, Q2 : out std_ulogic
	);
end Cell_Row;

architecture Custom of RegFile is
	constant WORDS : integer := 64;	-- do not change
	constant WIDTH : integer := UMAX;

	component Cell_Row
		generic (
			BITS : integer := 64
		);
		port (
			D0, D1 : in std_ulogic;
			W0, W1 : in std_ulogic_vector(BITS-1 downto 0);
			R0, R1, r2 : in std_ulogic_vector(BITS-1 downto 0);
		--
			Q0, Q1, Q2 : out std_ulogic
		);
	end component;

	function decode3to8 (X : in std_ulogic_vector(2 downto 0))
						return std_ulogic_vector is
		variable yy : std_ulogic_vector(7 downto 0);
	begin
		yy(0) := not X(2) and not X(1) and not X(0);
		yy(1) := not X(2) and not X(1) and     X(0);
		yy(2) := not X(2) and     X(1) and not X(0);
		yy(3) := not X(2) and     X(1) and     X(0);
		yy(4) :=     X(2) and not X(1) and not X(0);
		yy(5) :=     X(2) and not X(1) and     X(0);
		yy(6) :=     X(2) and     X(1) and not X(0);
		yy(7) :=     X(2) and     X(1) and     X(0);
		return yy;
	end decode3to8;

	function decode6to64 (X : in std_ulogic_vector(5 downto 0);
						  E : in std_ulogic) return std_ulogic_vector is
		variable aa, bb : std_ulogic_vector(7 downto 0);
		variable yy : std_ulogic_vector(63 downto 0);
	begin
		aa := decode3to8(X(5 downto 3));
		bb := decode3to8(X(2 downto 0));
		for i in 7 downto 0 loop
			for j in 7 downto 0 loop
				yy(8*i+j) := aa(i) and bb(j) and E;
			end loop;
		end loop;
		return yy;
	end decode6to64;

	signal D0, D1, Q0, Q1, Q2 : F_VECTOR;
	signal W0, W1, R0, R1, R2 : std_ulogic_vector(WORDS-1 downto 0);
begin
	-- address decoders
	W0 <= decode6to64(WA0, WE0);
	W1 <= decode6to64(WA1, WE1);
	R0 <= decode6to64(RA0, RE0);
	R1 <= decode6to64(RA1, RE1);
	R2 <= decode6to64(RA2, RE2);

	rows : for bit in WIDTH-1 downto 0 generate
		mem : Cell_Row
			generic map (
				BITS => WORDS
			)
			port map (
				D0 => WD0(bit),
				D1 => WD1(bit),
				W0 => W0,
				W1 => W1,
				R0 => R0,
				R1 => R1,
				R2 => R2,
				Q0 => RD0(bit),
				Q1 => RD1(bit),
				Q2 => RD2(bit)
			);
	end generate;
end Custom;

architecture Low_Level of Cell_Row is
	-- Note: using std_logic because I need a resolved type
	signal pos : std_logic_vector(BITS-1 downto 0);
	signal neg : std_logic_vector(BITS-1 downto 0);
	signal pd0, pd1, nd0, nd1 : std_logic;
	signal p0, p1, p2 : std_logic;
	signal n0, n1, n2 : std_logic;
begin
	pd0 <= '0' when to_X01(D0) = '0' else 'Z';
	nd0 <= '0' when to_X01(D0) = '1' else 'Z';
	pd1 <= '0' when to_X01(D1) = '0' else 'Z';
	nd1 <= '0' when to_X01(D1) = '1' else 'Z';

	-- Note: if both ports write to the same bit, the result is undefined
	write : for bit in BITS-1 downto 0 generate
		-- open drain transistors, two per bit and port
		pos(bit) <= pd0 when W0(bit) = '1' else 'Z';
		neg(bit) <= nd0 when W0(bit) = '1' else 'Z';
		pos(bit) <= pd1 when W1(bit) = '1' else 'Z';
		neg(bit) <= nd1 when W1(bit) = '1' else 'Z';
	end generate;

	latches : for bit in BITS-1 downto 0 generate
		-- two inverters (with weak pull-up) per bit
		-- Note: the different delays force the simulator
		-- into a defined state during initialization and
		-- after ambiguous writes
		pos(bit) <= '0' when to_X01(neg(bit)) = '1' else 'H' after 1 ps;
		neg(bit) <= '0' when to_X01(pos(bit)) = '1' else 'H' after 0 ps;
	end generate;

	p0 <= 'H'; p1 <= 'H'; p2 <= 'H';
	n0 <= 'H'; n1 <= 'H'; n2 <= 'H';

	-- Note: if multiple bits are selected, the result is undefined
	read : for bit in BITS-1 downto 0 generate
		-- open drain transistors, two per bit and port
		p0 <= '0' when (R0(bit) = '1') and (pos(bit) = '0') else 'Z';
		n0 <= '0' when (R0(bit) = '1') and (neg(bit) = '0') else 'Z';
		p1 <= '0' when (R1(bit) = '1') and (pos(bit) = '0') else 'Z';
		n1 <= '0' when (R1(bit) = '1') and (neg(bit) = '0') else 'Z';
		p2 <= '0' when (R2(bit) = '1') and (pos(bit) = '0') else 'Z';
		n2 <= '0' when (R2(bit) = '1') and (neg(bit) = '0') else 'Z';
	end generate;

	Q0 <= '0' when (p0 = '0') else '1' when (n0 = '0') else 'X';
	Q1 <= '0' when (p1 = '0') else '1' when (n1 = '0') else 'X';
	Q2 <= '0' when (p2 = '0') else '1' when (n2 = '0') else 'X';
end Low_Level;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
