-- regfile_test.vhdl - Testbench for F-CPU Register File
-- Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: regfile_test.vhdl,v 1.4 2003/06/27 20:38:04 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;

use work.FCPU_config.all;

entity RegFile_testpart is
	generic (WHICH : integer := 3; STANDALONE : boolean := true);
	port (start : in boolean := true; ready : out boolean);
end RegFile_testpart;

architecture Arch_1 of RegFile_testpart is
	use work.RegFile;	-- make instantiated entity visible

	component RegFile
		port (
			-- write data
			WD0 : in F_VECTOR;
			WD1 : in F_VECTOR;
			-- write address
			WA0 : in std_ulogic_vector(5 downto 0);
			WA1 : in std_ulogic_vector(5 downto 0);
			-- write enable
			WE0 : in std_ulogic;
			WE1 : in std_ulogic;
			-- read address
			RA0 : in std_ulogic_vector(5 downto 0);
			RA1 : in std_ulogic_vector(5 downto 0);
			RA2 : in std_ulogic_vector(5 downto 0);
			-- read enable (ignored by implementation)
			RE0 : in std_ulogic;
			RE1 : in std_ulogic;
			RE2 : in std_ulogic;
		--
			-- read data
			RD0 : out F_VECTOR;
			RD1 : out F_VECTOR;
			RD2 : out F_VECTOR
		);
	end component;

	signal WD0, WD1, RD0, RD1, RD2 : F_VECTOR;
	signal WA0, WA1, RA0, RA1, RA2 : std_ulogic_vector(5 downto 0);
	signal WE0, WE1, RE0, RE1, RE2 : std_ulogic;

	constant std_0 : std_ulogic := '0';
	constant std_1 : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		writestr("WHOA THERE!!!");
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			wait;--return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut0 : if WHICH = 0 generate
		mut : entity work.RegFile(Behave_Generic)
			port map (
				WD0, WD1, WA0, WA1, WE0, WE1,
				RA0, RA1, RA2, RE0, RE1, RE2,
				RD0, RD1, RD2
			);
	end generate;
	mut1 : if WHICH = 1 generate
		mut : entity work.RegFile(Banked_SRAM)
			port map (
				WD0, WD1, WA0, WA1, WE0, WE1,
				RA0, RA1, RA2, RE0, RE1, RE2,
				RD0, RD1, RD2
			);
	end generate;
	mut2 : if WHICH = 2 generate
		mut : entity work.RegFile(Interleaved_SRAM)
			port map (
				WD0, WD1, WA0, WA1, WE0, WE1,
				RA0, RA1, RA2, RE0, RE1, RE2,
				RD0, RD1, RD2
			);
	end generate;
	mut3 : if WHICH = 3 generate
		mut : entity work.RegFile(Custom)
			port map (
				WD0, WD1, WA0, WA1, WE0, WE1,
				RA0, RA1, RA2, RE0, RE1, RE2,
				RD0, RD1, RD2
			);
	end generate;

	-- driver process
	process
		variable tmp : F_VECTOR;
	begin
		if STANDALONE then
			writestr("*** starting simulation ***");
		else
			wait until start;
		end if;

		case WHICH is
			when 0 => writestr("*** testing generic latch-based implementation ***");
			when 1 => writestr("*** testing banked SRAM implementation ***");
			when 2 => writestr("*** testing interleaved SRAM implementation ***");
			when 3 => writestr("*** testing custom SRAM implementation ***");
			when others => null;
		end case;

		RA0 <= (others => 'X');
		RA1 <= (others => 'X');
		RA2 <= (others => 'X');
		RE0 <= '0';
		RE1 <= '0';
		RE2 <= '0';
		WA0 <= (others => 'X');
		WA1 <= (others => 'X');
		WD0 <= (others => 'X');
		WD1 <= (others => 'X');
		WE0 <= '0';
		WE1 <= '0';

		writestr("*** bitwise pass-thru test ***");
		RE0 <= '1';
		for addr in 0 to 63 loop
			WA0 <= std_ulogic_vector(to_unsigned(addr, 6));
			RA0 <= std_ulogic_vector(to_unsigned(addr, 6));
			tmp := (others => '-');
			for bit in 0 to UMAX-1 loop
				for value in std_0 to std_1 loop
					tmp(bit) := value;
					WD0(bit) <= value;
					WE0 <= '1'; wait for 1 ns;
					check_logic("RD0(WE=1)", RD0, tmp);
					WE0 <= '0'; wait for 1 ns;
					check_logic("RD0(WE=0)", RD0, tmp);
				end loop;
				WD0(bit) <= 'X';
				tmp(bit) := '-';
			end loop;
		end loop;
		RE0 <= '0';

		writestr("*** writing data (0) ***");
		for addr in 0 to 63 loop
			WD0 <= (others => '0');
			WD0(addr) <= '1';
			WA0 <= std_ulogic_vector(to_unsigned(addr, 6));
			WE0 <= '1'; wait for 1 ns;
			WE0 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (0) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(addr) := '1';
			RA0 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE0 <= '1'; wait for 1 ns;
			check_logic("RD0", RD0, tmp);
			RE0 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (1) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(addr) := '1';
			RA1 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE1 <= '1'; wait for 1 ns;
			check_logic("RD1", RD1, tmp);
			RE1 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (2) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(addr) := '1';
			RA2 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE2 <= '1'; wait for 1 ns;
			check_logic("RD2", RD2, tmp);
			RE2 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** writing data (1) ***");
		for addr in 0 to 63 loop
			WD1 <= (others => '0');
			WD1(63 - addr) <= '1';
			WA1 <= std_ulogic_vector(to_unsigned(addr, 6));
			WE1 <= '1'; wait for 1 ns;
			WE1 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (0) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(63 - addr) := '1';
			RA0 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE0 <= '1'; wait for 1 ns;
			check_logic("RD0", RD0, tmp);
			RE0 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (1) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(63 - addr) := '1';
			RA1 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE1 <= '1'; wait for 1 ns;
			check_logic("RD1", RD1, tmp);
			RE1 <= '0'; wait for 1 ns;
		end loop;

		writestr("*** reading data (2) ***");
		for addr in 0 to 63 loop
			tmp := (others => '0');
			tmp(63 - addr) := '1';
			RA2 <= std_ulogic_vector(to_unsigned(addr, 6));
			RE2 <= '1'; wait for 1 ns;
			check_logic("RD2", RD2, tmp);
			RE2 <= '0'; wait for 1 ns;
		end loop;

		if STANDALONE then
			writestr("*** simulation complete ***");
			wait;
		else
			ready <= true;
		end if;
	end process;
end Arch_1;

use std.textio.all;

entity RegFile_test is
end RegFile_test;

architecture Arch_1 of RegFile_test is
	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	use work.RegFile_testpart;	-- make instantiated entity visible

	component RegFile_testpart
		generic (WHICH : integer := 3; STANDALONE : boolean := true);
		port (start : in boolean; ready : out boolean);
	end component;

	type bool_vector is array (natural range <>) of boolean;
	signal state : bool_vector(4 downto 0);
begin
	mut : for i in 0 to 3 generate
		part : RegFile_testpart
			generic map (WHICH => i, STANDALONE => false)
			port map (start => state(i), ready => state(i+1));
	end generate;

	process
	begin
		writestr("*** starting simulation ***");
		state(0) <= true;
		wait until state(4);
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
