-- inc.vhdl - Increment Execution Unit for the F-CPU.
-- Copyright (C) 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: inc.vhdl,v 1.4 2003/06/27 20:33:26 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

use work.FCPU_config.all;

entity EU_INC is
	port(
		-- operands
		Din_0    : in F_VECTOR;
		-- Inc flag (should be derived from opcode)
		Inc : in std_ulogic;
		-- Dec flag (should be derived from opcode)
		Dec : in std_ulogic;
		-- Neg flag (should be derived from opcode)
		Neg : in std_ulogic;
		-- Abs flag (should be derived from opcode)
		Xabs : in std_ulogic;	-- `Abs' is reserved in VHDL
		-- Lsb1 flag (should be derived from opcode)
		Lsb1 : in std_ulogic;
		-- Lsb0 flag (should be derived from opcode)
		Lsb0 : in std_ulogic;
		-- flag bits (directly copied from instruction word)
		Flags    : in std_ulogic_vector(13 downto 8);
		-- SIMD mode bits (decoded)
		Size     : in std_ulogic_vector(LOGMAXSIZE-1 downto 0);
		-- clock/reset/enable inputs
		Clk      : in std_ulogic;
		Rst      : in std_ulogic;
		En       : in std_ulogic;
	--
		-- Inc/Dec/Neg result (stage 1)
		Dout_0   : out F_VECTOR;
		-- Abs/Lsb result (stage 2)
		Dout_1   : out F_VECTOR;
		-- Nabs/Mask result (stage 2)
		Dout_2   : out F_VECTOR
	);
end EU_INC;

architecture Struct_1 of EU_INC is
	use work.Inc64;	-- make instantiated entity visible

	component Inc64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			Inc : in std_ulogic;
			Dec : in std_ulogic;
			Neg : in std_ulogic;
			Xabs : in std_ulogic;	-- `Abs' is reserved in VHDL
			Lsb1 : in std_ulogic;
			Lsb0 : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			X : out std_ulogic_vector(WIDTH-1 downto 0);
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;
begin
--pragma synthesis_off
	assert UMAX mod 64 = 0
		report "UMAX must be an integer multiple of 64"
		severity failure;
--pragma synthesis_on

	unit : Inc64
		generic map (WIDTH => UMAX)
		port map (
			A => Din_0,
			Inc => Inc,
			Dec => Dec,
			Neg => Neg,
			Xabs => Xabs,
			Lsb1 => Lsb1,
			Lsb0 => Lsb0,
			U => Size(2 downto 0),
			Clk => Clk,
			Rst => Rst,
			En => En,
			X => Dout_0,
			Y => Dout_1,
			Z => Dout_2
		);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
