-- im64test3.vhdl - F-CPU Integer Multiplication Unit Testbench #3
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: im64test3.vhdl,v 1.11 2003/06/27 20:33:26 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.IMul64_testconf.all;

entity IMul64_test3 is
end IMul64_test3;

architecture Arch_1 of IMul64_test3 is
	use work.IMul64;	-- make instantiated entity visible

	component IMul64
		generic (
			WIDTH : natural := 64;	-- DO NOT CHANGE!
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			X : in std_ulogic_vector(WIDTH-1 downto 0);
			Sig : in std_ulogic;
			Mac : in std_ulogic_vector(2 downto 0);
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y08l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y08h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y16l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y16h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y32l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y32h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y64l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=34 (6 stages)
			Y64h : out std_ulogic_vector(WIDTH-1 downto 0)	-- d=34 (6 stages)
		);
	end component;

	constant WIDTH : natural := 64;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(6 downto 0);
	signal Y08l, Y08h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16l, Y16h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32l, Y32h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64l, Y64h : std_ulogic_vector(WIDTH-1 downto 0);
	signal L, H : std_ulogic_vector(WIDTH-1 downto 0);

	signal Clk, Rst, En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("X := ")); write(lout, X); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;

	procedure pulse (signal x : out std_ulogic; t : time := 1 ns) is
	begin
		wait for t; x <= '1';
		wait for t; x <= '0';
	end pulse;
begin
	mut: IMul64
		generic map (PIPELINED => 1)
		port map (
			A => A, B => B, X => X, Sig => M(0),
			Mac => M(3 downto 1), U => M(6 downto 4),
			Clk => Clk, Rst => Rst, En => En,
			Y08l => Y08l, Y08h => Y08h,
			Y16l => Y16l, Y16h => Y16h,
			Y32l => Y32l, Y32h => Y32h,
			Y64l => Y64l, Y64h => Y64h
		);

	-- output selector
	L <= Y64l when M(6) = '1'
	else Y32l when M(5) = '1'
	else Y16l when M(4) = '1'
	else Y08l;
	H <= Y64h when M(6) = '1'
	else Y32h when M(5) = '1'
	else Y16h when M(4) = '1'
	else Y08h;

	test : process
		procedure pipe_test (pv : in std_ulogic_vector(2*WIDTH-1 downto 0);
							 n : in natural) is
			variable rv : std_ulogic_vector(2*WIDTH-1 downto 0);
		begin
			for i in 1 to n loop
				pulse(Clk);
				rv := H & L;
				check_logic("Y", rv, pv);
			end loop;
		end pipe_test;

		variable pv, rv : std_ulogic_vector(2*WIDTH-1 downto 0);
		variable gran : natural;
		variable lout : line;
	begin
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit pipelining ***"));
			writeline(output, lout);

			-- select mode
			M <= "0000001";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;

			-- reset registers
			-- output data should be '0'
			writestr("--- resetting ---");
			Clk <= '0'; En <= '1';
			pv := (others => '0');
			pulse(Rst); rv := H & L; check_logic("Y", rv, pv);

			writestr("--- pushing X in ---");
			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			pipe_test(pv, 1);

			-- 'X' is still in 2nd stage now, let it propagate
			writestr("--- pushing X almost to the end ---");
			pipe_test(pv, level);

			-- now clock it to the output
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, 1);

			-- clock 0*0 and (-1)*(-1) through the pipe
			writestr("--- pushing single 0 in ---");
			M(0) <= '1';
			A <= (others => '0');
			B <= (others => '0');
			pipe_test(pv, 1);
			A <= (others => 'X');
			B <= (others => 'X');
			pipe_test(pv, level);
			writestr("--- pushing single 0 out ---");
			pv := (others => '0');
			pipe_test(pv, 1);
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, level + 5);

			writestr("--- pushing single -1 in ---");
			A <= (others => '1');
			B <= (others => '1');
			pipe_test(pv, 1);
			A <= (others => 'X');
			B <= (others => 'X');
			pipe_test(pv, level);
			writestr("--- pushing single -1 out ---");
			pv := (others => '0');
			for j in 0 to 64/gran-1 loop
				pv(gran*j) := '1';
			end loop;
			pipe_test(pv, 1);
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, level + 5);
		end loop;

		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit pipelining w/ enable ***"));
			writeline(output, lout);

			-- select mode
			M <= "0000001";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;

			-- reset registers
			-- output data should be '0'
			writestr("--- resetting ---");
			Clk <= '0'; En <= 'X';
			pv := (others => '0');
			pulse(Rst); rv := H & L; check_logic("Y", rv, pv);

			writestr("--- pushing X in ---");
			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			En <= '1';
			pipe_test(pv, 1);
			En <= '0';

			-- 'X' is still in 2nd stage now, let it propagate
			writestr("--- pushing X almost to the end ---");
			pipe_test(pv, level);

			-- now clock it to the output
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, 5);

			-- clock 0*0 and (-1)*(-1) through the pipe
			writestr("--- pushing single 0 in ---");
			M(0) <= '1';
			En <= '1';
			A <= (others => '0');
			B <= (others => '0');
			pipe_test(pv, 1);
			A <= (others => 'X');
			B <= (others => 'X');
			if level = 0 then
				writestr("--- pushing single 0 out ---");
				pv := (others => '0');
				pipe_test(pv, 1);
				En <= '0';
			else
				pipe_test(pv, 1);
				En <= '0';
				pipe_test(pv, level - 1);
				writestr("--- pushing single 0 out ---");
				pv := (others => '0');
				pipe_test(pv, 1);
			end if;
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, 5);

			writestr("--- pushing single -1 in ---");
			En <= '1';
			A <= (others => '1');
			B <= (others => '1');
			pipe_test(pv, 1);
			A <= (others => 'X');
			B <= (others => 'X');
			if level = 0 then
				writestr("--- pushing single -1 out ---");
				pv := (others => '0');
				for j in 0 to 64/gran-1 loop
					pv(gran*j) := '1';
				end loop;
				pipe_test(pv, 1);
				En <= '0';
			else
				pipe_test(pv, 1);
				En <= '0';
				pipe_test(pv, level - 1);
				writestr("--- pushing single -1 out ---");
				pv := (others => '0');
				for j in 0 to 64/gran-1 loop
					pv(gran*j) := '1';
				end loop;
				pipe_test(pv, 1);
			end if;
			writestr("--- pushing X out ---");
			pv := (others => 'X');
			pipe_test(pv, 5);
		end loop;

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
