-- iatest7.vhdl - Testbench for F-CPU IAdd Unit
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: iatest7.vhdl,v 1.17 2003/06/27 20:33:26 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;

use work.Misc.all;

entity IAdd_test is
	generic (
		WIDTH : natural := 64;	-- or any integer multiple of 64
		PIPELINED : integer := 1
	);
end IAdd_test;

architecture Arch_1 of IAdd_test is
	use work.IAdd;	-- make instantiated entity visible

	component IAdd
		generic (
			WIDTH : natural := 64;
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Sub : in std_ulogic;
			Sat : in std_ulogic;
			Sig : in std_ulogic;
			Inc : in std_ulogic;
			Avg : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y8l : out std_ulogic_vector(WIDTH-1 downto 0);
			Y8h : out std_ulogic_vector(WIDTH-1 downto 0);
			Yl : out std_ulogic_vector(WIDTH-1 downto 0);
			Yh : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(7 downto 0) := (others => '0');
	signal A, B, H, L : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Yl, Y8l, Yh, Y8h : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Clk, Rst, En : std_ulogic;

	constant std_0 : std_ulogic := '0';
	constant std_1 : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_mode (simd : in natural) is
		variable lout : line;
	begin
		write(lout, string'("*** testing "));
		write(lout, simd);
		write(lout, string'("-bit mode ***"));
		writeline(output, lout);
	end print_mode;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : IAdd
		generic map (WIDTH => WIDTH, PIPELINED => PIPELINED)
		port map (
			A => A, B => B, U => M(2 downto 0), Sub => M(3),
			Sat => M(4), Inc => M(5), Avg => M(6), Sig => M(7),
			Clk => Clk, Rst => Rst, En => En,
			Y8l => Y8l, Y8h => Y8h, Yl => Yl, Yh => Yh
		);

	-- choose correct output
	L <= Yl when M(0) = '1' else Y8l;
	H <= Yh when M(0) = '1' else Y8h;

	-- driver process
	process
		variable lastl, lasth : std_ulogic_vector(WIDTH-1 downto 0);

		procedure check_last is
		begin
			if PIPELINED /= 0 then
				check_logic("Yl", Yl, lastl);
				check_logic("Yh", Yh, lasth);
				Clk <= '1'; wait for 1 ns;
				Clk <= '0'; wait for 1 ns;
				lastl := Yl;
				lasth := Yh;
			end if;
		end check_last;

		procedure test_add (gran, simd : in natural) is
			variable av, bv, ty, tz : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "00000";	-- add mode
			writestr("*** testing add/add1/addc/addc1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				ty := (others => '-');
				tz := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				-- forward
				for bit in right to left-1 loop
					av(bit+1) := '0';
					bv(bit+1) := '0';
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k;
								ty(bit) := xor3(i, j, k);
								ty(bit+1) := maj23(i, j, k);
								A <= av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
								end if;
								check_last; check_logic("Yl", Yl, ty);
								M(7) <= '1'; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
								end if;
								check_last; check_logic("Yl", Yl, ty);
								M(7) <= '0'; wait for 1 ns;
							end loop;
						end loop;
					end loop;
					av(bit) := '1';
					bv(bit) := '0';
					ty(bit) := '-';
				end loop;
				-- backward
				tz(left downto right) := (left downto right => '0');
				for bit in left downto right loop
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k;
								ty(bit) := xor3(i, j, k);
								tz(right) := maj23(i, j, k);
								A <= av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
									check_logic("Y8h", Y8h, tz);
								end if;
								check_last;
								check_logic("Yl", Yl, ty);
								check_logic("Yh", Yh, tz);
								M(7) <= '1'; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
									check_logic("Y8h", Y8h, tz);
								end if;
								check_last;
								check_logic("Yl", Yl, ty);
								check_logic("Yh", Yh, tz);
								M(7) <= '0'; wait for 1 ns;
							end loop;
						end loop;
					end loop;
					av(bit) := '0';
					bv(bit) := '1';
					ty(bit) := '-';
				end loop;
			end loop;
		end test_add;

		procedure test_sub (gran, simd : in natural) is
			variable av, bv, ty, tz : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "00001";	-- sub mode
			writestr("*** testing sub/sub1/subb/subb1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				ty := (others => '-');
				tz := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				-- forward
				for bit in right to left-1 loop
					av(bit+1) := '0';
					bv(bit+1) := '0';
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k;
								ty(bit) := xor3(i, j, k);
								ty(bit+1) := maj23(not i, j, k);
								A <= av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
								end if;
								check_last; check_logic("Yl", Yl, ty);
								M(7) <= '1'; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
								end if;
								check_last; check_logic("Yl", Yl, ty);
								M(7) <= '0'; wait for 1 ns;
							end loop;
						end loop;
					end loop;
					av(bit) := '0';
					bv(bit) := '0';
					ty(bit) := '-';
				end loop;
				-- backward
				tz(left downto right) := (left downto right => '0');
				for bit in left downto right loop
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k;
								ty(bit) := xor3(i, j, k);
								tz(right) := maj23(not i, j, k);
								A <= av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
									check_logic("Y8h", Y8h, tz);
								end if;
								check_last;
								check_logic("Yl", Yl, ty);
								check_logic("Yh", Yh, tz);
								M(7) <= '1'; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8l", Y8l, ty);
									check_logic("Y8h", Y8h, tz);
								end if;
								check_last;
								check_logic("Yl", Yl, ty);
								check_logic("Yh", Yh, tz);
								M(7) <= '0'; wait for 1 ns;
							end loop;
						end loop;
					end loop;
					av(bit) := '1';
					bv(bit) := '1';
					ty(bit) := '-';
				end loop;
			end loop;
		end test_sub;

		procedure test_addus (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "00010";	-- add with unsigned saturation
			writestr("*** testing addus/addus1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				tmp(left downto right) := (left downto right => '1');
				for bit in left downto right loop
					av(bit) := '1';
					bv(bit) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '0';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
		end test_addus;

		procedure test_subus (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "00011";	-- subtract with unsigned saturation
			writestr("*** testing subus/subus1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				tmp(left downto right) := (left downto right => '0');
				for bit in left downto right loop
					av(bit) := '0';
					bv(bit) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '0';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
		end test_subus;

		procedure test_addss (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "10010";	-- add with signed saturation
			writestr("*** testing addss/addss1 instructions ***");
			-- positive overflow
			-- Note: this happens only when both operands are positive
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				av(left) := '0';
				bv(left) := '0';
				tmp(left) := '0';
				tmp(left-1 downto right) := (left-1 downto right => '1');
				for bit in left-1 downto right loop
					av(bit) := '1';
					bv(bit) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '0';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
			-- negative overflow
			-- Note: this happens only when both operands are negative
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				av(left) := '1';
				bv(left) := '1';
				tmp(left) := '1';
				tmp(left-1 downto right) := (left-1 downto right => '0');
				for bit in left-1 downto right loop
					av(bit) := '0';
					bv(bit) := '0';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '1';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
		end test_addss;

		procedure test_subss (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "10011";	-- subtract with signed saturation
			writestr("*** testing subss/subss1 instructions ***");
			-- positive overflow
			-- Note: this only happens if a >= 0 and b < 0
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				av(left) := '0';
				bv(left) := '1';
				tmp(left) := '0';
				tmp(left-1 downto right) := (left-1 downto right => '1');
				for bit in left-1 downto right loop
					av(bit) := '1';
					bv(bit) := '0';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '1';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
			-- negative overflow
			-- Note: this only happens if a < 0 and b >= 0
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				av(left) := '1';
				bv(left) := '0';
				tmp(left) := '1';
				tmp(left-1 downto right) := (left-1 downto right => '0');
				for bit in left-1 downto right loop
					av(bit) := '0';
					bv(bit) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_last; check_logic("Yl", Yl, tmp);
					bv(bit) := '0';
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '1';
				A <= av; B <= bv; wait for 1 ns;
				check_last; check_logic("Yl", Yl, tmp);
				M(5) <= '0';
			end loop;
		end test_subss;

		procedure test_avg (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "X1000";	-- signed/unsigned average
			writestr("*** testing avg/avg1/avgs/avgs1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				for bit in right+1 to left loop
					av(bit-1) := '1';
					bv(bit-1) := '0';
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k; A <= av; B <= bv;
								tmp(bit-1) := xor3(i, j, k);
								M(7) <= '0'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
								M(7) <= '1'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
							end loop;
						end loop;
					end loop;
					tmp(bit-1) := '-';
				end loop;
				-- carry logic
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for bit in right+1 to left loop
					av(bit) := '0';
					bv(bit) := '0';
					for i in std_0 to std_1 loop
						av(bit-1) := i;
						for j in std_0 to std_1 loop
							bv(bit-1) := j;
							for k in std_0 to std_1 loop
								M(5) <= k; A <= av; B <= bv;
								tmp(bit-1) := maj23(i, j, k);
								M(7) <= '0'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
								M(7) <= '1'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
							end loop;
						end loop;
					end loop;
					av(bit-1) := '1';
					bv(bit-1) := '0';
					tmp(bit-1) := '-';
				end loop;
				-- msbit
				for i in std_0 to std_1 loop
					av(left) := i;
					for j in std_0 to std_1 loop
						bv(left) := j;
						for k in std_0 to std_1 loop
							M(5) <= k; A <= av; B <= bv;
							tmp(left) := maj23(i, j, k);
							M(7) <= '0'; wait for 1 ns;
							check_last; check_logic("Yh", Yh, tmp);
							tmp(left) := xor3(i, j, tmp(left));
							M(7) <= '1'; wait for 1 ns;
							check_last; check_logic("Yh", Yh, tmp);
						end loop;
					end loop;
					tmp(left) := '-';
				end loop;
			end loop;
		end test_avg;

		procedure test_diff (gran, simd : in natural) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right : natural;
			variable lout : line;
		begin
			M(7 downto 3) <= "X1001";	-- signed/unsigned difference
			writestr("*** testing diff/diff1/diffs/diffs1 instructions ***");
			for slice in 0 to WIDTH/simd-1 loop
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				right := slice * simd;
				left := right + simd - 1;
				for bit in right+1 to left loop
					av(bit-1) := '0';
					bv(bit-1) := '0';
					for i in std_0 to std_1 loop
						av(bit) := i;
						for j in std_0 to std_1 loop
							bv(bit) := j;
							for k in std_0 to std_1 loop
								M(5) <= k; A <= av; B <= bv;
								tmp(bit-1) := xor3(i, j, k);
								M(7) <= '0'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
								M(7) <= '1'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
							end loop;
						end loop;
					end loop;
					tmp(bit-1) := '-';
				end loop;
				-- carry logic
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for bit in right+1 to left loop
					av(bit) := '0';
					bv(bit) := '0';
					for i in std_0 to std_1 loop
						av(bit-1) := i;
						for j in std_0 to std_1 loop
							bv(bit-1) := j;
							for k in std_0 to std_1 loop
								M(5) <= k; A <= av; B <= bv;
								tmp(bit-1) := maj23(not i, j, k);
								M(7) <= '0'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
								M(7) <= '1'; wait for 1 ns;
								check_last; check_logic("Yh", Yh, tmp);
							end loop;
						end loop;
					end loop;
					av(bit-1) := '1';
					bv(bit-1) := '1';
					tmp(bit-1) := '-';
				end loop;
				-- msbit
				for i in std_0 to std_1 loop
					av(left) := i;
					for j in std_0 to std_1 loop
						bv(left) := j;
						for k in std_0 to std_1 loop
							M(5) <= k; A <= av; B <= bv;
							tmp(left) := maj23(not i, j, k);
							M(7) <= '0'; wait for 1 ns;
							check_last; check_logic("Yh", Yh, tmp);
							tmp(left) := xor3(i, j, tmp(left));
							M(7) <= '1'; wait for 1 ns;
							check_last; check_logic("Yh", Yh, tmp);
						end loop;
					end loop;
					tmp(left) := '-';
				end loop;
			end loop;
		end test_diff;

		variable simd : natural;
	begin
		writestr("*** starting simulation ***");
		if PIPELINED /= 0 then
			writestr("*** pipelining is ON ***");
		else
			writestr("*** pipelining is OFF ***");
		end if;
		Clk <= '0'; Rst <= '1'; wait for 1 ns;
		Rst <= '0'; En <= '1'; wait for 1 ns;
		lastl := (others => '-');
		lasth := (others => '-');
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			print_mode(simd);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			test_add(gran, simd);
			test_sub(gran, simd);
			test_avg(gran, simd);
			test_diff(gran, simd);
			test_addus(gran, simd);
			test_subus(gran, simd);
			test_addss(gran, simd);
			test_subss(gran, simd);
		end loop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;
