-- afun_test.vhdl -- Testbench for Additional Functions Unit
-- Copyright (C) 2001 - 2004 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: afun_test.vhdl,v 1.5 2004/01/23 15:30:38 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;
use work.Bit_Manipulation.all;

entity AFun_test is
	generic (WIDTH : natural := 64);
end AFun_test;

architecture Arch_1 of AFun_test is
	use work.AFun;	-- make instantiated entity visible

	component AFun
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Hamming : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Popc08 : out std_ulogic_vector(WIDTH-1 downto 0);
			Popc16 : out std_ulogic_vector(WIDTH-1 downto 0);
			Popc32 : out std_ulogic_vector(WIDTH-1 downto 0);
			Popc64 : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(3 downto 0);
	signal A : std_ulogic_vector(WIDTH-1 downto 0);
	signal B : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y08 : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16 : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32 : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64 : std_ulogic_vector(WIDTH-1 downto 0);
	signal Clk : std_ulogic;
	signal Rst : std_ulogic;
	signal En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("B", B);
		print_vector("M", M);
		print_vector("Y", Y);
	end print_signals;

	procedure do_report (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;

	procedure print_mode (bits : in natural) is
		variable lout : line;
	begin
		write(lout, string'("*** testing "));
		write(lout, bits);
		write(lout, string'("-bit mode ***"));
		writeline(output, lout);
	end print_mode;

	procedure pulse (signal x : out std_ulogic;
					 n : in natural := 1;
					 t : in time := 1 ns) is
	begin
		for i in 1 to n loop
			wait for t; x <= '1';
			wait for t; x <= '0';
		end loop;
	end pulse;
begin
	-- module under test
	mut : AFun
		generic map (WIDTH => WIDTH)
		port map (
			A => A,
			B => B,
			Hamming => M(3),
			U => M(2 downto 0),
			Clk => Clk,
			Rst => Rst,
			En => En,
			Popc08 => Y08,
			Popc16 => Y16,
			Popc32 => Y32,
			Popc64 => Y64
		);

	Y <= Y64 when M(2) = '1'
	else Y32 when M(1) = '1'
	else Y16 when M(0) = '1'
	else Y08;

	-- driver process
	process
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';

		function bitcount (A : in std_ulogic_vector) return natural is
			variable yy : natural;
		begin
			yy := 0;
			for i in A'range loop
				if to_X01(A(i)) = '1' then
					yy := yy + 1;
				end if;
			end loop;
			return yy;
		end bitcount;

		procedure test_popc is
			type deltab is array (3 downto 0) of natural;
			constant delay_table : deltab := (
				0 => 2, 1|2 => 3, 3 => 4
			);

			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
			variable k : natural;
		begin
			writestr("*** testing popcount instruction ***");
			for gran in 0 to 0 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "0XXX";
				B <= (others => 'X');
				for i in 0 to 2 loop
					if i < gran then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					for aa in 0 to 255 loop
						av(left downto right) := std_ulogic_vector(to_unsigned(aa, simd));
						k := bitcount(av(left downto right));
						tmp(left downto right) := std_ulogic_vector(to_unsigned(k, simd));
						A <= av; pulse(Clk, delay_table(gran));
						check_logic("Y", Y, tmp);
					end loop;
				end loop;
			end loop;
			for gran in 1 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "0XXX";
				B <= (others => 'X');
				for i in 0 to 2 loop
					if i < gran then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					for aa in 0 to simd/2 loop
						for bb in 0 to simd/2 loop
							av(left downto right) := (left downto right => '0');
							for i in 1 to aa loop
								av(right+i-1) := '1';
							end loop;
							for i in 1 to bb loop
								av(left-i+1) := '1';
							end loop;
							tmp(left downto right) :=
								std_ulogic_vector(to_unsigned(aa+bb, simd));
							A <= av; pulse(Clk, delay_table(gran));
							check_logic("Y", Y, tmp);
						end loop;
					end loop;
				end loop;
			end loop;
		end test_popc;

		procedure test_hamming is
			type deltab is array (3 downto 0) of natural;
			constant delay_table : deltab := (
				0 => 2, 1|2 => 3, 3 => 4
			);

			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable bv : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
			variable k : natural;
		begin
			writestr("*** testing hamming instruction ***");
			for gran in 0 to 0 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "1XXX";
				for i in 0 to 2 loop
					if i < gran then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					bv(left downto right) := (left downto right => '1');
					tmp := (others => '-');
					for aa in 0 to 255 loop
						av(left downto right) := std_ulogic_vector(to_unsigned(aa, simd));
						k := bitcount(not av(left downto right));
						tmp(left downto right) := std_ulogic_vector(to_unsigned(k, simd));
						A <= av; B <= bv; pulse(Clk, delay_table(gran));
						check_logic("Y", Y, tmp);
					end loop;
				end loop;
			end loop;
			for gran in 1 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "1XXX";
				for i in 0 to 2 loop
					if i < gran then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					bv(left downto right) := (left downto right => '1');
					tmp := (others => '-');
					for aa in 0 to simd/2 loop
						for bb in 0 to simd/2 loop
							av(left downto right) := (left downto right => '1');
							for i in 1 to aa loop
								av(right+i-1) := '0';
							end loop;
							for i in 1 to bb loop
								av(left-i+1) := '0';
							end loop;
							tmp(left downto right) :=
								std_ulogic_vector(to_unsigned(aa+bb, simd));
							A <= av; B <= bv; pulse(Clk, delay_table(gran));
							check_logic("Y", Y, tmp);
						end loop;
					end loop;
				end loop;
			end loop;
		end test_hamming;
	begin
		Clk <= '0'; Rst <= '0'; En <= '1';

		test_popc;
		test_hamming;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
