-- omega_network.vhdl -- bit shuffling with an omega network
-- Copyright (C) 2001 - 2004 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: omega_network.vhdl,v 1.1 2004/01/25 05:41:27 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

package Omega_Network is
	-- normalize signed and unsigned operands
	-- Note: A'length must be a power of two
	function normalize (A : in std_ulogic_vector;
						Sig : in std_ulogic) return std_ulogic_vector;

	-- single stage of an omega network
	-- Note: result'length = A'length = 2 * X'length
	function omega_F_1 (A, X : in std_ulogic_vector) return std_ulogic_vector;

	-- single omega network stage, reversed
	-- Note: result'length = A'length = 2 * X'length
	function omega_R_1 (A, X : in std_ulogic_vector) return std_ulogic_vector;
end Omega_Network;

package body Omega_Network is
	use work.Bit_Manipulation.all;

--pragma synthesis_off
	function power_of_two (X : in natural) return boolean is
	begin
		for i in 0 to 30 loop
			exit when X < 2 ** i;
			next when X > 2 ** i;
			-- this is what we were looking for
			return true;
		end loop;
		return false;
	end power_of_two;
--pragma synthesis_on

	-- single stage of an omega network
	function omega_F_1 (A, X : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		constant N : natural := L / 2;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable xx : std_ulogic_vector(N-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
--pragma synthesis_off
		assert A'length = L;
		assert X'length = N;
--pragma synthesis_on
		aa := A;
		xx := to_X01(X);
		yy := (others => 'X');
		for i in N-1 downto 0 loop
			if xx(i) = '1' then
				yy(2*i+0) := aa(i+N);
				yy(2*i+1) := aa(i+0);
			else
				yy(2*i+0) := aa(i+0);
				yy(2*i+1) := aa(i+N);
			end if;
		end loop;
		return yy;
	end omega_F_1;

	-- single omega network stage, reversed
	function omega_R_1 (A, X : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		constant N : natural := L / 2;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable xx : std_ulogic_vector(N-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
--pragma synthesis_off
		assert A'length = L;
		assert X'length = N;
--pragma synthesis_on
		aa := A;
		xx := to_X01(X);
		yy := (others => 'X');
		for i in N-1 downto 0 loop
			if xx(i) = '1' then
				yy(i+N) := aa(2*i+0);
				yy(i+0) := aa(2*i+1);
			else
				yy(i+0) := aa(2*i+0);
				yy(i+N) := aa(2*i+1);
			end if;
		end loop;
		return yy;
	end omega_R_1;

	function normalize (A : in std_ulogic_vector;
						Sig : in std_ulogic) return std_ulogic_vector is
		constant L : natural := A'length;
		constant N : natural := L / 2;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable bb : std_ulogic_vector(L-1 downto 0);
		variable cc : std_ulogic_vector(N-1 downto 0);
		variable tc : std_ulogic_vector(N-1 downto 0);
		variable mm : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
		variable stride : natural;
		variable jumps : natural;
		variable t : std_ulogic;
		variable k : integer;
	begin
--pragma synthesis_off
		assert power_of_two(N);
--pragma synthesis_on
		aa := A;

		-- bb indicates how much to shift:
		-- if bb(L-1) = '1' then do not shift at all
		-- else if bb(L-2) = '1' then shift by 1
		-- else if bb(L-3) = '1' then shift by 2
		-- and so on until
		-- else if bb(0) = '1' then shift by L-1
		-- else shift by L (which will return zero)

		if to_X01(Sig) = '1' then
			bb := aa xor lshift(aa, 1);
		else
			bb := aa;
		end if;
		bb := bit_reverse(cascade_or(bit_reverse(bb)));

		-- mask off high bits (which will become low bits after rotating)
		yy := aa and bb;

		-- build as many stages as necessary to rotate L bits
		-- allows a maximum width of 2**31 bits (which is utopic)
		for level in 30 downto 0 loop	-- range MUST be descending!
			stride := 2 ** level;
			next when stride >= L;	-- nothing more to do at this level
--pragma synthesis_off
			assert stride <= N;
--pragma synthesis_on
			jumps := N / stride;
			cc := (others => 'X');
			for j in 1 to stride loop
				k := L - j;
				tc := (others => 'X');
				for i in jumps-1 downto 0 loop
					tc(i) := bb(k) xor bb(k-stride);
					k := k - 2 * stride;
				end loop;
				if jumps > 1 then
					t := reduce_or(tc(jumps-1 downto 0));
				else
					t := tc(0);
				end if;
				k := N - jumps * j;
				cc(k+jumps-1 downto k) := (k+jumps-1 downto k => t);
			end loop;
			yy := omega_F_1(yy, cc);
			-- Note:
			-- shift_count(level) := cc(N-1);
		end loop;
		return yy;
	end normalize;
end Omega_Network;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
