-- generic_adder_test.vhdl -- Testbench for Generic Adder
-- Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- $Id: generic_adder_test.vhdl,v 1.7 2003/04/12 15:09:02 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;
use work.Generic_Adder.all;

entity Generic_Adder_test is
	generic (WIDTH : natural := 929);
end Generic_Adder_test;

architecture Arch_1 of Generic_Adder_test is
	signal A, B : std_ulogic_vector(WIDTH-1 downto 0) := (others => '1');
	signal Y, Z : std_ulogic_vector(WIDTH-1 downto 0);
	signal G, P : std_ulogic_vector(0 downto 0);

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_signals is
		variable lout : line;
	begin
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, string'("Z := ")); write(lout, Z); writeline(output, lout);
		write(lout, string'("G := ")); write(lout, G); writeline(output, lout);
		write(lout, string'("P := ")); write(lout, P); writeline(output, lout);
	end print_signals;

	procedure do_report (lbl : string; x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		print_signals;
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : process (A, B)
		variable yy, zz : std_ulogic_vector(WIDTH-1 downto 0);
		variable gg, pp : std_ulogic;
	begin
		CSAdd(A, B, yy, zz, gg, pp);
		Y <= yy;
		Z <= zz;
		G(0) <= gg;
		P(0) <= pp;
	end process;

	-- driver process
	run : process
		constant std_0 : std_ulogic := '0';
		constant std_1 : std_ulogic := '1';
		variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable lout : line;
	begin
		write(lout, string'("*** testing generic adder ("));
		write(lout, WIDTH);
		write(lout, string'(" bits wide) ***"));
		writeline(output, lout);

		-- outputs
		writestr("*** testing outputs ***");
		av := (others => 'X');
		bv := (others => 'X');
		tmp := (others => '-');
		for bit in 0 to WIDTH-1 loop
			for i in std_0 to std_1 loop
				av(bit) := i;
				for j in std_0 to std_1 loop
					bv(bit) := j;
					A <= av; B <= bv; wait for 1 ns;
					tmp(bit) := i xor j;
					check_logic("Y", Y, tmp);
					tmp(bit) := not tmp(bit);
					check_logic("Z", Z, tmp);
				end loop;
			end loop;
			av(bit) := '1';
			bv(bit) := '0';
			tmp(bit) := '-';
		end loop;

		-- carry logic
		writestr("*** testing carry chain ***");
		av := (others => 'X');
		bv := (others => 'X');
		tmp := (others => '-');
		for bit in 1 to WIDTH-1 loop
			av(bit) := '0';
			bv(bit) := '0';
			for i in std_0 to std_1 loop
				av(bit-1) := i;
				for j in std_0 to std_1 loop
					bv(bit-1) := j;
					A <= av; B <= bv; wait for 1 ns;
					tmp(bit) := i and j;
					check_logic("Y", Y, tmp);
					tmp(bit) := i or j;
					check_logic("Z", Z, tmp);
				end loop;
			end loop;
			av(bit-1) := '1';
			bv(bit-1) := '0';
			tmp(bit) := '-';
		end loop;

		-- G/P outputs
		writestr("*** testing generate/propagate outputs ***");
		av := (others => 'X');
		bv := (others => 'X');
		for i in WIDTH-1 downto 0 loop
			av(i) := '1';
			bv(i) := '1';
			A <= av; B <= bv; wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("1"));
			check_logic("P", P, std_ulogic_vector'("0"));
			av(i) := '0';
			bv(i) := '0';
			A <= av; B <= bv; wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("0"));
			check_logic("P", P, std_ulogic_vector'("0"));
			av(i) := '1';
		end loop;
		A <= av; wait for 1 ns;
		tmp := (others => '1');
		check_logic("G", G, std_ulogic_vector'("0"));
		check_logic("P", P, std_ulogic_vector'("1"));
		check_logic("Y", Y, tmp);
		check_logic("Z", Z, not tmp);
		for i in WIDTH-1 downto 0 loop
			av(i) := not av(i);
			bv(i) := not bv(i);
			A <= av; B <= bv; wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("0"));
			check_logic("P", P, std_ulogic_vector'("1"));
			check_logic("Y", Y, tmp);
			check_logic("Z", Z, not tmp);
		end loop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
