-- idiv64_test2.vhdl - F-CPU Integer Divider Testbench #2
-- Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: idiv64_test2.vhdl,v 1.6 2003/04/06 20:52:04 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.Bit_Manipulation.all;

entity IDiv64_test2 is
	generic (WIDTH : natural := 64);
end IDiv64_test2;

architecture Arch_1 of IDiv64_test2 is
	component IDiv64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			S : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(3 downto 0);
	signal Y, Z : std_ulogic_vector(WIDTH-1 downto 0);
	signal Clk, Rst, En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string; x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		--write(lout, string'("Z := ")); write(lout, Z); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	function match (a, b : std_ulogic_vector) return boolean is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			return false;
		end loop;
		return true;
	end match;

	procedure check_logic (lbl : string; a, b : std_ulogic_vector) is
	begin
		if not match(a, b) then
			do_report(lbl, a, b);
			wait;
		end if;
	end check_logic;

	procedure pulse (signal x : out std_ulogic; n : in natural := 1) is
	begin
		for i in 1 to n loop
			wait for 1 ns; x <= '1';
			wait for 1 ns; x <= '0';
		end loop;
	end pulse;
begin
	mut: IDiv64
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, S => M(0), U => M(3 downto 1),
			Clk => Clk, Rst => Rst, En => En, Y => Y, Z => Z
		);

	test : process
		variable av, bv, tmp, last : std_ulogic_vector(WIDTH-1 downto 0);
		variable simd, chunks, left, right : natural;
		variable ax, limit : integer;
		variable lout : line;
	begin
		Clk <= '0'; Rst <= '0'; En <= '0'; pulse(Rst);
		last := (others => '-');
		for gran in 0 to 3 loop
			simd := 2 ** (gran + 3);
			-- limit used to be constant 256,
			-- but the simulation took too long...
			limit := 256 / 2 ** gran;
			chunks := WIDTH / simd;
			M <= "XXX0";
			for i in 1 to 3 loop
				if i - 1 >= gran then
					M(i) <= '0';
				else
					M(i) <= '1';
				end if;
			end loop;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit unsigned mode ***"));
			writeline(output, lout);
			for bb in 1 to limit-1 loop
				for aa in 0 to limit/chunks-1 loop
					for cc in 0 to chunks-1 loop
						right := simd * cc;
						left := right + simd - 1;
						ax := chunks * aa + cc;
						av(left downto right) :=
							std_ulogic_vector(to_unsigned(ax, simd));
						bv(left downto right) :=
							std_ulogic_vector(to_unsigned(bb, simd));
						tmp(left downto right) :=
							std_ulogic_vector(to_unsigned(ax / bb, simd));
					end loop;
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, 3);
					check_logic("Y", Y, last);
					pulse(Clk, simd - 4);
					last := tmp;
				end loop;
			end loop;
			M(0) <= '1';
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit signed mode ***"));
			writeline(output, lout);
			for bb in -limit/2 to limit/2-1 loop
				next when bb = 0;
				for aa in 0 to limit/chunks-1 loop
					for cc in 0 to chunks-1 loop
						right := simd * cc;
						left := right + simd - 1;
						ax := chunks * aa + cc - limit/2;
						av(left downto right) :=
							std_ulogic_vector(to_signed(ax, simd));
						bv(left downto right) :=
							std_ulogic_vector(to_signed(bb, simd));
						tmp(left downto right) :=
							std_ulogic_vector(to_signed(ax / bb, simd));
					end loop;
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, 3);
					check_logic("Y", Y, last);
					pulse(Clk, simd - 4);
					last := tmp;
				end loop;
			end loop;
		end loop;
		pulse(Clk, 4);
		check_logic("Y", Y, last);

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
