-- iatest7.vhdl - Testbench for F-CPU IAdd Unit
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: iatest7.vhdl,v 1.3 2003/03/14 22:38:40 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;

entity IAdd_test is
	generic (
		WIDTH : natural := 64;	-- or any integer multiple of 64
		PIPELINED : integer := 1
	);
end IAdd_test;

architecture Arch_1 of IAdd_test is
	component IAdd
		generic (
			WIDTH : natural := 64;
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Sub : in std_ulogic;
			Sat : in std_ulogic;
			Inc : in std_ulogic;
			Avg : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y8l : out std_ulogic_vector(WIDTH-1 downto 0);
			Y8h : out std_ulogic_vector(WIDTH-1 downto 0);
			Yl : out std_ulogic_vector(WIDTH-1 downto 0);
			Yh : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(6 downto 0) := (others => '0');
	signal A, B, H, L : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Yl, Y8l, Yh, Y8h : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Clk, Rst, En : std_ulogic;

	constant std_0 : std_ulogic := '0';
	constant std_1 : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : IAdd
		generic map (WIDTH => WIDTH, PIPELINED => PIPELINED)
		port map (
			A => A, B => B, Sub => M(0), Sat => M(1), Inc => M(2),
			Avg => M(3), U => M(6 downto 4),
			Clk => Clk, Rst => Rst, En => En,
			Y8l => Y8l, Y8h => Y8h, Yl => Yl, Yh => Yh
		);

	-- choose correct output
	L <= Yl when M(4) = '1' else Y8l;
	H <= Yh when M(4) = '1' else Y8h;

	-- driver process
	process
		variable lastl, lasth : std_ulogic_vector(WIDTH-1 downto 0);

		procedure check_last is
		begin
			if PIPELINED /= 0 then
				check_logic("Yl", Yl, lastl);
				check_logic("Yh", Yh, lasth);
				Clk <= '1'; wait for 1 ns;
				Clk <= '0'; wait for 1 ns;
				lastl := Yl;
				lasth := Yh;
			end if;
		end check_last;

		procedure test_addsub is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, bits, carry, simd, res : natural;
			variable lout : line;
		begin
			lastl := (others => '-');
			lasth := (others => '-');
			M <= "UUU0000";	-- add mode
			writestr("*** testing add/sub instructions ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n + 4) <= '1';
					else
						M(n + 4) <= '0';
					end if;
				end loop;
				-- main test loop
				for slice in 0 to WIDTH/simd-1 loop
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					right := slice * simd;
					left := right + simd - 1;
					for bit in right to left loop
						for i in std_0 to std_1 loop
							av(bit) := i;
							for j in std_0 to std_1 loop
								bv(bit) := j;
								for k in std_0 to std_1 loop
									M(2) <= k;
									tmp(bit) := (i xor j) xor k;
									A <= av; B <= bv; wait for 1 ns;
									if gran = 0 then
										check_logic("Y8l", Y8l, tmp);
									end if;
									check_last;
									check_logic("Yl", Yl, tmp);
									M(0) <= '1';
									A <= not av; B <= bv; wait for 1 ns;
									if gran = 0 then
										check_logic("Y8l", not Y8l, tmp);
									end if;
									check_last;
									check_logic("Yl", not Yl, tmp);
									M(0) <= '0';
								end loop;
							end loop;
						end loop;
						av(bit) := '1';
						bv(bit) := '0';
						tmp(bit) := '-';
					end loop;
					-- carry logic
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					for bit in right+1 to left loop
						av(bit) := '0';
						bv(bit) := '0';
						for i in std_0 to std_1 loop
							av(bit-1) := i;
							for j in std_0 to std_1 loop
								bv(bit-1) := j;
								for k in std_0 to std_1 loop
									M(2) <= k;
									tmp(bit) := (i and j) or (j and k) or (k and i);
									A <= av; B <= bv; wait for 1 ns;
									if gran = 0 then
										check_logic("Y8l", Y8l, tmp);
									end if;
									check_last;
									check_logic("Yl", Yl, tmp);
									M(0) <= '1';
									A <= not av; B <= bv; wait for 1 ns;
									if gran = 0 then
										check_logic("Y8l", not Y8l, tmp);
									end if;
									check_last;
									check_logic("Yl", not Yl, tmp);
									M(0) <= '0';
								end loop;
							end loop;
						end loop;
						av(bit-1) := '1';
						bv(bit-1) := '0';
						tmp(bit) := '-';
					end loop;
					-- carry-out
					tmp(left downto right) := (left downto right => '0');
					for i in std_0 to std_1 loop
						av(left) := i;
						for j in std_0 to std_1 loop
							bv(left) := j;
							for k in std_0 to std_1 loop
								M(2) <= k;
								tmp(right) := (i and j) or (j and k) or (k and i);
								A <= av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8h", Y8h, tmp);
								end if;
								check_last;
								check_logic("Yh", Yh, tmp);
								M(0) <= '1';
								A <= not av; B <= bv; wait for 1 ns;
								if gran = 0 then
									check_logic("Y8h", Y8h, tmp);
								end if;
								check_last;
								check_logic("Yh", Yh, tmp);
								M(0) <= '0';
							end loop;
						end loop;
					end loop;
				end loop;
			end loop;
		end test_addsub;

		procedure test_sat is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, bits, carry, simd, res : natural;
			variable lout : line;
		begin
			M <= "UUU0010";	-- add+saturate mode
			writestr("*** testing adds/subf instructions ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n + 4) <= '1';
					else
						M(n + 4) <= '0';
					end if;
				end loop;
				-- main test loop
				for slice in 0 to WIDTH/simd-1 loop
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					right := slice * simd;
					left := right + simd - 1;
					tmp(left downto right) := (left downto right => '1');
					for bit in left downto right loop
						av(bit) := '1';
						bv(bit) := '1';
						A <= av; B <= bv; wait for 1 ns;
						check_last;
						check_logic("Yl", Yl, tmp);
						M(0) <= '1';
						A <= not av; B <= bv; wait for 1 ns;
						check_last;
						check_logic("Yl", not Yl, tmp);
						M(0) <= '0';
						bv(bit) := '0';
					end loop;
					M(2) <= '1';
					A <= av; B <= bv; wait for 1 ns;
					check_last;
					check_logic("Yl", Yl, tmp);
					M(0) <= '1';
					A <= not av; B <= bv; wait for 1 ns;
					check_last;
					check_logic("Yl", not Yl, tmp);
					M(0) <= '0';
					M(2) <= '0';
				end loop;
			end loop;
		end test_sat;

		procedure test_avg is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, bits, carry, simd, res : natural;
			variable lout : line;
		begin
			M <= "UUU1000";	-- avg mode
			writestr("*** testing avg instructions ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n + 4) <= '1';
					else
						M(n + 4) <= '0';
					end if;
				end loop;
				-- main test loop
				for slice in 0 to WIDTH/simd-1 loop
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					right := slice * simd;
					left := right + simd - 1;
					for bit in right+1 to left loop
						av(bit-1) := '1';
						bv(bit-1) := '0';
						for i in std_0 to std_1 loop
							av(bit) := i;
							for j in std_0 to std_1 loop
								bv(bit) := j;
								for k in std_0 to std_1 loop
									M(2) <= k;
									tmp(bit-1) := (i xor j) xor k;
									A <= av; B <= bv; wait for 1 ns;
									check_last;
									check_logic("Yh", Yh, tmp);
									M(0) <= '1';
									A <= not av; B <= bv; wait for 1 ns;
									check_last;
									check_logic("Yh", Yh, tmp);
									M(0) <= '0';
								end loop;
							end loop;
						end loop;
						tmp(bit-1) := '-';
					end loop;
					-- carry logic
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					for bit in right+1 to left loop
						av(bit) := '0';
						bv(bit) := '0';
						for i in std_0 to std_1 loop
							av(bit-1) := i;
							for j in std_0 to std_1 loop
								bv(bit-1) := j;
								for k in std_0 to std_1 loop
									M(2) <= k;
									tmp(bit-1) := (i and j) or (j and k) or (k and i);
									A <= av; B <= bv; wait for 1 ns;
									check_last;
									check_logic("Yh", Yh, tmp);
									M(0) <= '1';
									A <= not av; B <= bv; wait for 1 ns;
									check_last;
									check_logic("Yh", Yh, tmp);
									M(0) <= '0';
								end loop;
							end loop;
						end loop;
						av(bit-1) := '1';
						bv(bit-1) := '0';
						tmp(bit-1) := '-';
					end loop;
					-- carry-out
					for i in std_0 to std_1 loop
						av(left) := i;
						for j in std_0 to std_1 loop
							bv(left) := j;
							for k in std_0 to std_1 loop
								M(2) <= k;
								tmp(left) := (i and j) or (j and k) or (k and i);
								A <= av; B <= bv; wait for 1 ns;
								check_last;
								check_logic("Yh", Yh, tmp);
								M(0) <= '1';
								A <= not av; B <= bv; wait for 1 ns;
								check_last;
								check_logic("Yh", not Yh, tmp);
								M(0) <= '0';
							end loop;
						end loop;
						tmp(left) := '-';
					end loop;
				end loop;
			end loop;
		end test_avg;
	begin
		writestr("*** starting simulation ***");
		if PIPELINED /= 0 then
			writestr("*** pipelining is ON ***");
		else
			writestr("*** pipelining is OFF ***");
		end if;
		Clk <= '0'; Rst <= '1'; wait for 1 ns;
		Rst <= '0'; En <= '1'; wait for 1 ns;
		test_addsub;
		test_sat;
		test_avg;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;
