-- generic_adder_test.vhdl -- Generic Carry-Increment Adder
-- Copyright (C) 2000 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- $Id: generic_adder_test.vhdl,v 1.6 2002/06/22 15:02:39 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;
use work.Generic_Adder.all;

entity Generic_Adder_test is
	generic (WIDTH : natural := 39);
end Generic_Adder_test;

architecture Arch_1 of Generic_Adder_test is
	signal A, B : std_ulogic_vector(WIDTH-1 downto 0) := (others => '1');
	signal Y, C : std_ulogic_vector(WIDTH-1 downto 0);
	signal G, P : std_ulogic_vector(0 downto 0);

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, string'("C := ")); write(lout, C); writeline(output, lout);
		write(lout, string'("G := ")); write(lout, G); writeline(output, lout);
		write(lout, string'("P := ")); write(lout, P); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : S_CIAdd(A, B, Y, C, G(0), P(0));

	-- driver process
	run : process
		variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable left, right, bits, res : natural;
		variable lout : line;
	begin
		write(lout, string'("*** testing Generic_Adder entity ("));
		write(lout, WIDTH);
		write(lout, string'(" bits wide) ***"));
		writeline(output, lout);

		av := (others => 'X');
		bv := (others => 'X');
		for slice in 0 to (WIDTH-1)/4 loop
			write(lout, string'("*** testing slice "));
			write(lout, slice);
			write(lout, string'(" ***"));
			writeline(output, lout);

			-- calculate slice range and width
			right := 4 * slice; left := right + 4;
			if left >= WIDTH then
				-- leftmost 4-bit slice, strip MSBits
				left := WIDTH - 1;
			end if;
			bits := left - right + 1;
			-- carry-in on/off loop
			for k in 0 to 1 loop
				-- turn carry-in on/off
				if right /= 0 then
					if k = 0 then
						av(right-1) := '0';
						bv(right-1) := '0';
					else
						av(right-1) := '1';
						bv(right-1) := '1';
					end if;
				end if;
				-- input values loops
				for i in 0 to 2**bits-1 loop
					av(left downto right) := std_ulogic_vector(to_unsigned(i, bits));
					A <= av;
					for j in 0 to 2**bits-1 loop
						bv(left downto right) := std_ulogic_vector(to_unsigned(j, bits));
						B <= bv;
						wait for 1 ns;
						-- test normal output
						res := i + j;
						if right /= 0 then
							res := res + k;
						end if;
						check_numeric("Y", Y(left downto right), res);
					end loop;
				end loop;
			end loop;
			-- clear slice
			av(left downto right) := (others => 'X');
			bv(left downto right) := (others => 'X');
			if right /= 0 then
				av(right-1) := 'X';
				bv(right-1) := 'X';
			end if;
		end loop;

		writestr("*** testing G/P outputs ***");
		av := (others => 'X');
		bv := (others => 'X');
		for i in WIDTH-1 downto 0 loop
			av(i) := '1';
			bv(i) := '1';
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("1"));
			check_logic("P", P, std_ulogic_vector'("0"));
			av(i) := '0';
			bv(i) := '0';
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("0"));
			check_logic("P", P, std_ulogic_vector'("0"));
			av(i) := '1';
		end loop;
		A <= av;
		wait for 1 ns;
		tmp := (others => '1');
		check_logic("G", G, std_ulogic_vector'("0"));
		check_logic("P", P, std_ulogic_vector'("1"));
		check_logic("C", C, tmp);
		for i in WIDTH-1 downto 0 loop
			av(i) := not av(i);
			bv(i) := not bv(i);
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("G", G, std_ulogic_vector'("0"));
			check_logic("P", P, std_ulogic_vector'("1"));
			check_logic("C", C, tmp);
		end loop;

		writestr("*** testing C output ***");
		av := (others => 'X');
		bv := (others => 'X');
		tmp := (others => '-');
		tmp(0) := '1';
		check_logic("C", C, tmp);
		for i in 1 to WIDTH-1 loop
			tmp(i) := '0';
			av(i-1) := '1';
			bv(i-1) := '1';
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("C", C, tmp);
			av(i-1) := '0';
			bv(i-1) := '0';
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("C", C, tmp);
			tmp(i) := '1';
			av(i-1) := '1';
			A <= av;
			wait for 1 ns;
			check_logic("C", C, tmp);
			av(i-1) := '0';
			bv(i-1) := '1';
			A <= av;
			B <= bv;
			wait for 1 ns;
			check_logic("C", C, tmp);
		end loop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
