-- shuffle64_test1.vhdl -- Testbench for 64-Bit F-CPU Bit Shuffling Unit
-- Copyright (C) 2001, 2002 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- $Id: shuffle64_test1.vhdl,v 1.2 2002/05/13 18:56:33 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;
use work.Bit_Manipulation.all;

entity Shuffle64_test1 is
end Shuffle64_test1;

architecture Arch_1 of Shuffle64_test1 is
	component Shuffle64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			CommonShiftCount : in std_ulogic;
			ShiftL : in std_ulogic;
			ShiftR : in std_ulogic;
			ShiftRA : in std_ulogic;
			RotL : in std_ulogic;
			RotR : in std_ulogic;
			Bitrev : in std_ulogic;
			Byterev : in std_ulogic;
			Sdup : in std_ulogic;
			Mix : in std_ulogic;
			Expand : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Y2 : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant WIDTH : natural := 64;

	signal M : std_ulogic_vector(12 downto 0) := (others => '0');
	signal A, B, Y, Z : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal CommonShiftCount : std_ulogic := '0';
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("B", B);
		print_vector("M", M);
		print_vector("Y", Y);
		print_vector("Z", Z);
	end print_signals;

	procedure do_error (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_error;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_error(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_error(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Shuffle64
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, U => M(2 downto 0),
			CommonShiftCount => CommonShiftCount,
			ShiftL => M(3), ShiftR => M(4), ShiftRA => M(5),
			RotL => M(6), RotR => M(7), Bitrev => M(8),
			Byterev => M(9), Sdup => M(10),
			Mix => M(11), Expand => M(12),
			Clk => Clk, Rst => Rst, En => En,
			Y => Y, Y2 => Z
		);

	-- driver process
	process
		procedure print_mode (simd : in natural) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
		end print_mode;

		type shl_op is (
			FCOP_SHIFTL, FCOP_SHIFTR, FCOP_SHIFTRA,
			FCOP_ROTL, FCOP_ROTR, FCOP_BITREV, FCOP_BYTEREV,
			FCOP_SDUP, FCOP_MIX, FCOP_EXPAND
		);

		function bitop (op : in shl_op; alt_output : boolean;
						A : in std_ulogic_vector;
						B : in natural) return std_ulogic_vector is
			constant cs : natural := A'length;
			alias aa : std_ulogic_vector(cs-1 downto 0) is A;
			variable yy : std_ulogic_vector(cs-1 downto 0);
			variable zz : std_ulogic_vector(cs-1 downto 0);
			variable j : natural;
		begin
			assert B < cs;
			case op is
				when FCOP_SHIFTL =>
					yy := lshift(aa, B);
					zz := rshift(aa, cs - B);
				when FCOP_SHIFTR =>
					yy := rshift(aa, B);
					zz := lshift(aa, cs - B);
				when FCOP_SHIFTRA =>
					yy := rshifta(aa, B);
					zz := lshift(aa, cs - B);
				when FCOP_ROTL =>
					yy := lrotate(aa, B);
					zz := (others => '-');
				when FCOP_ROTR =>
					yy := rrotate(aa, B);
					zz := (others => '-');
				when FCOP_BITREV =>
					yy := rshift(bit_reverse(aa), cs - B - 1);
					zz := lshift(bit_reverse(aa), B + 1);
				when FCOP_BYTEREV =>
					assert not alt_output;
					for i in cs-1 downto 0 loop
						j := cs - i - 1;
						j := j - j rem 8 + i rem 8;
						yy(i) := aa(j);
					end loop;
					zz := (others => '-');
				when others =>
					assert false;
			end case;
			if alt_output then
				return zz;
			end if;
			return yy;
		end bitop;

		function shlmode (op : in shl_op;
						  gran : in natural) return std_ulogic_vector is
			variable m : std_ulogic_vector(12 downto 0);
		begin
			case op is
				when FCOP_SHIFTL =>
					m := "0000000001XXX";
				when FCOP_SHIFTR =>
					m := "0000000010XXX";
				when FCOP_SHIFTRA =>
					m := "0000000100XXX";
				when FCOP_ROTL =>
					m := "0000001000XXX";
				when FCOP_ROTR =>
					m := "0000010000XXX";
				when FCOP_BITREV =>
					m := "0000100000XXX";
				when FCOP_BYTEREV =>
					m := "0001000000XXX";
				when FCOP_SDUP =>
					m := "0010000000XXX";
				when FCOP_MIX =>
					m := "0100000000XXX";
				when FCOP_EXPAND =>
					m := "1000000000XXX";
				when others =>
					assert false;
			end case;
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					m(n) := '1';
				else
					m(n) := '0';
				end if;
			end loop;
			return m;
		end shlmode;

		procedure write (l : inout line; op : in shl_op) is
		begin
			case op is
				when FCOP_SHIFTL =>
					write(l, string'("shiftl"));
				when FCOP_SHIFTR =>
					write(l, string'("shiftr"));
				when FCOP_SHIFTRA =>
					write(l, string'("shiftra"));
				when FCOP_ROTL =>
					write(l, string'("rotl"));
				when FCOP_ROTR =>
					write(l, string'("rotr"));
				when FCOP_BITREV =>
					write(l, string'("bitrev"));
				when others =>
					assert false;
			end case;
		end write;

		procedure print_testing (op : in shl_op) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, op);
			write(lout, string'(" instruction ***"));
			writeline(output, lout);
		end print_testing;

		procedure test_bitop (op : in shl_op;
							  full_simd : in boolean) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;

			function ignore_x (A : in std_ulogic_vector) return std_ulogic_vector is
				constant w : natural := A'length;
				alias aa : std_ulogic_vector(w-1 downto 0) is A;
				variable yy : std_ulogic_vector(w-1 downto 0);
			begin
				yy := to_X01(aa);
				for i in w-1 downto 0 loop
					if yy(i) = 'X' then
						yy(i) := '-';
					end if;
				end loop;
				return yy;
			end ignore_x;

			procedure do_test (count : in natural) is
			begin
				tmp(left downto right) :=
					ignore_x(bitop(op, false, av(left downto right), count));
				check_logic("Y", Y, tmp);
				tmp(left downto right) :=
					ignore_x(bitop(op, true, av(left downto right), count));
				check_logic("Z", Z, tmp);
			end do_test;
		begin
			print_testing(op);
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(op, gran);
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					for shift_count in 0 to simd-1 loop
						if full_simd then
							bv(right+gran+2 downto right) :=
								std_ulogic_vector(to_unsigned(shift_count, gran+3));
						else
							bv(gran+2 downto 0) :=
								std_ulogic_vector(to_unsigned(shift_count, gran+3));
						end if;
						tmp := (others => '-');
						A <= av; B <= bv; wait for 1 ns;
						do_test(shift_count);
						for bit_pos in 0 to simd-1 loop
							for bit in std_ulogic'('1') downto std_ulogic'('0') loop
								av(right + bit_pos) := bit;
								A <= av; B <= bv; wait for 1 ns;
								do_test(shift_count);
							end loop;
							av(right + bit_pos) := 'X';
						end loop;
					end loop;
				end loop;
			end loop;
		end test_bitop;
	begin
		for full_simd in boolean'(false) to boolean'(true) loop
			if full_simd then
				writestr("+++ testing full-SIMD bitwise instructions +++");
				CommonShiftCount <= '0';
			else
				writestr("+++ testing semi-SIMD bitwise instructions +++");
				CommonShiftCount <= '1';
			end if;
			test_bitop(FCOP_SHIFTL, full_simd);
			test_bitop(FCOP_SHIFTR, full_simd);
			test_bitop(FCOP_SHIFTRA, full_simd);
			test_bitop(FCOP_ROTL, full_simd);
			test_bitop(FCOP_ROTR, full_simd);
			test_bitop(FCOP_BITREV, full_simd);
			writestr("");
		end loop;
		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
