--------------------------------------------------------------------------
-- f-cpu/vhdl/common/fanout_tree.vhdl - Generic fanout tree (2**n) for the F-CPU
-- Copyright (C) 2001 Yann GUIDON (whygee@f-cpu.org)
--
-- created sam dec  1 00:39:05 GMT 2001 by YG for the ROP2 unit, but will
--  certainly be used in a slew other cases.
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- This component implements a balanced fanout tree, for use when a signal
-- must be sent to more than 2**n inputs, where n > 2.
--
-- The first version is for "quick and dirty coding", or as a failsafe
-- alternative. It is also suited for behavioural simulations where
-- it is not necessary to know every net.
-- The second version is a binary tree for synthesis. It tries to trick the
-- "dumb" optimizers into thinking that the whole tree contains different
-- signals and values, so that the nets won't be tied dumbly together
-- (which would void the use of this component). Using inverters,
-- which are somewhat faster than buffers that are probably not inferred,
-- we can break the net into sub-levels. This means that one inverter
-- level must be added if log2_width is odd.
-- Depending on the software, this will be more or less efficient. Try each
-- implementation to be sure.
-- More alternative versions can be added if these ones don't fit your needs.
--
--------------------------------------------------------------------------

-- some standard librairies
LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
    
-- component interface :
Entity fanout_tree is
  generic (
    log2_width : natural
  );
  port(
    root : in Std_ulogic;
    leaf : out Std_ulogic_vector((2**log2_width)-1 downto 0)
  );
end fanout_tree;

-- failsafe alternative or for behavioural simulations :
Architecture simple of fanout_tree is
begin
  leaf <= (others => root);
end;

-- more sophisticated version :
Architecture tree of fanout_tree is
  signal t : Std_ulogic_vector(leaf'range);

  function binary_tree_index (
    index : integer)
  return integer is
    variable i, j : integer;
  begin  -- binary_tree_index

    -- 1 ) count the number of LSB that are set to 1
    -- (the loop could be avoided but requires a XOR which
    --  is not defined for integers :-( )
    i := index;
    j := 4;

    while (i mod 2) = 1 loop
      i := i / 2;
      j := j * 2;
    end loop;

    -- 2) the incredible magic formula !
    -- don't ask me where it comes, it was built from observation.
    i := (j - 1) / 2;
    return ((index / j)*j) + i;
  end binary_tree_index;
  
begin

  assert (log2_width > 2 ) and (log2_width < 8)
    report "wrong size range for the binary tree"
    severity FAILURE;

  assert (leaf'low) = 0
    report "array index should start with 0"
    severity FAILURE;

  -- compensation for the odd/even cases.
  odd : if (log2_width mod 2) = 1 generate
    t(t'high) <= root;
  end generate odd;
  even: if (log2_width mod 2) = 0 generate
    t(t'high) <= not root;
  end generate even;

  i_loop: for i in leaf'range generate
    leaf(i) <= not t((i/2) *2);     -- connect to the even numbered temporary nodes.
    saturate: if i<leaf'high generate
      t(i) <= not t(binary_tree_index(i));  -- map the binary tree to a linear vector.
    end generate saturate;
  end generate i_loop;
end;
