 /*

       Gamma project
 programming model simulator
       Whygee 1/1999

 This program should compile and run anywhere
 because it is independent from the int size and
 the endianness of the machine. The obvious
 drawback is that it runs slowly. This is the
 cost of portability.

 This program is a testbench for the Gamma architecture.
 Only the raw addressing mode is simulated here,
 the simulated programs are normal applications and not OS
 (the protection mechanism is not complete)
 and the behaviour may differ from the definitions
 in the manual (there is some lag between both).
 It implements pointers where the 8 MSB are the memory type.

 rev 2: Thu Feb  4 02:48:31 CET 1999 :
  fixed some blurbs, cleaned the code to make it work like it should

 rev 3: Thu Feb 18 18:23:02 CET 1999
  changed the debug file format (merged the binary and the listing output),
  separated the simulator's error messages from the simulated processor faults,
  added the CPFQ restriction,
  changed the pointer's format, added the MOVE instruction

 rev 4: Thu Mar  4 00:21:37 CET 1999
  found the silly SIGSEGV bug.
  problem: pointer format ??? 16:48, 8:56, 16:64 ??? 8:56 seems ok
   but it will change (again). in any case, pointer size == MAXSIZE !
  added the serialize opcode

 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* declaration */
void fault(char * message);

/* for the simulator */
typedef unsigned long int uli;
/* "uli" is used when much precision is needed, at least 32 bits.
   "int" is used when simple numbers (less than 1000 for example) are used.*/

/* Programming model type definitions : */

/* for the processor */
typedef unsigned char u8;
typedef unsigned char u16[2];
typedef unsigned char u32[4];
typedef unsigned char u64[8];

/* implementation dependent: */
#define MAXSIZE 8
#define UMAX u64

UMAX const_zero; /* used to add constant numbers through the carry */

/* Programming model blocks : */

/* the four bytes that hold the instruction : */
u8 ic0,ic1,ic2,ic3;

/* definition of the registers: */
UMAX  registers[64];
/* The associated flags */
u64 flag_zero, flag_carry, flag_sign;


/* Some Special Purpose Registers */
#define MAXSPR 40
uli int_val[4],   /* SPR[0..3] */
 max_int_value,   /* SPR4 */
 current_pfq;     /* SPR5 */
UMAX cycle_count; /* SPR6 */

char *SPRname[]={
 "SIZE0",
 "SIZE1",
 "SIZE2",
 "SIZE3",
 "MAXSIZE",
 "CPFQ",
 "CYCLE",
 "MAXSPR",
 "PFQ0_LIMIT", 
 "PFQ1_LIMIT", 
 "PFQ2_LIMIT", 
 "PFQ3_LIMIT", 
 "PFQ4_LIMIT", 
 "PFQ5_LIMIT", 
 "PFQ6_LIMIT", 
 "PFQ7_LIMIT", 
 "PFQ8_LIMIT", 
 "PFQ9_LIMIT", 
 "PFQ10_LIMIT", 
 "PFQ11_LIMIT", 
 "PFQ12_LIMIT", 
 "PFQ13_LIMIT", 
 "PFQ14_LIMIT", 
 "PFQ15_LIMIT", 
 "PFQ16_LIMIT", 
 "PFQ17_LIMIT", 
 "PFQ18_LIMIT", 
 "PFQ19_LIMIT", 
 "PFQ20_LIMIT", 
 "PFQ21_LIMIT", 
 "PFQ22_LIMIT", 
 "PFQ23_LIMIT", 
 "PFQ24_LIMIT", 
 "PFQ25_LIMIT", 
 "PFQ26_LIMIT", 
 "PFQ27_LIMIT", 
 "PFQ28_LIMIT", 
 "PFQ29_LIMIT", 
 "PFQ30_LIMIT", 
 "PFQ31_LIMIT"
};

struct PFQ_type {
UMAX
/* the data and status registers are "volatile" */
 index,
 base,
 limit,
 read_increment,
 write_increment,
 granularity,
 properties;
int memory_type;
} PFQ[32];

/* Definition of "properties" */

#define EXECUTABLE 1
#define WRITEONLY  2
#define READONLY   4
#define READWRITE  6

/* for a beginning: */
#define MAXMEMTYPE 4

struct memory_descriptor_type {
 UMAX
  /*speed,   in processor cycles per block */
  /* latency, in processor cycles */
  properties, /* RWX etc */
   /*  granularity, in bytes */
  index,       /* in bytes */
  size;        /* in bytes */
 u8 * pointer; /* points to the actual memory in the simulator */
} memory_descriptor [MAXMEMTYPE];

/*
    SOME LOW LEVEL ROUTINES
 */

void error(char * message) {
 printf("\n\nSimulator error: %s\n",message);
 exit(-1);
}

/* converts a word to an int */
uli read_value (uli size, u8 *c) {
 uli t=0;

 do {
  size--;
  t<<=8;
  t|=c[size];
 } while (size);

 return t;
}

/* converts an int to a word */
void write_value (uli size, u8 *c, uli value) {
 do {
  *(c++)= (u8)(value);
  value>>=8;
  size--;
 } while (size);
}

/* compares two words */
int compare_superior_values(uli size, u8 * s1, u8 *s2) {
 do {
  size--;
  if (s1[size]>s2[size]) return 1;
  if (s1[size]<s2[size]) return 0;
 } while (size);
 return 0;
}

int compare_equal_values(uli size, u8 * s1, u8 *s2) {
 do {
  size--;
  if (s1[size]!=s2[size]) return 0;
 } while (size);
 return 1;
}

/* adds two words with carry */
int add_values (uli size, u8 *res, u8 *op1, u8 *op2, uli carry) {
 uli i=0, j=carry;
 do {
  j+=op1[i]+op2[i];
  res[i]=(char)j;
  j>>=8;
  i++;
 } while (i<size);
 return j;
}

u8 char_reg(u8 reg) {
 switch (reg>>5) {
  default:
  case 0:
  case 1: return 'R';
  case 2: return 'D';
  case 3: return 'I';
 }
}

uli num_reg(u8 reg) {
 if (reg & 64)
  reg=reg & 31;   /* PFQ */
 return reg;
}

/* displays a word */
void display_value(uli size, u8 *c) {
 do
  printf("%02X",c[--size]);
 while (size);
}

/* returns a pointer to the memory pointed to by a PFQ */
u8 * get_pointer(u8 pfq, uli size) {
 UMAX temp;
 u8 * p;

 p=memory_descriptor[PFQ[pfq].memory_type].pointer
  +read_value(MAXSIZE,PFQ[pfq].index);
/* check the bounds */
 add_values(MAXSIZE,temp,PFQ[pfq].index,const_zero,size);
 if (compare_superior_values(MAXSIZE,temp,PFQ[pfq].limit))
   return NULL;
 return p;
}

void display_PFQ(u8 r) {
 u8 * p;
 printf("PFQ%02d:[",r);
 display_value(MAXSIZE,PFQ[r].index);
 printf("]=");
 if ((p=get_pointer(r,MAXSIZE))==NULL)
  printf("{out of bounds}");
 else
  display_value(MAXSIZE,p);
 printf(" (%d) ",PFQ[r].memory_type);
}

/* displays a register's content */
void display_register(u8 r) {
 u8  number,mask,temp[]="(0c+) ";

 if (r<64) {
  printf("R%02d:",r);
  display_value(MAXSIZE,registers[r]);

/* zero, carry, neg */
  number=r>>3;
  mask=1<<(r & 7);
  if (flag_zero[number] & mask)
   temp[1]='1';
  if (flag_carry[number] & mask)
   temp[2]='C';
  if (flag_sign[number] & mask)
   temp[3]='-';
  printf(temp);
 }
 else display_PFQ(r&31);
}

/* write to a register and update the flags */
void write_register (uli size, u8 reg, u8 *s, u8 update) {
 u8 t, /* to detect the sign */
  z=0, /* for the ZERO flag */
  *d,  /* destination */
  reg_,regn,mask; /* some temp data */

 if (reg<64) { /* we write to a GPR */

   /* data copy */
  d=registers[reg];
  do {
   t=*(s++); /* as a temporary char, t holds the last value at the end, */
   z|=t;     /* as to test the sign too. z is the OR of every byte. */
   *(d++)=t;
   size--;
  } while (size);

   /* flags */
  regn=reg>>3;
  mask=1<<(reg & 7);

   /* sign flag */
  t=((signed char)t >>7) & mask; /* get the sign */
  flag_sign[regn]=t | ((~mask)& flag_sign[regn]);

   /* zero flag */
  flag_zero[regn] &= ~mask;
  if (z)
   flag_zero[regn] |= mask;
 }

   /* PFQs */

 else {
  reg_=reg & 31;
  if (reg_ == current_pfq)
   fault ("attempt to write to the CPFQ");

  if (reg & 32) {

        /* index register */
    /*   if (compare_superior_values(MAXSIZE,s,PFQ[reg_].limit))
         fault ("attempt to point an index past the buffer limit"); */

   /* copy the index */
   if (size<MAXSIZE)
    memcpy(PFQ[reg_].index,s,size);
   else {
    if (s[MAXSIZE-1]>=MAXMEMTYPE)
     fault ("invalid memory type");
    memcpy(PFQ[reg_].index,s,MAXSIZE-1);
    PFQ[reg_].index[MAXSIZE-1]=0;
    PFQ[reg_].memory_type=s[MAXSIZE-1];
   }
  }

  else {

          /* data window */
   d=get_pointer(reg_,size);
   if (d==NULL)
    fault ("attempt to write past the buffer limit");
   /* actual write */
   memcpy(d,s,size);
   if (update)
/*    add_values(MAXSIZE,PFQ[reg_].index,PFQ[reg_].index,PFQ[reg_].write_increment,s); */
    add_values(MAXSIZE-2,PFQ[reg_].index,PFQ[reg_].index,const_zero,size);
  }
 }
}

/* read from a register */
void read_register (uli size, u8 reg, u8 *d, u8 update) {
 u8 *s,reg_; /* source */

 if (reg<64) /* we write to a GPR */

   /* copy the data data */
  memcpy (d,registers[reg],size);

   /* PFQs */
 else {
  reg_=reg & 31;
  if (reg & 32) {

        /* copy the index register */
   memcpy(d,PFQ[reg_].index,size);
   if (size == MAXSIZE) {
    d[MAXSIZE-1]=PFQ[reg_].memory_type;
   }
  }
  else {
          /* data window */
   if (reg_ == current_pfq)
    fault ("attempt to read the CPFQ");
   s=get_pointer(reg_,size);
   if (s==NULL)
    fault ("attempt to read past the buffer limit");
   /* actual write */
   memcpy(d,s,size);
   if (update)
    add_values(MAXSIZE-1,PFQ[reg_].index,PFQ[reg_].index,const_zero,size);
  }
 }
}

void write_processor_state (void) {
 u8 i,r;

 /* GPR */
 r=0;
 for (i=0; i<32; i++) {
  display_register(r++);
  display_register(r++);
  putchar('\n');
 }

 /* Prefetch Queues */
 r=0;
 for (i=0; i<32; i++) {
  display_PFQ(r++);
  putchar('\n');
 }
}

void fault(char * message) {
 printf("\n\nProcessor fault: %s\nProcessor state follows:\n",message);
 write_processor_state ();
 exit(-1);
}

u8 h2c(u8 c) {
 if (c>'9') return c-'A'+10;
 return c-'0';
}

u8 f_ok(FILE *file) {
 int i=fgetc(file);
 if (i==EOF) return 0;
 ungetc(i,file);
 return 1;
}

void initialize_processor (char *filename) {
FILE *file;
uli i, index, size;
u8 *p;
u8 buf[20];
u8 not_loaded=1;

/* reset the registers */
 memset(registers, 0, sizeof(registers));
 memset(flag_zero, 0, 8);
 memset(flag_carry, 0, 8);
 memset(flag_sign, 0, 8);

/* The Special Purpose Registers */
 current_pfq=0;
 memset(cycle_count, 0, MAXSIZE);
 max_int_value=MAXSIZE;
 int_val[0]=1; /* in bytes */
 int_val[1]=2;
 int_val[2]=4;
 int_val[3]=8;

/* Memory properties descriptors */
 memset(memory_descriptor, 0, sizeof(memory_descriptor));

/* read the object file: */
 file=fopen(filename,"rb");
 if (file==NULL) error("can't open source file");
 printf("\nReading file : %s\n\n",filename);

 while (f_ok(file)) {
  /* search the space number  */
  do {
   p=fgets(buf,sizeof(buf),file);
   if (p==NULL)
    error("can't read source file");
   if (strncmp("EOF",buf,3)==0)
    goto exit_loop; /* end of file */
  } while (strncmp("$Space ",buf,7));
  i=atoi(&buf[7]);
  if (i>=MAXMEMTYPE)
   error ("memory type number too big");
  if (memory_descriptor[i].pointer!=NULL)
   error("space already defined");

  /* search the properties */
  do {
   memset(buf, 0, sizeof(buf));
   p=fgets(buf,sizeof(buf),file);
  } while ((p!=NULL)&&(strncmp("$Properties ",buf,12)!=0));
  /* search R */
  if ((buf[14]=='R')|(buf[12]=='R')|(buf[13]=='R'))
   memory_descriptor[i].properties[0]=READONLY;
  if ((buf[14]=='W')|(buf[12]=='W')|(buf[13]=='W'))
   memory_descriptor[i].properties[0]|=WRITEONLY;
  if ((buf[14]=='X')|(buf[12]=='X')|(buf[13]=='X'))
   memory_descriptor[i].properties[0]|=EXECUTABLE;

  /* search the index */
  do
   p=fgets(buf,sizeof(buf),file);
  while ((p!=NULL)&&(strncmp("$Index ",buf,7)!=0));
  index=atoi(&buf[7]);

  /* search the size */
  do
   p=fgets(buf,sizeof(buf),file);
  while ((p!=NULL)&&(strncmp("$Size ",buf,6)!=0));
  size=atoi(&buf[6]);

  if ((p=malloc(index+size))==NULL) error ("not enough memory for loading a space");
  write_value (MAXSIZE,memory_descriptor[i].size,index+size);
  write_value (MAXSIZE,memory_descriptor[i].index,index);
  memory_descriptor[i].pointer=p;
  memset(p,0,index+size); /* clears the part before the actual beginning */

  while (size>0) {
   fgets(buf,sizeof(buf),file);
   p[0]=(h2c(buf[0])<<4)|h2c(buf[1]);
   p[1]=(h2c(buf[2])<<4)|h2c(buf[3]);
   p[2]=(h2c(buf[4])<<4)|h2c(buf[5]);
   p[3]=(h2c(buf[6])<<4)|h2c(buf[7]);
   size-=4;
   p+=4;
  }

  not_loaded=0;
 }
exit_loop:
 fclose(file);

 if (not_loaded)
  error ("nothing has been loaded.");
 if (compare_equal_values(MAXSIZE, const_zero,memory_descriptor[0].size))
  error ("SPACE#0 (start CPFQ) has not been defined.");

/* PreFetch Queues */
 memset(PFQ, 0, sizeof(PFQ)); /* initial "instruction pointer" */
 memcpy(PFQ[0].limit,memory_descriptor[0].size,MAXSIZE);
 memcpy(PFQ[0].properties,memory_descriptor[0].properties,MAXSIZE);
 memcpy(PFQ[0].index,memory_descriptor[0].index,MAXSIZE);
}

void fetch_instruction(void) {
 UMAX temp;
 uli i;
 u8 *p;

/* increment the cycle counter : */
 add_values(MAXSIZE,cycle_count,cycle_count,const_zero,1);
 printf("Cycle #");
 display_value(MAXSIZE,cycle_count);

/* round the instruction's index */
 PFQ[current_pfq].index[0] &= ~3;
/* check the bounds */
 write_value(MAXSIZE,temp,3);
 add_values(MAXSIZE,temp,PFQ[current_pfq].index,temp,0);
 if (compare_superior_values(MAXSIZE,temp,PFQ[current_pfq].limit))
  fault ("instruction fetch past the buffer limit");

 printf(", CPFQ : #%ld",current_pfq);

/* read the index */
 i=read_value(MAXSIZE,PFQ[current_pfq].index);

/* fetch the instruction : */
 p=memory_descriptor[PFQ[current_pfq].memory_type].pointer;
 ic0=p[i];
 ic1=p[i+1];
 ic2=p[i+2];
 ic3=p[i+3];

 printf("\n     Instruction: %02X%02X%02X%02X, Current index: ",
  ic0,ic1,ic2,ic3);
 display_value(MAXSIZE,PFQ[current_pfq].index);
 putchar('\n');

/* increment the instruction pointer : */
 add_values(MAXSIZE,PFQ[current_pfq].index,PFQ[current_pfq].index,const_zero,4);
}

u8 decode_instruction(void) {
 /*
    returns 0 when OK
    returns 1 when end of stream met
    returns 2 when wrong opcode
 */

/*
The 5 instruction formats: (now supported, out of 7)
1) op3r size,reg,reg,reg: 7,2,7,7,7 (add,sub, etc)
2) opri size,reg,imm14: 6,2,7,14 (movimm,addimm, readSPR,writeSPR)
3) op2r size,reg,reg,op_extension: 2,7,7,7 (neg,fneg,not,move,endian,PAL1,PAL2,readSPR,writeSPR...)
4) jpfq PFQ,skip/jump,size,reg,cond,priority: 5,1,2,7,4,2,(8) (jpfq, skip)
5) PreF PFQ,long/short,(size,reg,imm12)/(imm21),rwx: 5,1,(2,7,12)/21,2

6):PFQ management
7):loops
*/

 UMAX temp, temp2, res;
 uli imm14,imm16,size2,i;
 u8 t,t2,r,format3,op6,place4,dest7,sourceA7,sourceB7,pfq5,update1,update2;

/* some early and common field decoding: */
 format3=ic0>>5;
 pfq5=ic0 & 31;
 op6=((ic0<<1)|(ic1>>7)) & 63;
 size2=int_val[ic2>>6];
 dest7=ic1 & 127;
 sourceA7=((ic2 & 63)<<1)|(ic3>>7);
 sourceB7=ic3 & 127;
 if (sourceA7==dest7) update1=0;
                 else update1=1;
 if (sourceB7==dest7) update2=0;
                 else update2=1;

 switch(format3) {
  case 0:
/* format OPRI :
  OP6: opcode
  SIZE2: operation size
  DEST7: destination register
  IMM14: immediate value, sign extended
 */
   imm16=ic3|(ic2<<8);
   imm14=imm16 & 0x3FFF;

  /* write imm14 in temp, sign extended ! */
   if (imm14 >= (1<<13)) { /* tests the sign */
    temp[0]=ic3;
    temp[1]=ic2|0xC0;
    for (i=2; i<MAXSIZE; i++)
     temp[i]=0xFF;
   }
   else write_value(MAXSIZE,temp,imm14);


   switch (op6) {
    case 0:

/* Opcodes without operands */

     if (dest7)
      return 2;

     switch (imm16) {

/* NOP */
      case 0:
       printf("NOP\n");
       return 0;

/* SERIALIZE */
      case 1:
       printf("SERIALIZE\n");
       return 0;

/* ENDSTREAM */
      case 2:
       printf("ENDSTREAM\n");
       return 1;

      default: return 2;
     }


/* "movi": immediate move to register */
    case 1:
     printf("MOVI.%ld %c%ld, 0x%04lX   ",size2<<3,char_reg(dest7),num_reg(dest7),imm14);
     write_register(size2,dest7,temp,1);
     display_register(dest7);
     putchar('\n');
     return 0;

/* "addi": immediate add to register */
    case 2:
     printf("ADDI.%ld %c%ld, 0x%04lX   ",size2<<3,char_reg(dest7),num_reg(dest7),imm14);
     read_register (size2,dest7,temp2,0);
     i=add_values(size2<<3,temp,temp,temp2,0);
     write_register(size2,dest7,temp,1);
     if (i)
      flag_carry[dest7>>3]|=1<<(dest7 & 7);
     else
      flag_carry[dest7>>3]&=~(1<<(dest7 & 7));
     display_register(dest7);
     putchar('\n');
     return 0;

/* CLEAR */
    case 3:
     printf("CLEAR.%ld %c%ld     %c%ld=0\n",size2<<3,char_reg(dest7),num_reg(dest7),char_reg(dest7),num_reg(dest7));
     write_register(size2,dest7,const_zero,1);
     return 0;


/* WRSPR */
    case 4:
     printf("WRSPR.%ld %c%ld, %s  ",size2<<3,char_reg(dest7),num_reg(dest7),SPRname[imm14]);
     read_register (size2,dest7,temp2,1);
     
     /* select the SPR: */

     if (imm14<4) { /* SIZEx */
      i=read_value (size2,temp2);   
      if ((i & -i)!=i)
       fault ("size is not a power of two");
      if (i>MAXSIZE)
       fault ("size too big");
      int_val[imm14]=i;
     }
     else
      if (imm14==6) { /* CYCLE */
       memcpy(cycle_count,temp2,size2);
      }
      else
       if ((imm14>7)&&(imm14<40)) { /* PFQ */
        memcpy(PFQ[imm14-8].limit,temp2,size2);
       }
       else fault ("wrong SPR number");

     display_register(dest7);
     putchar('\n');
     return 0;


/* RDSPR */
    case 5:
     printf("RDSPR.%ld %c%ld, %s   ",size2<<3,char_reg(dest7),num_reg(dest7),SPRname[imm14]);

     /* select the SPR: */

     if (imm14<4) { /* SIZEx */
      write_value(size2,temp,int_val[imm14]);
     }
     else
      if (imm14==6) { /* CYCLE */
       memcpy(temp,cycle_count,size2);
      }
      else
       if ((imm14>7)&&(imm14<40)) { /* PFQ */
        memcpy(temp,PFQ[imm14-8].limit,size2);
       }
       else fault ("wrong SPR number");

     write_register(size2,dest7,temp,1);
     display_register(dest7);
     putchar('\n');
     return 0;


/* "movp": move a 16 bit immediate to a partial register */
    case 8:
    case 9:
    case 10:
    case 11:
    case 12:
    case 13:
    case 14:
    case 15:
     dest7&=63;
     place4=(ic1>>6)|((ic0 & 3)<<2);
     printf("MOVP R%d[%d], 0x%04lX     ",dest7,place4,imm16);
     place4<<=1;
     if (place4<MAXSIZE) {
      registers[dest7][place4++]=ic3;
      registers[dest7][place4]=ic2;
     }
     display_register(dest7);
     putchar('\n');
     return 0;

    default: return 2;
   }



  case 1:
/* format OPRR:
  OP6: opcode
  SIZE2: operation size
  DEST7: destination register
  sourceA7: source register 1
  sourceB7: 7-bit immediate value or opcode extension
 */

   switch (op6) {
    case 0:

     switch (sourceB7) {
      case 0:
/* MOVE */
       printf("MOVE.%ld %c%ld, %c%ld   ",size2<<3,char_reg(dest7),num_reg(dest7),
          char_reg(sourceA7),num_reg(sourceA7));
       read_register(size2,sourceA7,temp,update1);
       write_register(size2,dest7,temp,1);
       display_register(dest7);
       putchar('\n');
       return 0;

      default: /* fault ("unknown opcode"); */
     }

     default: return 2;
   }




  case 2:
/* format int OP3R */

   if (op6<16) {
/* Logic operations */

    printf("OPL%X.%ld %c%ld, %c%ld, %c%ld   ",op6,size2<<3,char_reg(dest7),num_reg(dest7),
          char_reg(sourceA7),num_reg(sourceA7),char_reg(sourceB7),num_reg(sourceB7));
    read_register(size2,sourceA7,temp,update1);
    read_register(size2,sourceB7,temp2,update2);
    for (i=0; i<size2; i++) {
      t=temp[i];
      t2=temp2[i];
      r= (  t &  t2 & (-( op6     & 1)))
       | (  t & ~t2 & (-((op6>>1) & 1)))
       | ( ~t &  t2 & (-((op6>>2) & 1)))
       | ( ~t & ~t2 & (-((op6>>3) & 1))) ;
      res[i]=r;
    }
    write_register(size2,dest7,res,1);
    display_register(dest7);
    putchar('\n');
    return 0;

   } else
/*
   switch (op6) {
    case 0:
    case 0:
    case 0:
    case 0:
     default:
   }
*/
   return 2;




  case 3:
/* format FP OP3R:
  OP6: opcode
  SIZE2: operation size
  DEST7: destination register
  sourceA7: source register 1
  sourceB7: source register 2
 */
   fault ("Floating point not supported");





  case 4:
/* format JPFCond:
  OP6: destination PFQ
  SIZE2: register size
  S_J1: : jump or skip
  DEST6: tested register
  COND4: condition code
 */
   /*  break; */




  case 5:
/* format PFRImm12 :
  PFQ5 : prefetched PFQ
  LS1  : short with register(0) or long(1)
  SIZE2: register size
  DEST7: register
  IMM12: immediate value
  RWX2 : PFQ intent (R,W,RW,X)
*/
/* imm12=(ic3 & ~3)|(ic2 & 63);
   imm21=(ic3 & ~3)|ic2|(ic1 & 127);
   rwx2=ic3 & 3;
 */
   /* break; */



  case 6:
  case 7:
  default: return 2;
 }

 return 0;
}

/* loop */
int main() {
 u8 c,d;
 initialize_processor ("out.lst");
 do {
  fetch_instruction();
  d=decode_instruction();
  if (d==2)
   fault ("unknown opcode");
  if (d==1) break;
  c=getchar();
  if (c=='r')
   write_processor_state ();
 } while (c!='q');
 printf ("\nSimulator finished.\n");
 return 0;
}
